/**
 * bmc_msg_tracker.c
 *
 * Tracks ipmi messages in order to match their responses
 * 
 * (c) 2004 Peppercon AG, 12/03/2004, tbr@peppecon.de
 */

#include <assert.h>
#include <pp/base.h>
#include <pp/rb_tree.h>
#include <pp/selector.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/bmc_router.h>
#include "bmc_seq_number.h"
#include "bmc_msg_tracker.h"

#ifdef DMALLOC
# warning undefining alloc,realloc,free for dmalloc macros to work

# undef free
# undef alloc
# undef realloc
#endif

typedef struct {
    pp_bmc_mtrack_cb_t sc_cb;    /* success callback */
    pp_bmc_mtrack_to_cb_t to_cb; /* timeout callback */
    int to_id;                   /* id of timeout timer */
    void* ctx;
} mtracked_t;

static pp_rb_tree_t* msg_dict = NULL;

static mtracked_t* mtracked_create(pp_bmc_mtrack_cb_t sc_cb,
				   pp_bmc_mtrack_to_cb_t to_cb,
				   int to_id, void* ctx) {
    mtracked_t* mt = malloc(sizeof(mtracked_t));
    
    mt->sc_cb = sc_cb;
    mt->to_cb = to_cb;
    mt->to_id = to_id;
    mt->ctx = ctx;
    return mt;
}

static inline void mtracked_destroy(pp_mallocator_t* a, void* p) {
    a->free(a, p);
}

static inline unsigned short msg_key_create(unsigned char chan,
					    unsigned char seqno) {
    return chan << 8 | seqno;
}

static inline unsigned char msg_key_to_chan(unsigned short msg_key) {
    return (unsigned char)(msg_key >> 8);
}

static inline unsigned char msg_key_to_seqno(unsigned short msg_key) {
    return (unsigned char)(msg_key & 0xff);
}

static int timeout_handler (const int item_id UNUSED, void* ctx) {
    unsigned short msg_key = (unsigned short)(long)ctx;
    mtracked_t* mtracked = pp_rb_tree_search(msg_dict, (void*)(long)msg_key);
    assert(mtracked);

    pp_bmc_seqno_release(msg_key_to_chan(msg_key),
			 msg_key_to_seqno(msg_key));
    
    if (mtracked->to_cb) {
	mtracked->to_cb(mtracked->ctx);
    } 

    pp_rb_tree_remove(msg_dict, (void*)(long)msg_key, NULL);

    return PP_SUC;
}

int pp_bmc_mtrack_init() {
    assert(msg_dict == NULL);
    msg_dict = pp_rb_tree_int_new();
    return PP_SUC;
}

void pp_bmc_mtrack_cleanup() {
    if (msg_dict != NULL) {
	pp_rb_iter_t* i = pp_rb_iter_new(msg_dict);
	mtracked_t* mtracked;
	void* key;
	unsigned short msg_key;
	
	while (pp_rb_iter_next(i)) {
	    /* call timeout_cb, caller may need this for cleanup */
	    mtracked = pp_rb_iter_data(i);
	    if (mtracked->to_cb) {
		mtracked->to_cb(mtracked->ctx);
	    }
	    /* release sequence numbers */
	    key = pp_rb_iter_key(i);
	    msg_key = (unsigned short)(int)key;
	    pp_bmc_seqno_release(msg_key_to_chan(msg_key),
				 msg_key_to_seqno(msg_key));
	}
	/* and all the rest */
	pp_rb_tree_destroy(msg_dict);
	msg_dict = NULL;

        pp_rb_iter_destroy(i);
    }
}

int pp_bmc_mtrack_send_msg(imsg_t* imsg, pp_bmc_mtrack_cb_t sc_cb,
			   pp_bmc_mtrack_to_cb_t to_cb, void* ctx) {
    mtracked_t* mtracked;
    int rb_ret, to_id, msg_key, ret;
    assert(sc_cb != NULL);
    
    imsg->rq_seq = pp_bmc_seqno_assign(imsg->chan);
    msg_key = msg_key_create(imsg->chan, imsg->rq_seq);

    if (PP_ERR != (ret = pp_bmc_router_send_msg(imsg))) {
	/* set timeout and insert msg into pending response table */
	to_id = pp_select_add_to(PP_BMC_BRIDGE_RSP_TIMEOUT,
				0, timeout_handler, (void*)(long)msg_key);
	mtracked = mtracked_create(sc_cb, to_cb, to_id, ctx);
	rb_ret = pp_rb_tree_insert (msg_dict, (void*)(long)msg_key, mtracked,
				    mtracked_destroy, 0);
	assert(rb_ret == 0);
    } else {
	pp_bmc_seqno_release(imsg->chan, imsg->rq_seq);
    }

    return ret;
}
    
int pp_bmc_mtrack_handle(imsg_t* imsg) {
    mtracked_t* mtracked;
    int msg_key = msg_key_create(imsg->chan, imsg->rq_seq);
    
    if (NULL == (mtracked = pp_rb_tree_search(msg_dict, (void*)(long)msg_key)))
	return PP_ERR;

    pp_select_remove_to(mtracked->to_id);
    pp_bmc_seqno_release(imsg->chan, imsg->rq_seq);
    
    if (PP_ERR == mtracked->sc_cb(imsg, mtracked->ctx))
	pp_bmc_log_pwarn("[APP]:pp_bmc_mtrack_handle: callback fkt failed");

    pp_rb_tree_remove(msg_dict, (void*)(long)msg_key, NULL);

    return PP_SUC;
}	


