/**
 * \file debug.c
 *
 * Description: Logging Code
 *
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <time.h>
#include <sys/time.h>
#include <stdio.h>
#include <pp/base.h>
#include <pp/bmc/debug.h>

static const char *level_str[] = {
    "      ",
    "FATAL ",
    "ALERT ",
    "CRITIC",
    "ERROR ",
    "WARN  ",
    "NOTICE",
    "INFO  ",
    "DEBUG ",
};

#define BRING_COLOR_TO_LIFE
#define noCOLOR_ON_BLACK_SCREEN

#ifdef BRING_COLOR_TO_LIFE
#define BLACK   "0"
#define RED     "1"
#define GREEN   "2"
#define YELLOW  "3"
#define BLUE    "4"
#define MAGENTA "5"
#define CYAN    "6"
#define WHITE   "7"
#define DK      "0"
#define LT      "1"

// color code is: "ESC[<brightness>;<fg col>;<bg col>
// fg and bg col may be missing, this means "use defaults"

#define COLOR(b, c) "\x1b[" b ";3" c "m"
#define DEFCOL "\x1b[0m"

static const char *color_code[] = {
    DEFCOL,             // default
    COLOR(DK, RED),     // FATAL
    COLOR(DK, RED),     // ALERT
    COLOR(DK, RED),     // CRITIC
    COLOR(DK, RED),     // ERROR
    COLOR(DK, YELLOW),  // WARN
    COLOR(DK, CYAN),    // NOTICE
    COLOR(DK, GREEN),   // INFO
    COLOR(DK, MAGENTA),   // DEBUG
};
#endif

static void pp_bmc_vlog_core(int level, const char* format, va_list ap) {
    struct timeval tv;
    time_t t;
    struct tm *t2;
    int errorno;

    if (level < 0) level = 0;
    else if (level > PP_BMC_LOG_DEBUG) level = PP_BMC_LOG_DEBUG;

    errorno = errno; // save errno, localtime may overwrite it
    gettimeofday(&tv, (void*)0);
    t = time((void*)0);
    t2 = localtime(&t);
    errno = errorno; // restore errno

    fprintf(stderr, "%.4d-%.2d-%.2d %.2d:%.2d:%.2d.%.3d ",
        t2->tm_year + 1900, t2->tm_mon + 1, t2->tm_mday,
        t2->tm_hour, t2->tm_min, t2->tm_sec, (int)(tv.tv_usec/1000));

#ifdef BRING_COLOR_TO_LIFE
    fprintf(stderr, color_code[level]);
#endif

    fprintf(stderr, "%s ", level_str[level]);
    vfprintf(stderr, format, ap);

#ifdef BRING_COLOR_TO_LIFE
    fprintf(stderr, color_code[0]);
#endif
}

void pp_bmc_vlog(int level, const char* format, va_list ap) {
    pp_bmc_vlog_core(level, format, ap);
    fprintf(stderr, "\n");
}


void pp_bmc_log(int level, const char* format, ...) {
    va_list ap;
    va_start(ap, format);
    pp_bmc_vlog(level, format, ap);
    va_end(ap);
}

void pp_bmc_plog(int level, const char* format, ...) {
    va_list ap;
    va_start(ap, format);
    pp_bmc_vlog_core(level, format, ap);
    fprintf(stderr, ": %s\n", pp_error_string(errno));
    va_end(ap);
}

#if PP_BMC_DEBUG_LEVEL>=8
void pp_bmc_printbuf(const unsigned char* prefix,
                     const unsigned char* buf, int buf_len)
{
    if (prefix == NULL) {
        pp_bmc_log_debug("printbuf (%d)", buf_len);
    } else {
        pp_bmc_log_debug("%s (%d)", prefix, buf_len);
    }
    
    int i;
    for (i=0; i<buf_len; i++) {
        if ((i%16) == 0) {
             if (i != 0)
                 printf("\n ");
             else
                 printf("<");
        } else {
            if ((i%4) == 0)
                printf(" ");
        }
        printf("%.2x ", buf[i]);
    }
    printf(">\n");
}
#else    
void pp_bmc_printbuf(const unsigned char* prefix UNUSED,
                     const unsigned char* buf UNUSED, int buf_len UNUSED) {}
#endif
