/**
 * session_manager.h
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * Basic session management. Assigns session ids, passwords etc.
 * Provides mapping to erla usermanagement.
 */

#ifndef __SESSION_MANAGER_H__
#define __SESSION_MANAGER_H__

#include <pp/bmc/bmc_imsg.h>

/**
 * Create an empty, unactivated session.
 * Dont use this until you know what you do.
 * 
 * The resulting session is stored in the internal
 * list and a session inactivity timeout is added. 
 * The session is refcounted for internal use only.
 * Do not call pp_bmc_session_delete on the result.
 * 
 * @return NULL if the logical session device is full
 */
imsg_session_t* pp_bmc_create_session(void);

/**
 * Tries to create an IPMI v1.5 session based on the given username and channel.
 * If a session can be established, an unactivated imsg_session_t is returned.
 * If the requested username is invalid NULL is returned. An non-NULL return
 * value does not automatically mean the the session can be activated. The
 * activation step may still fail because the requested authtype is not
 * available for the requested privilege level.
 * The returned session is refcounted, use pp_bmc_session_delete to free the it.
 */
imsg_session_t* pp_bmc_create_session15(unsigned char* username, unsigned char channel);

/**
 * Create a RMCP+ session for IPMIv2.0 LAN. The session will be unactivated
 * and empty. It is refcounted, so use pp_bmc_session_delete to free the session.
 */
imsg_session_t* pp_bmc_create_session20(unsigned char req_priv_lvl, 
                                        unsigned char auth_cipher,
                                        unsigned char integrity_cipher,
                                        unsigned char conf_cipher);

/**
 * Bind an existing IPMIv2.0 LAN session to username, privilege level and
 * cipher suites. This is necessary for RMCP+ authentication.
 * The session reference count is not affected by this command.
 */
int pp_bmc_open_session20(imsg_session_t* session, unsigned char username_len,
                          unsigned char* username, unsigned char name_only_lookup,
                          unsigned int req_max_priv);

/**
 * Get a session by its session id and increase the session reference
 * count. Use pp_bmc_session_delete to release the retrieved session.
 * @return NULL if the specified session does not exist
 */
imsg_session_t* pp_bmc_get_session(unsigned int sessionId);

/**
 * Get a session by its session handle and increase the session reference
 * count. Use pp_bmc_session_delete to release the retrieved session. This
 * function is usually only used by GetSessionInfo.
 * @return NULL if the specified session does not exist
 */
imsg_session_t* pp_bmc_get_session_by_handle(unsigned char session_handle);

/**
 * Get the n-th active session and increase the session reference count.
 * Index 0 is reserved. Use pp_bmc_session_delete to release the retrieved
 * session. This function is usually only used by GetSessionInfo.
 * @return NULL if the specified session does not exist
 */
imsg_session_t* pp_bmc_get_nth_session(unsigned char n);

/**
 * duplicates a session by incrementing its ref-count
 */
static inline imsg_session_t* pp_bmc_session_dup(imsg_session_t* session) {
    if (session != NULL) session->ref_count++;
    return session;
}

/**
 * Close the specified session. Returns PP_SUC or
 * PP_ERR if session does not exist
 */
int pp_bmc_close_session(unsigned int sessionId);

/**
 * Iterates through all active sessions and close each one
 */
void pp_bmc_session_manager_close_all_sessions(void);

/**
 * Call this function to refresh the session inactivity timeout.
 * This function must be called whenever a valid message is received
 * on a sessionbased channel to prevent the session from being 
 * automatically closed.
 */
void pp_bmc_session_refresh(imsg_session_t* session);

/**
 * Init the session manager
 */
int pp_bmc_session_manager_init(void);

/**
 * Cleanup session manager and close remaining sessions
 */
void pp_bmc_session_manager_cleanup(void);

/**
 * Gets number of open sessions for the requested channel
 * 
 * @param chan_id Desired Channel ID
 * @return number of open sessions 
 */
unsigned char pp_bmc_session_manager_get_session_cnt(unsigned int chan_id);

/**
 * Gets total number of open sessions
 * 
 * @return total number of open sessions 
 */
unsigned char pp_bmc_session_manager_get_all_session_cnt(void);

#endif
