/**
 * ipmi_sdr.c
 *
 * some utils to deal with sensor data records
 * 
 * (c) 2004 Peppercon AG, 12/20/2004, tbr@peppecon.de
 */

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/ipmi_chan.h>

static ipmi_sdr_full_sensor_t full_sdr_template = {
    header: {
	/* id: reserved to be used by SDRR */
	version: 0x51,
	type:    0x01,
	length:  sizeof(ipmi_sdr_full_sensor_t) - sizeof(ipmi_sdr_header_t)
    },
    key: {
	owner_id:   PP_BMC_IPMB_ADDR,
	lun:        0,
	channel:    0,
	sensor_num: 0   /* is filled out by concrete sensor */
    },
    entity: {
	id:         IPMI_ENTITY_ID_UNSPECIFIED,
	instance:   0x0,
	is_logical: 0
    },
    sensor: {
	init: {
	    sensor_scan  : 1, /* scanning initially enabled */
	    event_gen    : 1, /* events initially enabled  */
	    type         : 0,
	    hysteresis   : 1, /* ipmitool wants this to accept hysteresis */
	    thresholds   : 1, /* ipmitool needs this to accept threshold */
	    events       : 0, 
	    scanning     : 0
	},
	caps: {
	    event_msg    : 0x0, /* non */
	    threshold    : 0x0, /* non */
	    hysteresis   : 0x0, /* non */ 
	    rearm        : 1,   /* auto rearm */
	    ignore       : 0    /* don't ignore, if entity is not present */
	},
	type:       IPMI_SENSOR_TYPE_RESERVED,
	event_type: IPMI_EVENT_READING_TYPE_UNSPECIFIED,
    },
    mask: {
	threshold: {
	    assert_event_le16: cpu_to_le16_const(0),  /* no assertion event */
	    deassert_event_le16: cpu_to_le16_const(0),/*no deassertion event */
	    set:   0,                              /* no threshold setable */
	    read:  0                               /* no threshold readable */
	}
    },
    unit: {
	pct:       0,              /* no percentage */
	modifier:  IPMI_MODIFIER_UNIT_NONE,
	rate:      IPMI_RATE_UNIT_NONE,
	analog:    0,
	type: {
	    base:      0,
	    modifier:  IPMI_UNIT_TYPE_UNSPECIFIED
	},
    },
    linearization: IPMI_LINEARIZATION_LINEAR,
    mtol:          0, /* y = mx + b */
    bacc:          0, /* y = mx + b */
    direction:     0, /* sensor direction       */
    accexp:        0, /* accuracy exp, unsigned */
    acc:           0, /* accuracy, MS 4 bit     */
    bexp:          0, /* b exponent, 2'compl, signed */
    rexp:          0, /* result exponent, 2'compl, signed */

    analog_flag: {
	nominal_read: 1, /* nominal read specified */
	normal_max:   1, /* normal max specified */
	normal_min:   1  /* normal min specified */
    },
    nominal_read:     0,
    normal_max:       0,
    normal_min:       0,
    sensor_max:       0,
    sensor_min:       0,
    threshold: {           /* all thresholds are ignored for the moment */
	upper: {
	    non_recover:    0,
	    critical:       0,
	    non_critical:   0
	},
	lower: {
	    non_recover:    0,
	    critical:       0,
	    non_critical:   0
	},
	hysteresis: {
	    positive:       0,
	    negative:       0
	},
    },
    oem: 0,
    id: {
	length: 16,
	code:   IPMI_STR_ASCII8,
	string: "" /* will be filled in by the instance */
    }
};

static ipmi_sdr_compact_sensor_t compact_sdr_template = {
    header: {
	/* id: reserved to be used by SDRR */
	version: 0x51,
	type:    IPMI_SDR_COMPACT_SENSOR,
	length:  sizeof(ipmi_sdr_compact_sensor_t) - sizeof(ipmi_sdr_header_t)
    },
    key: {
	owner_id: PP_BMC_IPMB_ADDR,
	lun:      0,
	channel:  0
	/* sensor_num: is filled out by concrete sensor */
    },
    entity: {
	id:         IPMI_ENTITY_ID_UNSPECIFIED,
	instance:   0x0,
	is_logical: 0 /* TODO: entity groups? */
    },
    sensor: {
	init: {
	    sensor_scan  : 1, /* scanning initially enabled */
	    event_gen    : 1, /* events initially disabled  */
	    type         : 0,
	    hysteresis   : 0,
	    thresholds   : 0,
	    events       : 0, 
	    scanning     : 0
	},
	caps: {
	    event_msg    : 0x3, /* non */
	    threshold    : 0x0, /* non */
	    hysteresis   : 0x0, /* non */ 
	    rearm        : 1,   /* auto rearm */
	    ignore       : 0    /* don't ignore, if entity is not present */
	},
	type:       IPMI_SENSOR_TYPE_RESERVED,
	event_type: IPMI_EVENT_READING_TYPE_UNSPECIFIED,
    },
    mask: {
	discrete: {
	    assert_event_le16: cpu_to_le16_const(0), /* no assertion event */
	    deassert_event_le16: cpu_to_le16_const(0), /*no deassertion event*/
	    read_le16: cpu_to_le16_const(0)
	}
    },
    unit: {
	pct:       0,              /* no percentage */
	modifier:  IPMI_MODIFIER_UNIT_NONE,
	rate:      IPMI_RATE_UNIT_NONE,
	analog:    IPMI_ANALOG_DATA_FORMAT_NOT_ANALOG,
	type: {
	    base:      0,
	    modifier:  IPMI_UNIT_TYPE_UNSPECIFIED
	},
    },
    share: {
	count:		0,
	mod_type:	0,
	mod_offset:	0,
	entity_inst:	0,
    },
    threshold: {
	hysteresis: {
	    positive:	0,
	    negative:	0,
	},
    },
    oem: 0,
	id: {
	length: 16,
	code:   IPMI_STR_ASCII8,
	string: "" /* will be filled in by the instance */
    }    
};

ipmi_sdr_full_sensor_t* ipmi_sdr_full_create(pp_tp_sdr_init_values_t *sv) {    
    ipmi_sdr_full_sensor_t* sdr;

    sdr = malloc(sizeof(ipmi_sdr_full_sensor_t));
    memcpy(sdr, &full_sdr_template, sizeof(ipmi_sdr_full_sensor_t));

    ipmi_sdr_fill_sdr((ipmi_sdr_header_t*)sdr, sv);

    sdr->unit.analog = sv->adf;
    sdr->unit.rate = sv->run;
    sdr->unit.modifier = sv->mnc;
    sdr->unit.pct = sv->pct;
    sdr->unit.type.base = sv->bun;
    sdr->unit.type.modifier = sv->mun;

    sdr->mask.threshold.assert_event_le16 = sv->mask.threshold.ltm;
    sdr->mask.threshold.deassert_event_le16 = sv->mask.threshold.utm;
    sdr->mask.threshold.set = sv->mask.threshold.stm;
    sdr->mask.threshold.read = sv->mask.threshold.rtm;
    
    sdr->linearization = sv->type.full.lin;
    sdr->mtol = PP_BMC_SDR_M_TOL_2_MTOL(sv->type.full.m, sv->type.full.tol);
    sdr->bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(sv->type.full.b, sv->type.full.acc);
    sdr->acc = ((u_char)sv->type.full.acc & 0x3C0) >> 6;
    sdr->accexp = sv->type.full.aex;
    sdr->direction = sv->type.full.sed;

    sdr->rexp = sv->type.full.rex;
    sdr->bexp = sv->type.full.bex;

    *(unsigned char*)&sdr->analog_flag = sv->type.full.acf;

    sdr->nominal_read = sv->type.full.nrd;
    sdr->normal_max = sv->type.full.nma;
    sdr->normal_min = sv->type.full.nmi;
    sdr->sensor_max = sv->type.full.sma;
    sdr->sensor_min = sv->type.full.smi;

    sdr->threshold.upper.non_recover = sv->type.full.nru;
    sdr->threshold.upper.critical = sv->type.full.cru;
    sdr->threshold.upper.non_critical = sv->type.full.ncu;

    sdr->threshold.lower.non_recover = sv->type.full.nrl;
    sdr->threshold.lower.critical = sv->type.full.crl;
    sdr->threshold.lower.non_critical = sv->type.full.ncl;

    sdr->threshold.hysteresis.positive = sv->pos;
    sdr->threshold.hysteresis.negative = sv->neg;

    sdr->oem = sv->oem;

    return sdr;
}

ipmi_sdr_compact_sensor_t* ipmi_sdr_compact_create(pp_tp_sdr_init_values_t *sv) {
    ipmi_sdr_compact_sensor_t* sdr;

    sdr = malloc(sizeof(ipmi_sdr_compact_sensor_t));
    memcpy(sdr, &compact_sdr_template, sizeof(ipmi_sdr_compact_sensor_t));

    ipmi_sdr_fill_sdr((ipmi_sdr_header_t*)sdr, sv);

    sdr->unit.analog = sv->adf;
    sdr->unit.rate = sv->run;
    sdr->unit.modifier = sv->mnc;
    sdr->unit.pct = sv->pct;
    sdr->unit.type.base = sv->bun;
    sdr->unit.type.modifier = sv->mun;

    sdr->mask.discrete.assert_event_le16 = sv->mask.discrete.eam;
    sdr->mask.discrete.deassert_event_le16 = sv->mask.discrete.edm;
    sdr->mask.discrete.read_le16 = sv->mask.discrete.rdm;
    
    sdr->share.direction = sv->type.compact.sed;
    sdr->share.mod_type = sv->type.compact.ist;
    sdr->share.count = sv->type.compact.shc;
    sdr->share.mod_offset = sv->type.compact.eis;
    sdr->share.entity_inst = sv->type.compact.iso;
    
    sdr->threshold.hysteresis.positive = sv->pos;
    sdr->threshold.hysteresis.negative = sv->neg;

    sdr->oem = sv->oem;

    return sdr;
}

int ipmi_sdr_fill_sdr(ipmi_sdr_header_t* this, pp_tp_sdr_init_values_t *sv) {    
    ipmi_sdr_key_sensor_t* key;
    ipmi_sdr_entity_t* entity;
    ipmi_sdr_sensor_t* sensor;
    ipmi_sdr_id_t* id;
    
    int len;

    if (NULL != (key = ipmi_sdr_get_key_sensor_ptr(this))) {
        key->owner_id = PP_BMC_IPMB_ADDR;
        key->lun = 0;
        key->channel = IPMI_CHAN_PRIMARY_IPMB;
        key->sensor_num = sv->num;
    }

    if (NULL != (entity = ipmi_sdr_get_entity_ptr(this))) {
	entity->id = sv->eid;    
	entity->instance = sv->ein & 0x7f;
	entity->is_logical = (sv->ein & 0x80) >> 7;
    }

    if (NULL != (sensor = ipmi_sdr_get_sensor_ptr(this))) {
	*(unsigned char*)&sensor->caps = sv->cap;
	sensor->type = sv->sty;
	sensor->event_type = sv->erc;
    }

    if ((len = strlen(sv->ids)) > (IPMI_SENSOR_ID_STRING_LEN - 1)) {
	len = IPMI_SENSOR_ID_STRING_LEN - 1;
    }

    if (NULL != (id = ipmi_sdr_get_id_ptr(this))) {
        id->code = IPMI_STR_ASCII8;
        id->length = len;
        strncpy(id->string, sv->ids, len);
        id->string[len] = '\0';
    }

    return PP_SUC;
}

ipmi_sdr_full_sensor_t* ipmi_sdr_full_part_create(unsigned char sens_type,
						  unsigned char reading_type,
						  unsigned char base_unit,
						  unsigned char analog_format,
						  unsigned short mtol,
						  unsigned short bacc,
						  char bexp,
						  char rexp,
						  unsigned char sens_max,
						  unsigned char sens_min) {
    ipmi_sdr_full_sensor_t* sdr;

    sdr = malloc(sizeof(ipmi_sdr_full_sensor_t));
    memcpy(sdr, &full_sdr_template, sizeof(ipmi_sdr_full_sensor_t));

    sdr->sensor.type = sens_type;
    sdr->sensor.event_type = reading_type;
    sdr->unit.type.base = base_unit;
    sdr->unit.analog = analog_format;
    sdr->mtol = mtol;
    sdr->bacc = bacc;
    sdr->bexp = bexp;
    sdr->rexp = rexp;
    sdr->sensor_max = sens_max;
    sdr->sensor_min = sens_min;

   // TODO: we have probably to prepare some more fields here as like following:
   if (reading_type != 1) sdr->unit.analog = 0x3; // no analog reading

    return sdr;
}

ipmi_sdr_compact_sensor_t* ipmi_sdr_compact_part_create(unsigned char sens_type,
						        unsigned char reading_type) {
    ipmi_sdr_compact_sensor_t* sdr;

    sdr = malloc(sizeof(ipmi_sdr_compact_sensor_t));
    memcpy(sdr, &compact_sdr_template, sizeof(ipmi_sdr_compact_sensor_t));

    sdr->sensor.type = sens_type;
    sdr->sensor.event_type = reading_type;
    return sdr;
}

ipmi_sdr_header_t* ipmi_sdr_copy(ipmi_sdr_header_t* this) {
    ipmi_sdr_header_t* that;
    size_t rec_size = sizeof(ipmi_sdr_header_t) + this->length;
    that = malloc(rec_size);
    memcpy(that, this, rec_size);
    return that;
}


/* Note:
 * ipmitool prints the real max and min values as 'unspecified'
 * so we code here 1 above/below the min/max value */
unsigned char ipmi_sdr_min_by_form(unsigned char analog_format) {
    unsigned ret;
    switch(analog_format) {
      case IPMI_ANALOG_DATA_FORMAT_UNSIGNED:    ret = 0x1; break;
      case IPMI_ANALOG_DATA_FORMAT_1_COMPL:     ret = 0xfe; break;
      case IPMI_ANALOG_DATA_FORMAT_2_COMPL:     ret = 0x81; break;
      case IPMI_ANALOG_DATA_FORMAT_NOT_ANALOG:  // fall through
      default:                                  ret = 0x0;
    }
    return ret;
}

unsigned char ipmi_sdr_max_by_form(unsigned char analog_format) {
    unsigned ret;
    switch(analog_format) {
      case IPMI_ANALOG_DATA_FORMAT_UNSIGNED:    ret = 0xfe; break;
      case IPMI_ANALOG_DATA_FORMAT_1_COMPL:     // fall through
      case IPMI_ANALOG_DATA_FORMAT_2_COMPL:     ret = 0x7e; break;
      case IPMI_ANALOG_DATA_FORMAT_NOT_ANALOG:  // fall through
      default:                                  ret = 0x0;
    }
    return ret;
}

ipmi_sdr_id_t* ipmi_sdr_get_id_ptr(ipmi_sdr_header_t* sdr) {
    ipmi_sdr_id_t* p;
    switch (sdr->type) {
      case IPMI_SDR_FULL_SENSOR:
	  p = &((ipmi_sdr_full_sensor_t*)sdr)->id;
	  break;
      case IPMI_SDR_COMPACT_SENSOR:
	  p = &((ipmi_sdr_compact_sensor_t*)sdr)->id;
	  break;
      case IPMI_SDR_EVENTONLY:
	  p = &((ipmi_sdr_eventonly_sensor_t*)sdr)->id;
	  break;
      default:
	  p = NULL;
    }
    return p;
}

unsigned char* ipmi_sdr_get_sensor_type_ptr(ipmi_sdr_header_t* sdr) {
    switch (sdr->type) {
      case IPMI_SDR_FULL_SENSOR:    
	  return &((ipmi_sdr_full_sensor_t*)sdr)->sensor.type;
      case IPMI_SDR_COMPACT_SENSOR: 
	  return &((ipmi_sdr_compact_sensor_t*)sdr)->sensor.type;
      case IPMI_SDR_EVENTONLY:      
	  return &((ipmi_sdr_eventonly_sensor_t*)sdr)->sensor_type;
      default:                      
	  return NULL;
    }
}

unsigned char* ipmi_sdr_get_event_type_ptr(ipmi_sdr_header_t* sdr) {
    switch (sdr->type) {
      case IPMI_SDR_FULL_SENSOR:    
	  return &((ipmi_sdr_full_sensor_t*)sdr)->sensor.event_type;
      case IPMI_SDR_COMPACT_SENSOR: 
	  return &((ipmi_sdr_compact_sensor_t*)sdr)->sensor.event_type;
      case IPMI_SDR_EVENTONLY:      
	  return &((ipmi_sdr_eventonly_sensor_t*)sdr)->event_type;
      default:                      
	  return NULL;
    }
}

ipmi_sdr_key_sensor_t* ipmi_sdr_get_key_sensor_ptr(ipmi_sdr_header_t* sdr) {
    ipmi_sdr_key_sensor_t* p;
    switch (sdr->type) {
      case IPMI_SDR_FULL_SENSOR:
	  p = &((ipmi_sdr_full_sensor_t*)sdr)->key;
	  break;
      case IPMI_SDR_COMPACT_SENSOR:
	  p = &((ipmi_sdr_compact_sensor_t*)sdr)->key;
	  break;
      case IPMI_SDR_EVENTONLY:
	  p = &((ipmi_sdr_eventonly_sensor_t*)sdr)->key;
	  break;
      default:
	  p = NULL;
    }
    return p;
}

ipmi_sdr_entity_t* ipmi_sdr_get_entity_ptr(ipmi_sdr_header_t* sdr) {
    switch (sdr->type) {
      case IPMI_SDR_FULL_SENSOR:    
	  return &((ipmi_sdr_full_sensor_t*)sdr)->entity;
      case IPMI_SDR_COMPACT_SENSOR: 
	  return &((ipmi_sdr_compact_sensor_t*)sdr)->entity;
      default:                      
	  return NULL;
    }
}

ipmi_sdr_sensor_t* ipmi_sdr_get_sensor_ptr(ipmi_sdr_header_t* sdr) {
    switch (sdr->type) {
      case IPMI_SDR_FULL_SENSOR:    
	  return &((ipmi_sdr_full_sensor_t*)sdr)->sensor;
      case IPMI_SDR_COMPACT_SENSOR: 
	  return &((ipmi_sdr_compact_sensor_t*)sdr)->sensor;
      default:                      
	  return NULL;
    }
}

ipmi_sdr_evt_mask_t* ipmi_sdr_get_event_mask_ptr(ipmi_sdr_header_t* sdr) {
    switch (sdr->type) {
      case IPMI_SDR_FULL_SENSOR:    
	  return &((ipmi_sdr_full_sensor_t*)sdr)->mask;
      case IPMI_SDR_COMPACT_SENSOR: 
	  return &((ipmi_sdr_compact_sensor_t*)sdr)->mask;
      default:                      
	  return NULL;
    }
}

ipmi_sdr_threshold_t* ipmi_sdr_get_threshold_ptr(ipmi_sdr_header_t* sdr) {
    switch (sdr->type) {
      case IPMI_SDR_FULL_SENSOR:
	  return &((ipmi_sdr_full_sensor_t*)sdr)->threshold;
      default:
	  return NULL;
    }
}

unsigned char ipmi_sdr_get_analog_data_format(ipmi_sdr_header_t*sdr) {
    switch (sdr->type) {
      case IPMI_SDR_FULL_SENSOR:
	  return ((ipmi_sdr_full_sensor_t*)sdr)->unit.analog;
      default:
	  return IPMI_ANALOG_DATA_FORMAT_NOT_ANALOG;
    }
    
}

int ipmi_sdr_is_discrete(ipmi_sdr_header_t* sdr) {
    unsigned char* evt_type_ptr;
    evt_type_ptr = ipmi_sdr_get_event_type_ptr(sdr);
    return ipmi_sdr_event_type_is_discrete(*evt_type_ptr);
}

int ipmi_sdr_event_type_is_discrete(unsigned char evt_type) {
    return (/* generic discrete */
	    (evt_type >= IPMI_EVENT_READING_TYPE_DISCRETE_USAGE  &&
	     evt_type <= IPMI_EVENT_READING_TYPE_DISCRETE_ACPI_POWER) ||
	    /* sensor specific discrete */
	     evt_type == IPMI_EVENT_READING_TYPE_SENSOR_SPECIFIC ||
	    /* OEM discrete */
	    (evt_type >= 0x70 && evt_type <= 0x7F));
}


/* pp_bmc_utos  -  convert unsigned 32bit value to 2's complement signed
 *
 * @val:	unsigned value to convert
 * @bits:	number of bits in value
 *
 * returns 2s complement signed integer
 */
int32_t
pp_bmc_utos(uint32_t val, int bits)
{
	int x;

	assert(bits > 0);

	x = 1 << (bits-1);
	if (val & x) return -(int)((~val & (x-1))+1);
	return val;
}
