/**
 * user_manager.c
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 *
 * Basic user management. Allows access to the lara UM.
 * Both empty usernames and role only logins (for IPMIv20
 * null users) are possible with this interface. Configured
 * users will be returned even if their ipmi privilege
 * level is stil unspecified.
 * 
 * Practically anonymous users are not implemented
 * yet. Setting null usernames will yield an error.
 * 
 * Note: anonymous users are not necessarily mapped to
 * IPMI user ID 1.
 */

/*
 * Implementation notes:
 * 
 * 1. IPMI user IDs are implicitly assigned to IPMI enabled lara
 * users at runtime (as soon as the usercache is updated). The
 * uid is only needed to manage the users through ipmi. Lara 
 * users do not need it to access ipmi functionality. If lara
 * users have no uid, the config key is not present.
 * 
 * 2. In this implementation, user_enable is derived from the
 * privilege level. As a consequence, the old privilege level
 * will be lost if a user is disabled.
 * 
 * 3. anonymous users could be mapped to lara users by naming
 * conventions (eg 'IPMI-anonymous-user').
 * 
 * 4. user id 0 is reserved.
 *
 * 5. user id 1 is reserved for the null user. As we currently
 * have no null user user id 1 is not used at all.
 */


#include <pp/bmc/user_manager.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/utils.h>

#include <pp/base.h>
#include <pp/vector.h>
#include <pp/um.h>
#include <pp/cfg.h>
#include <pp/xdefs.h>

#define PP_CFGKEY_IPMI_UID           "user[%u].ipmi_uid"
#define PP_CFGKEY_IPMI_PRIV_LEVEL    "user[%u].ipmi.channel[%u].priv_level"
#define PP_CFGKEY_IPMI_CALLBACK_ONLY "user[%u].ipmi.channel[%u].callback_only"
#define PP_CFGKEY_IPMI_LINK_AUTH     "user[%u].ipmi.channel[%u].link_auth"
#define PP_CFGKEY_IPMI_MSG_ENABLED   "user[%u].ipmi.channel[%u].msg_enabled"

#define PP_CFGKEY_IPMI_PRIV_LEVEL_ACL   "ipmi_priv"
#define PP_CFGKEY_IPMI_MAY_USE_SOL_ACL  "ipmi_sol"
#define PP_CFGKEY_CFGTOOL_ACL           "cfgtool"

#define MAX_UID 63

// dummy libpp_um principal name for unnamed IPMI users
#define ANONYMOUS_USER_PREFIX "_ipmi_"

/*
 * User Fast Lookup Table
 * - stores uid<->user associations for faster access
 * - will be validated at each access
 */

static char* user_cache[MAX_UID + 1];

static void init_user_cache(void)
{
    memset(user_cache, 0, sizeof(user_cache));
}

static void clear_user_cache(void)
{
    int i;

    for (i = 0; i <= MAX_UID; i++)
        if (user_cache[i]) {
            free(user_cache[i]);
            user_cache[i] = NULL;
        }
}

static void update_user_cache(void)
{
    vector_t *usrs;
    clear_user_cache();
    if (!(usrs = pp_um_get_all_users())) return;
    while (vector_size(usrs) > 0) {
	int uid, pp_uid;
	char *name = vector_get(usrs, 0);
	pp_uid = pp_um_user_get_uid(name);
	if (pp_uid >= 0 && PP_SUCCED( pp_cfg_get_int_nodflt(&uid, PP_CFGKEY_IPMI_UID, pp_uid) )) {
	    user_cache[uid] = strdup(name);
	}
	vector_remove(usrs, 0);
    }
    vector_delete(usrs);
}

/*
 * Internal Helpers for accessing the usercache in a save way
 */

static const char* get_name_by_uid(int uid)
{
    assert(uid >= 1 && uid <= MAX_UID);

    // look into cache
    if (!user_cache[uid]) {
        // not found in cache: update cache for surety
        pp_bmc_log_debug("[UserMan] cache invalid: cant find user %d", uid);
        // update cache to retrieve the newest values for this user
        update_user_cache();
    } else {
        int myuid;
        // validate cache info
        int pp_uid = pp_um_user_get_uid(user_cache[uid]);
        if ( (pp_uid < 0) ||
             (pp_cfg_get_int(&myuid, PP_CFGKEY_IPMI_UID, pp_uid) == PP_ERR) ||
             (myuid != uid) )
        {
            // the cache entry is not correct, better update the whole cache
            update_user_cache();
        }
    }

    return user_cache[uid];
}

static int get_uid_by_name(const char *name)
{
    int uid = PP_ERR;
    int pp_uid;

    // look into config
    pp_uid = pp_um_user_get_uid(name);
    if (pp_uid < 0 || pp_cfg_get_int(&uid, PP_CFGKEY_IPMI_UID, pp_uid) == PP_ERR) 
    {
        // specified username does not exist
        return PP_ERR;
    }
    // uid is the value that we need
    
    // validate the cache, not necessary for correct completion of this function
    if (pp_strcmp_safe(name, user_cache[uid]) != 0) {
        pp_bmc_log_debug("[UserMan] cache invalid: user '%s' does not match with cache user '%s'", name, user_cache[uid]);
        update_user_cache();
    }

    return uid;
}

static int create_new_user(const char* name, unsigned char uid, int* pp_uid)
{
    int id;

    assert(name);
    if (name[0] == '\0') {
	// create dummy name for anonymous user
	char dummy[20];
	snprintf(dummy, sizeof(dummy), ANONYMOUS_USER_PREFIX "%d", uid);
	name = dummy;
    }

    // create a new lara/eric user
    if (PP_FAILED(pp_um_user_create(name, NULL, 0, NULL, NULL, uid))
        || PP_FAILED(pp_um_user_set_enabled(name, 0))
        || PP_FAILED(pp_cfg_save(DO_FLUSH))
        || (id = pp_um_user_get_uid(name)) < 0) {
        return PP_ERR;
    }

    free(user_cache[uid]);
    user_cache[uid] = strdup(name);

    if (pp_uid) *pp_uid = id;
    return PP_SUC;
}

// check whether a user with the given IPMI uid exists; create if necessary
static const char *create_user_if_absent(unsigned char uid)
{
    update_user_cache();
    if (!user_cache[uid]) {
	if (PP_FAILED( create_new_user("", uid, NULL) )) return NULL;
    }
    return user_cache[uid];
}

// delete user if all relevant data (name, password, privilege, ...) is empty
static int delete_user_if_unused(unsigned char uid)
{
    const char *name;
    char *password = NULL;
    int pp_uid, channel;
    int ret = 0; // nothing deleted

    name = get_name_by_uid(uid);
    pp_uid = pp_um_user_get_uid(name);
    if (!name || pp_uid < 0) goto exit; // no such user

    if (pp_um_user_is_enabled(name) > 0) goto exit; // user is enabled

    if (strncmp(name, ANONYMOUS_USER_PREFIX, strlen(ANONYMOUS_USER_PREFIX)) != 0) {
	goto exit; // valid username
    }

    pp_um_user_get_password(name, &password);
    if (password && password[0] != '\0') goto exit; // valid password

    if (pp_bmc_user_may_use_sol(uid)) goto exit;

    for (channel = 0; channel <= IPMI_CHAN_MAX; channel++) {
	if (pp_bmc_user_get_priv_level(uid, channel) != IPMI_PRIV_UNSPEC) goto exit;
	if (!pp_cfg_is_default(PP_CFGKEY_IPMI_CALLBACK_ONLY, pp_uid, channel)) goto exit;
	if (!pp_cfg_is_default(PP_CFGKEY_IPMI_LINK_AUTH, pp_uid, channel)) goto exit;
	if (!pp_cfg_is_default(PP_CFGKEY_IPMI_MSG_ENABLED, pp_uid, channel)) goto exit;
    }

    pp_um_user_delete(name);
    update_user_cache();
    ret = 1;

exit:
    free(password);
    return ret;
}

/*
 * User Management interface functions
 */

int pp_bmc_user_manager_init(void)
{
    /* init and read user cache */
    init_user_cache();
    update_user_cache();
    return PP_SUC;
}

void pp_bmc_user_manager_cleanup(void)
{
    clear_user_cache();
}

int pp_bmc_user_uid_by_name(const char* name)
{
    if (name[0] == '\0') return 1; // anonymous user
    return get_uid_by_name(name);
}

int pp_bmc_user_uid_by_name_privlevel(const char* name, int channel, unsigned char privilege_level)
{
    if (name[0]) {
        // user name given: since our user names are unique, we don't have to iterate through users
        return get_uid_by_name(name);
    } else {
        // name not given: iterate users for first one that matches the role
        int uid = 0, i;
        vector_t *usrs = pp_um_get_all_users();
        assert(usrs);

        for (i = 0; i < (int)vector_size(usrs); i++) {
            char *myname = vector_get(usrs, i);
            int pp_uid = pp_um_user_get_uid(myname);

            if (PP_FAILED(pp_cfg_get_int_nodflt(&uid, PP_CFGKEY_IPMI_UID, pp_uid))) continue;
            if (pp_bmc_user_get_priv_level(uid, channel) < privilege_level) continue; // insufficent priv level

            break; // found
        }

        vector_delete(usrs);
        return uid;
    }
}

int pp_bmc_user_count_enabled(int channel)
{
    int i;
    int ret = 0;
    int pp_uid;
    
    update_user_cache();
    
    for (i = 1; i < MAX_UID; i++) {
        if ( (user_cache[i] != NULL) &&
             ((pp_uid = pp_um_user_get_uid(user_cache[i])) > 0) &&
             (pp_bmc_user_get_enable(i)) &&
             (pp_bmc_user_get_priv_level(i, channel) != IPMI_PRIV_UNSPEC) ) {
            // this is an enabled user
            ret++;
        }
    }

    return ret;
}

int pp_bmc_user_get_max_uid(void)
{
    return MAX_UID;
}

int pp_bmc_user_get_login_types(int* anonymous_user_count, int* null_user_count, int* regular_user_count)
{
    int i;

    *anonymous_user_count = 0;
    *null_user_count = 0;
    *regular_user_count = 0;
    update_user_cache();

    for (i = 1; i < MAX_UID; i++) {
	int pp_uid;
	if ( (user_cache[i] != NULL) &&
	     ((pp_uid = pp_um_user_get_uid(user_cache[i])) >= 0) &&
	     (pp_bmc_user_get_enable(i)) ) {
	    if (strncmp(user_cache[i], ANONYMOUS_USER_PREFIX, strlen(ANONYMOUS_USER_PREFIX))) {
		(*regular_user_count)++;
	    } else {
		char *pw = NULL;
		pp_um_user_get_password(user_cache[i], &pw);
		if (pw && pw[0] != '\0') {
		    (*null_user_count)++;
		} else {
		    (*anonymous_user_count)++;
		}
		free(pw);
	    }
	}
    }

    return PP_SUC;
}

int pp_bmc_user_get_name(int uid, char* name, unsigned int buflen)
{
    const char *myname;

    if ((uid < 1) || (uid > MAX_UID)) return PP_ERR;
    assert (name && buflen);
    
    myname = get_name_by_uid(uid);
    if (!myname) myname = ""; // unused users have no name
    if (strncmp(myname, ANONYMOUS_USER_PREFIX, strlen(ANONYMOUS_USER_PREFIX)) == 0) {
	// anonymous user
	myname = "";
    }
    
    if (buflen < strlen(myname) + 1) return PP_ERR;
    
    strncpy(name, myname, buflen);
    return PP_SUC;
}

int pp_bmc_user_set_name(int uid, const char* name)
{
    const char *myname;

    if ((uid <= 1) || (uid > MAX_UID)) return PP_ERR;
    assert(name);

    if (name[0] == '\0') {
        // create dummy name for anonymous user
        char dummy[20];
        snprintf(dummy, sizeof(dummy), ANONYMOUS_USER_PREFIX "%d", uid);
        name = dummy;
    }
    
    myname = get_name_by_uid(uid);
    if (!myname) {
        if (PP_FAILED(create_new_user(name, uid, NULL))) {
            pp_bmc_log_error("[UserMan] failed to create user '%s' by setting name", name);
            return PP_ERR;
        } else {
            pp_bmc_log_debug("[UserMan] user '%s' created by setting name", name);
        }
    }
    else if(strcmp(myname, name))
    {
        // rename user
        if ( PP_ERR == pp_um_user_rename(myname, name)
             || PP_ERR == pp_cfg_save(DO_FLUSH) )
        {
            pp_bmc_log_error("[UserMan] failed to rename user '%s' to '%s'", myname, name);
            return PP_ERR;
        }
    }
    else
    {
        // nothing to do (the name is already correct)
        return PP_SUC;
    }

    if (PP_ERR == pp_cfg_save(DO_FLUSH)) {
        pp_bmc_log_error("[UserMan] failed to flush configuration");
        return PP_ERR;
    }
    delete_user_if_unused(uid);
    update_user_cache();
    
    return PP_SUC;
}

int pp_bmc_user_get_enable(int uid)
{
    const char *name = get_name_by_uid(uid);
    if (!name) return 0;
    return pp_um_user_is_enabled(name);
}

int pp_bmc_user_set_enable(int uid, int enable)
{
    const char *myname;
    if ((uid < 1) || (uid > MAX_UID)) return PP_ERR;

    myname = get_name_by_uid(uid);
    if (!myname) {
	if (!enable) return PP_SUC; // no need to disable nonexisting user
	myname = create_user_if_absent(uid);
	if (!myname) return PP_ERR;
    }

    if ( PP_ERR == pp_um_user_set_enabled(myname, enable)
      || PP_ERR == pp_cfg_save(DO_FLUSH) ) {
	return PP_ERR;
    }
    delete_user_if_unused(uid);
    return PP_SUC;
}

int pp_bmc_user_get_password(int uid, char* password, unsigned int buflen)
{
    char *pw = NULL;
    const char *myname;

    if ((uid < 1) || (uid > MAX_UID)) return PP_ERR;
    assert(password && buflen);

    myname = get_name_by_uid(uid);
    if (!myname) {
        password[0] = '\0';
        goto exit;
    }

    if (PP_FAILED( pp_um_user_get_password(myname, &pw) ) || !pw) {
        password[0] = '\0';
        goto exit;
    }

    if (buflen < strlen(pw) + 1) goto exit;
    strcpy(password, pw);

exit:
    free(pw);
    return PP_SUC;
}

int pp_bmc_user_set_password(int uid, const char* password)
{
    const char *myname;

    if ((uid < 1) || (uid > MAX_UID)) return PP_ERR;
    assert(password);

    myname = create_user_if_absent(uid);
    if (!myname) return PP_ERR;

    if (PP_ERR == pp_um_user_set_password(myname, password,
                                          PP_UM_AUTH_NO_FLAGS)) {
        return PP_ERR;
    }
    delete_user_if_unused(uid);
    return PP_SUC;
}

int pp_bmc_user_get_pp_uid(int uid)
{
    const char *name = get_name_by_uid(uid);
    return name ? pp_um_user_get_uid(name) : -1;
}

int pp_bmc_user_get_priv_level(int uid, int channel UNUSED)
{
    int priv_level;
    const char *myname;

    if ((uid < 1) || (uid > MAX_UID)) return IPMI_PRIV_UNSPEC;

    myname = get_name_by_uid(uid);
    if (!myname) return IPMI_PRIV_UNSPEC;

#if !defined(PP_FEAT_BMC_OEMCMDS_ONLY)
    char *priv = NULL;
    if (channel == IPMI_CHAN_LAN) {
	/* the LAN channel privilege level is stored separately */
	pp_um_user_get_permission(myname, PP_CFGKEY_IPMI_PRIV_LEVEL_ACL, &priv);
    } else {
	int pp_uid = pp_um_user_get_uid(myname);
	if (pp_uid < 0) return IPMI_PRIV_UNSPEC;
	pp_cfg_get_nodflt(&priv, PP_CFGKEY_IPMI_PRIV_LEVEL, pp_uid, channel);
    }
    priv_level = pp_bmc_get_privlevel_value(priv);
    if (priv) free(priv);
#else /* PP_FEAT_BMC_OEMCMDS_ONLY */
    priv_level = PP_SUCCED(pp_um_user_has_permission(myname,
        PP_CFGKEY_CFGTOOL_ACL, pp_acl_raasip_yes_str)) ?
        IPMI_PRIV_ADMIN : IPMI_PRIV_UNSPEC;
#endif /* PP_FEAT_BMC_OEMCMDS_ONLY */

    return priv_level;
}

int pp_bmc_user_set_priv_level(int uid, int channel, int priv_level)
{
    const char *myname;
    int pp_uid;
    int ret;

    if ((uid < 1) || (uid > MAX_UID)) return PP_ERR;

    if (pp_bmc_user_get_priv_level(uid, channel) == priv_level) {
        /* nothing to do */
        return PP_SUC;
    }
    
    myname = create_user_if_absent(uid);
    if (!myname) return PP_ERR;
    pp_uid = pp_um_user_get_uid(myname);
    if (pp_uid < 0) return PP_ERR;

#if !defined(PP_FEAT_BMC_OEMCMDS_ONLY)
    if (channel == IPMI_CHAN_LAN) {
	/* the LAN channel privilege level is stored separately */
	ret = pp_um_ipmi_user_set_permission(myname,
		PP_CFGKEY_IPMI_PRIV_LEVEL_ACL,
		pp_bmc_get_privlevel_string(priv_level));
    } else {
	ret = pp_cfg_set(pp_bmc_get_privlevel_string(priv_level),
		PP_CFGKEY_IPMI_PRIV_LEVEL, pp_uid, channel);
    }
#else /* PP_FEAT_BMC_OEMCMDS_ONLY */
    ret = pp_um_user_set_permission(myname, PP_CFGKEY_CFGTOOL_ACL,
        priv_level == IPMI_PRIV_ADMIN || priv_level == IPMI_PRIV_OEM ?
        pp_acl_raasip_yes_str : pp_acl_raasip_no_str);
#endif /* PP_FEAT_BMC_OEMCMDS_ONLY */
    if (PP_FAILED(ret) || PP_FAILED(pp_cfg_save(DO_FLUSH))) return PP_ERR;
    delete_user_if_unused(uid);
    return PP_SUC;
}

int pp_bmc_user_callback_only(int uid, int channel)
{
    const char *myname;
    int callback_only = 0;
    int pp_uid;

    if ((uid < 1) || (uid > MAX_UID)) return 0;

    myname = get_name_by_uid(uid);
    if (!myname) return 0;
    pp_uid = pp_um_user_get_uid(myname);
    if (pp_uid < 0) return 0;

    if (PP_ERR == pp_cfg_is_enabled(&callback_only,
		PP_CFGKEY_IPMI_CALLBACK_ONLY, pp_uid, channel)) {
	return 0;
    }
    return callback_only;
}

int pp_bmc_user_link_auth(int uid, int channel)
{
    const char *myname;
    int link_auth = 0;
    int pp_uid;

    if ((uid < 1) || (uid > MAX_UID)) return 0;

    myname = get_name_by_uid(uid);
    if (!myname) return 0;
    pp_uid = pp_um_user_get_uid(myname);
    if (pp_uid < 0) return 0;

    if (PP_ERR == pp_cfg_is_enabled(&link_auth,
		PP_CFGKEY_IPMI_LINK_AUTH, pp_uid, channel)) {
	return 0;
    }
    return link_auth;
}

int pp_bmc_user_msg_enabled(int uid, int channel)
{
    const char *myname;
    int msg_enabled = 0;
    int pp_uid;

    if ((uid < 1) || (uid > MAX_UID)) return 0;

    myname = get_name_by_uid(uid);
    if (!myname) return 0;
    pp_uid = pp_um_user_get_uid(myname);
    if (pp_uid < 0) return 0;

    if (PP_ERR == pp_cfg_is_enabled(&msg_enabled,
		PP_CFGKEY_IPMI_MSG_ENABLED, pp_uid, channel)) {
	return 0;
    }
    return msg_enabled;
}

int pp_bmc_user_set_priv_level_and_flags(int uid, int channel,
	int priv_level, int callback_only, int link_auth, int msg_enabled)
{
    const char *myname;
    int pp_uid;

    if ((uid < 1) || (uid > MAX_UID)) return PP_ERR;

    myname = create_user_if_absent(uid);
    if (!myname) return PP_ERR;
    pp_uid = pp_um_user_get_uid(myname);
    if (pp_uid < 0) return PP_ERR;

#if !defined(PP_FEAT_BMC_OEMCMDS_ONLY)
    if (channel == IPMI_CHAN_LAN) {
	/* the LAN channel privilege level is stored separately */
	if (PP_FAILED( pp_um_ipmi_user_set_permission(myname,
			PP_CFGKEY_IPMI_PRIV_LEVEL_ACL,
			pp_bmc_get_privlevel_string(priv_level)) )) return PP_ERR;
    } else {
	if (PP_FAILED( pp_cfg_set(pp_bmc_get_privlevel_string(priv_level),
		    PP_CFGKEY_IPMI_PRIV_LEVEL, pp_uid, channel) )) return PP_ERR;
    }
#else /* PP_FEAT_BMC_OEMCMDS_ONLY */
    if (PP_FAILED(pp_um_user_set_permission(myname, PP_CFGKEY_CFGTOOL_ACL,
        priv_level == IPMI_PRIV_ADMIN || priv_level == IPMI_PRIV_OEM ? 
        pp_acl_raasip_yes_str : pp_acl_raasip_no_str))) return PP_ERR;
#endif /* PP_FEAT_BMC_OEMCMDS_ONLY */
    if ( PP_FAILED( pp_cfg_set_enabled(callback_only,
		PP_CFGKEY_IPMI_CALLBACK_ONLY, pp_uid, channel) )
      || PP_FAILED( pp_cfg_set_enabled(link_auth,
		PP_CFGKEY_IPMI_LINK_AUTH, pp_uid, channel) )
      || PP_FAILED( pp_cfg_set_enabled(msg_enabled,
		PP_CFGKEY_IPMI_MSG_ENABLED, pp_uid, channel) )
      || PP_FAILED( pp_cfg_save(DO_FLUSH) ) ) {
	return PP_ERR;
    }

    return PP_SUC;
}

int pp_bmc_user_may_use_sol(int uid)
{
    const char *myname;

    if ((uid < 1) || (uid > MAX_UID)) return 0;

    myname = get_name_by_uid(uid);
    if (!myname) return 0;

    if (PP_ERR == pp_um_user_has_permission(myname, 
                                            PP_CFGKEY_IPMI_MAY_USE_SOL_ACL,
                                            pp_acl_raasip_yes_str))
    {
	return 0;
    }
    return 1;
}

int pp_bmc_user_set_may_use_sol(int uid, int may_use_sol)
{
    const char *myname;
    int cfg_retval;

    if ((uid < 1) || (uid > MAX_UID)) return PP_ERR;

    myname = create_user_if_absent(uid);
    if (!myname) return PP_ERR;
    
    cfg_retval = pp_um_ipmi_user_set_permission(myname,
                     PP_CFGKEY_IPMI_MAY_USE_SOL_ACL,
                     may_use_sol ? pp_acl_raasip_yes_str
                                 : pp_acl_raasip_no_str);
    if (cfg_retval == PP_ERR) {
	return PP_ERR;
    }
    if (PP_ERR == pp_cfg_save(DO_FLUSH)) return PP_ERR;
    delete_user_if_unused(uid);
    return PP_SUC;
}
