/**
 * host_board_id.h
 * 
 * This module provides functions to detect the motherboard type of the
 * host system. This information is needed to load the correct sensor
 * configuration.
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#ifndef __HOST_BOARD_ID_H__
#define __HOST_BOARD_ID_H__

#include <pp/vector.h>

#define HOST_BOARD_ID_TYPE_UNDEFINED     0
#define HOST_BOARD_ID_TYPE_OPMA		 1
#define HOST_BOARD_ID_TYPE_MSI_FIXED     2
#define HOST_BOARD_ID_TYPE_ICP           3
#define HOST_BOARD_ID_TYPE_SUPERMICRO    4
#define HOST_BOARD_ID_TYPE_RPC           5
#define HOST_BOARD_ID_TYPE_TEST          100
#define HOST_BOARD_ID_TYPE_DUMMY         101
#define HOST_BOARD_ID_TYPE_OPMA_EVAL     102

typedef struct {
    int id_type;
    int id_length;       // directly derived from id_type
    void* id_value;      // may be null
} pp_bmc_host_board_id_t;

/**
 * Get the host motherboard id. The implementation of this function is 
 * highly hardware dependent. It may rely on gpio signals, IPMI-OEM commands,
 * configuration system values or other, yet unknown techniques. Due to 
 * hardware constraints, only some of these techniques are compiled into one
 * firmware.
 * 
 * @returns the board id structure (statically allocated)
 */
pp_bmc_host_board_id_t* pp_bmc_host_board_id_get(void);
			  
/**
 * Convert the host board id into a short string that can be used to address
 * the configuration file. The string is generic but hardly human readable.
 */
unsigned char* pp_bmc_host_board_id_to_filename(pp_bmc_host_board_id_t* hbid);

/**
 * Tries to convert the host board id into a human readable string. If the
 * specific board type is not known a generic, less human readable string
 * is returned.
 */
unsigned char* pp_bmc_host_board_id_to_string(pp_bmc_host_board_id_t* hbid);

/**
 * Gets an name for a small initial topology to be used when our specific 
 * board is not yet known but we need to set some signals anyway.
 *
 * @return name for the initial topo, NULL if not applicable or not existing
 */
unsigned char* pp_bmc_host_board_id_initial_filename(void);

#ifdef PP_FEAT_OPMA_HW
# define OPMA_VERSION_SUPPORTED		0x0102
# define OPMA_MAX_BOARD_COUNT		7

# define HOST_BOARD_ID_OPMA_ACCEPTED		0
# define HOST_BOARD_ID_OPMA_VERSION_UNSUPPORTED 1
# define HOST_BOARD_ID_OPMA_BOARD_UNSUPPORTED	2
# define HOST_BOARD_ID_OPMA_ALREADY_SET		3

/**
 * Set the board id to a specific OPMA board type
 *
 * @param oem_id	PCI SIG vendor id or OPMA specific id
 * @param impl_id	sensor subsystem id
 * @param opma_version	OPMA version, 4 digits
 * 
 * @return HOST_BOARD_ID_OPMA_ACCEPTED		board id set successfully
 *         HOST_BOARD_ID_OPMA_VERSION_MISMATCH  opma version too old
 *         HOST_BOARD_ID_OPMA_NOT_SUPPORTED     board type unknown
 *	   HOST_BOARD_ID_OPMA_ALREADY_SET       board id was already configured
 */
int pp_bmc_host_board_id_set_opma(u_short oem_id, u_short impl_id,
				  u_short opma_version);

/**
 * Get currently used OPMA board type
 *
 * @param oem_id		returns PCI SIG vendor id or OPMA specific id
 * @param impl_id		returns sensor subsystem id
 * @param opma_version		returns OPMA version, 4 digits
 * 
 * @return PP_SUC	OPMA board in use
 *         PP_ERR	no OPMA board in use
 */
int pp_bmc_host_board_id_get_opma(u_short *oem_id, u_short *impl_id,
				  u_short *opma_version);


typedef struct {
    u_short oem_id;
    u_short impl_id;
} opma_board_id_t;

/**
 * Returns a list of supported OPMA board ID combinatiors
 *
 * @return vector of type opma_board_id_t with all supported boards
 *         caller has to free the vector
 */
vector_t* pp_bmc_host_board_id_get_supported_opma(void);

#endif /* PP_FEAT_OPMA_HW */

#endif /* __HOST_BOARD_ID_H__ */
