/**
 * lan_cipher.c
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * Encapsulates the RAKP encryption algorithms and their properties.
 * (authentication, integrity, confidentiality ciphers)
 */


#include "lan_cipher.h"
#include "pp/bmc/lan_cipher_suites.h"

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_sess.h>

#include <openssl/hmac.h>
#include <openssl/evp.h>
#include <openssl/md5.h>

#include <string.h>

/* order of ciphersuites is important. Suites with a higher index are preferred */
static unsigned char cipher_data_raw[PP_BMC_LAN_CIPHER_SUITE_COUNT * sizeof(pp_cipher_suite_record_t)] =
{
    0xC0, 0x00,  0x00 | IPMI20_AUTH_NONE,      0x40 | IPMI20_INTEGRITY_NONE,          0x80 | IPMI20_CONF_NONE,

    0xC0, 0x01,  0x00 | IPMI20_AUTH_HMAC_SHA1, 0x40 | IPMI20_INTEGRITY_NONE,          0x80 | IPMI20_CONF_NONE,
    0xC0, 0x02,  0x00 | IPMI20_AUTH_HMAC_SHA1, 0x40 | IPMI20_INTEGRITY_HMAC_SHA1_96,  0x80 | IPMI20_CONF_NONE,
    0xC0, 0x03,  0x00 | IPMI20_AUTH_HMAC_SHA1, 0x40 | IPMI20_INTEGRITY_HMAC_SHA1_96,  0x80 | IPMI20_CONF_AES_CBC_128,

    0xC0, 0x06,  0x00 | IPMI20_AUTH_HMAC_MD5,  0x40 | IPMI20_INTEGRITY_NONE,          0x80 | IPMI20_CONF_NONE,
    0xC0, 0x07,  0x00 | IPMI20_AUTH_HMAC_MD5,  0x40 | IPMI20_INTEGRITY_HMAC_MD5_128,  0x80 | IPMI20_CONF_NONE,
    0xC0, 0x08,  0x00 | IPMI20_AUTH_HMAC_MD5,  0x40 | IPMI20_INTEGRITY_HMAC_MD5_128,  0x80 | IPMI20_CONF_AES_CBC_128,
    0xC0, 0x0b,  0x00 | IPMI20_AUTH_HMAC_MD5,  0x40 | IPMI20_INTEGRITY_MD5_128,       0x80 | IPMI20_CONF_NONE,
    0xC0, 0x0c,  0x00 | IPMI20_AUTH_HMAC_MD5,  0x40 | IPMI20_INTEGRITY_MD5_128,       0x80 | IPMI20_CONF_AES_CBC_128
};

pp_cipher_suite_record_t* pp_bmc_lan_enabled_cipher_suites = (pp_cipher_suite_record_t*)cipher_data_raw;


/* Internal prototypes */
static int pp_aes_cbc_128_decrypt(imsg_session_t* session,
                               unsigned char* source, unsigned short source_len,
                               unsigned char* destination);
static int pp_aes_cbc_128_encrypt(imsg_session_t* session,
                               unsigned char* source, unsigned short source_len,
                               unsigned char* destination);


/**
 * Calculate the encryption keys accordingly to the authentication cipher.
 * The new key is written to dest.
 * Requires a source key/password and some data.
 * @returns PP_ERR/PP_SUC
 */
int pp_key_calc(unsigned char auth_cipher,
                unsigned char* key, int key_len,
                unsigned char* source, int source_len,
                unsigned char* dest)
{
    unsigned int ui;

    switch (auth_cipher) {
    case IPMI20_AUTH_HMAC_SHA1:
        HMAC(EVP_sha1(), key, key_len, source, source_len, dest, &ui);
        break;
    case IPMI20_AUTH_HMAC_MD5:
        HMAC(EVP_md5(), key, key_len, source, source_len, dest, &ui);
        break;
    default:
        // should not happen but who knows
        pp_bmc_log_error("[LAN] unknown authentication type");
        return PP_ERR;
        break;
    }

    //pp_bmc_printbuf("mac-result", destination, ui);
    return PP_SUC;
}

/**
 * Get the length of the hashcode of the authentication cipher for the
 * selected RAKP message. (RAKP message defined as in payload_type).
 * The authcode length may depend on the RAKP message.
 * @returns the authcode length
 * @returns a negative number if cipher is unknown
 */
int pp_authentication_length(unsigned char auth_cipher, unsigned char rakp_message) {
    switch (auth_cipher) {
    case IPMI20_AUTH_NONE:
        return 0;
    case IPMI20_AUTH_HMAC_SHA1:
        // rakp message 4 is authenticated with SHA1-96 instead of standard SHA1
        if (rakp_message == IPMI_PAYLOAD_RAKP_MESSAGE4)
            return 12;
        else
            return 20;
    case IPMI20_AUTH_HMAC_MD5:
        return 16;
    default:
        pp_bmc_log_warn("[LAN] RAKP invalid authentication cipher");
        return -1;
    }
}

/**
 * Calculate the authentication code for the provided block, the provided
 * cipher and the selected RAKP message (as defined in payload_type).
 * The authcode is written to destination.
 * Requires session->password or session->k2
 * @returns PP_ERR/PP_SUC
 */
int pp_authentication_calculate(imsg_session_t* session, unsigned char rakp_message,
                                unsigned char* source, int source_len,
                                unsigned char* destination)
{
    unsigned int ui;
    unsigned char hash[20];

    //pp_bmc_printbuf("mac-src", source, source_len);
    
    switch (session->authentication_cipher) {
    case IPMI20_AUTH_HMAC_SHA1:
        if (rakp_message == IPMI_PAYLOAD_RAKP_MESSAGE4) {
            HMAC(EVP_sha1(), session->sik, session->sik_len, source, source_len, hash, &ui);
            ui = 12;
            memcpy (destination, hash, ui);
        }
        else {
            HMAC(EVP_sha1(), session->password, session->password_len, source, source_len, destination, &ui);
        }
        break;
    case IPMI20_AUTH_HMAC_MD5:
        if (rakp_message == IPMI_PAYLOAD_RAKP_MESSAGE4) {
            HMAC(EVP_md5(), session->sik, session->sik_len, source, source_len, destination, &ui);
        }
        else
            HMAC(EVP_md5(), session->password, session->password_len, source, source_len, destination, &ui);
        break;
    default:
        // should not happen but who knows
        pp_bmc_log_error("[LAN] unknown authentication type");
        return PP_ERR;
        break;
    }

    //pp_bmc_printbuf("mac-result", destination, ui);
    return PP_SUC;
}

/**
 * Check if the provided data block hashes to the provided authcode for
 * the selected RAKP message and cipher.
 * Requires session->password or session->k2
 * @returns PP_ERR/PP_SUC
 */
int pp_authentication_check(imsg_session_t* session, unsigned char rakp_message,
                            unsigned char* source, int source_len,
                            unsigned char* authcode)
{
    unsigned char hash[20]; // max hashlen for all current ciphers
    unsigned int ui;

    switch (session->authentication_cipher) {
    case IPMI20_AUTH_HMAC_SHA1:
        if (rakp_message == IPMI_PAYLOAD_RAKP_MESSAGE4) {
            // this will not happen as rakp4 authcode is calculated by bmc, but who knows
            HMAC(EVP_sha1(), session->sik, session->sik_len, source, source_len, hash, &ui);
            ui = 12;
        } else {
            HMAC(EVP_sha1(), session->password, session->password_len, source, source_len, hash, &ui);
        }
        break;
    case IPMI20_AUTH_HMAC_MD5:
        if (rakp_message == IPMI_PAYLOAD_RAKP_MESSAGE4) {
            // this will not happen as rakp4 authcode is calculated by bmc, but who knows
            HMAC(EVP_md5(), session->sik, session->sik_len, source, source_len, hash, &ui);
        } else {
            HMAC(EVP_md5(), session->password, session->password_len, source, source_len, hash, &ui);
        }
        break;
    default:
        // should not happen but who knows
        pp_bmc_log_error("[LAN] unknown authentication type");
        return PP_ERR;;
        break;
    }
    
    if (memcmp(hash, authcode, ui) == 0)
        return PP_SUC;
    else
        return PP_ERR;
}



/**
 * Get the length of the integrity hashcode related to the session
 * integrity cipher.
 * @returns the length of the integrity hashcode
 * @returns -1 if cipher is unknown
 */
int pp_integrity_length(unsigned char integrity_cipher) {
    switch (integrity_cipher) {
    case IPMI20_INTEGRITY_NONE:
        return 0;
    case IPMI20_INTEGRITY_HMAC_SHA1_96:
        return 12;
    case IPMI20_INTEGRITY_HMAC_MD5_128:
        return 16;
    case IPMI20_INTEGRITY_MD5_128:
        return MD5_DIGEST_LENGTH;
    default:
        pp_bmc_log_warn("[LAN] RAKP invalid integrity cipher %d", integrity_cipher);
        return -1;
    }
}

/**
 * Calculate the integrity code for the provided block and the provided cipher.
 * The authcode is written to destination. Assumes that source is padded according
 * to the spec and that destination has at least pp_integrity_length() bytes.
 * Relies on the existence of session->k1
 * @returns PP_SUC/PP_ERR
 */
int pp_integrity_calculate(imsg_session_t* session,
                           unsigned char* source, int source_len,
                           unsigned char* destination)
{
    unsigned int ui;
    unsigned char hash[20];
    
    switch (session->integrity_cipher) {
    case IPMI20_INTEGRITY_HMAC_SHA1_96:
        HMAC(EVP_sha1(), session->k1, session->sik_len, source, source_len, hash, &ui);
        // sha1 is truncated from 160 to 96 for sha1_96
        memcpy(destination, hash, 12);
        ui = 12;
        break;
    case IPMI20_INTEGRITY_HMAC_MD5_128:
        HMAC(EVP_md5(), session->k1, session->sik_len, source, source_len, destination, &ui);
        break;
    case IPMI20_INTEGRITY_MD5_128: {
            MD5_CTX md5;
            MD5_Init(&md5);
            MD5_Update(&md5, session->password, session->password_len);
            MD5_Update(&md5, source, source_len);
            MD5_Update(&md5, session->password, session->password_len);
            MD5_Final(destination, &md5);
            ui = MD5_DIGEST_LENGTH;
            break;
        }
    default:
        pp_bmc_log_warn("[LAN] RAKP invalid integrity cipher %d", session->integrity_cipher);
        return -1;
    }

    return ui;
}

/**
 * Check if the provided data block hashes to the provided authcode. Assumes
 * that source is padded according to the spec and that destination has at
 * least pp_integrity_length() bytes.
 * Relies on the existence of session->k1.
 * @returns PP_SUC/PP_ERR
 */
int pp_integrity_check(imsg_session_t* session,
                       unsigned char* source, int source_len,
                       unsigned char* authcode)
{
    /* use k1 to calculate the integrity */
    unsigned int ui;
    unsigned char hash[20];  // current maximum hashlength

    switch (session->integrity_cipher) {
    case IPMI20_INTEGRITY_HMAC_SHA1_96:
        HMAC(EVP_sha1(), session->k1, session->sik_len, source, source_len, hash, &ui);
        // sha1 is truncated from 160 to 96
        ui = 12;
        break;
    case IPMI20_INTEGRITY_HMAC_MD5_128:
        HMAC(EVP_md5(), session->k1, session->sik_len, source, source_len, hash, &ui);
        break;
    case IPMI20_INTEGRITY_MD5_128: {
            MD5_CTX md5;
            MD5_Init(&md5);
            MD5_Update(&md5, session->password, session->password_len);
            MD5_Update(&md5, source, source_len);
            MD5_Update(&md5, session->password, session->password_len);
            MD5_Final(hash, &md5);
            ui = MD5_DIGEST_LENGTH;
            break;
        }
    default:
        pp_bmc_log_warn("[LAN] RAKP invalid integrity cipher %d", session->integrity_cipher);
        return -1;
    }

    if (memcmp(hash, authcode, ui) != 0)
        return -1;

    return 0;
}


/**
 * Get the length of an encrypted payload (payload_len + conf_header + conf_trailer)
 * depending on the used cipher.
 * @returns a negative number if cipher is unknown
 */
int pp_confidentiality_length(unsigned char conf_cipher, unsigned short payload_len) {
    int len;
    switch (conf_cipher) {
    case IPMI20_CONF_NONE:
        len = payload_len;
        break;
    case IPMI20_CONF_AES_CBC_128:
        // IV + cipher(payload + payload_pad(pad minimum 1 byte)) (+ cipher_final = 0)
        len = 16 + (payload_len + 1 + (15 -((payload_len)%16)));
        break;
    case IPMI20_CONF_XRC4_128:
        // not supported yet, len = iv(16) + state(4) + payload_len
        len = 0;
    case IPMI20_CONF_XRC4_40:
        // not supported yet, len = iv(16) + state(4) + payload_len
        len = 0;
    default:
        pp_bmc_log_warn("[LAN] RAKP invalid confidentiality cipher");
        return -1;
    }
    
    /* maximum UDP frame size is 1460, minus some header bytes is *
     * approxmately 1400 bytes usable payload len.                */
    if (len > 1400) {
        pp_bmc_log_warn("[LAN] RAKP maximum payload size exceeded");
        return -1;
    }
    
    return len;
}

/**
 * Encryptes the specified source and writes the ciphertext including header
 * and trailer to destination. Destination must have at lease pp_confidentiality_length()
 * bytes in size. Relies on the existence of session->k2.
 * @returns the length of the encrypted block if successfull (max unsigned short),
 *          same value as pp_confidentiality_length()
 * @returns -1 if unsuccessfull
 */
int pp_confidentiality_encrypt(imsg_session_t* session,
                               unsigned char* source, unsigned short source_len,
                               unsigned char* destination)
{
    switch (session->confidentiality_cipher) {
    case IPMI20_CONF_AES_CBC_128:
        return pp_aes_cbc_128_encrypt(session, source, source_len, destination);
    case IPMI20_CONF_XRC4_128:
    case IPMI20_CONF_XRC4_40:
        pp_bmc_log_debug("[LAN] IPMIv20 encryption algorithm not supported");
        return -1;
    default:
        pp_bmc_log_debug("[LAN] IPMIv20 encryption algorithm unknown");
    }

    return -1;
}

/**
 * Decrypts the specified source data block and writes the plaintext to destination.
 * Destination should have source_len bytes and can overlap with source.
 * Relies on the existence of session->k2.
 * @returns the length of the decrypted data if successfull
 * @returns -1 if unsuccessfull
 */
int pp_confidentiality_decrypt(imsg_session_t* session,
                               unsigned char* source, unsigned short source_len,
                               unsigned char* destination)
{
    switch (session->confidentiality_cipher) {
    case IPMI20_CONF_AES_CBC_128:
        return pp_aes_cbc_128_decrypt(session, source, source_len, destination);
    case IPMI20_CONF_XRC4_128:
    case IPMI20_CONF_XRC4_40:
        pp_bmc_log_debug("[LAN] IPMIv20 decryption algorithm not supported");
        return -1;
    default:
        pp_bmc_log_debug("[LAN] IPMIv20 decryption algorithm unknown");
    }

    return -1;
}


/**
 * Specific encryption/decryption algorithms
 */

static int pp_aes_cbc_128_encrypt(imsg_session_t* session,
                               unsigned char* source, unsigned short source_len,
                               unsigned char* destination)
{
    int pad_len;              // the number of pad bytes + pad_len byte
    unsigned int dest_len;
    unsigned int final_len;
    int i;
    int rv;
    
    unsigned char* source_padded;
    unsigned int source_padded_len;

    EVP_CIPHER_CTX ctx;
    
    if (source_len == 0) {
         pp_bmc_log_notice("[LAN] IPMIv20 AES unable to encrypt empty payloads");
         return -1;
    }

    /*
     * limit length to fit data within a singel UDP packet, MTU = 14xx
     */
    if (source_len > 1024) {
        // this constraint is copied from ipmitool. I don't know if its properly applied
        pp_bmc_log_debug("[LAN] IPMIv20 AES encryption of blocks greater than 1024 bytes not supported yet");
        return -1;
    }

    // create iv
    srand(time(NULL));
    for (i=0; i<16; i++) {
        destination[i] = rand() % 0xff;;
    }
    
    // pad source (with at least the pad_len byte)
    pad_len = 16 - (source_len % 16);
    source_padded_len = source_len + pad_len;
    source_padded = malloc(source_padded_len);
    memcpy(source_padded, source, source_len);
    for (i=0; i<pad_len-1; i++) {
	source_padded[source_len+i] = i+1;
    }
    source_padded[source_len+pad_len-1] = pad_len-1;

    // ready to encrypt
    EVP_CIPHER_CTX_init(&ctx);
    EVP_EncryptInit_ex(&ctx, EVP_aes_128_cbc(), NULL, session->k2, destination);
    EVP_CIPHER_CTX_set_padding(&ctx, 0);

    rv = EVP_EncryptUpdate(&ctx, destination+16, &dest_len, source_padded, source_padded_len);
    free(source_padded);
    
    if (!rv) {
        pp_bmc_log_debug("[LAN] IPMIv20 AES encryption failure");
        return -1;
    }

    if(!EVP_EncryptFinal_ex(&ctx, destination+16 + dest_len, &final_len)) {
        pp_bmc_log_debug("[LAN] IPMIv20 AES encryption failure");
        return -1;
    }

    EVP_CIPHER_CTX_cleanup(&ctx);

    dest_len = 16 + dest_len + final_len;

    return dest_len;
}

static int pp_aes_cbc_128_decrypt(imsg_session_t* session,
                               unsigned char* source, unsigned short source_len,
                               unsigned char* destination)
{
    unsigned int payload_len;      // cipher (plaintext + pad)
    unsigned int destination_len;  // plaintextlen in destination
    
    unsigned int final_len;
    unsigned char conf_pad_cnt;
    int i;

    EVP_CIPHER_CTX ctx;
    
    if (source_len < 32) {
        pp_bmc_log_notice("[LAN] IPMIv20 AES decryption source to short (%d bytes)", source_len);
        return -1;
    }
    
    payload_len = source_len - 16;

    if ((payload_len % 16) != 0) {
        pp_bmc_log_notice("[LAN] IPMIv20 AES decryption with wrong blocksize");
        return -1;
    }

    // ready to start decryption

    EVP_CIPHER_CTX_init(&ctx);
    EVP_DecryptInit_ex(&ctx, EVP_aes_128_cbc(), NULL, session->k2, source);
    EVP_CIPHER_CTX_set_padding(&ctx, 0);
    
    if(!EVP_DecryptUpdate(&ctx, destination, &destination_len, source+16, payload_len)) {
        pp_bmc_log_notice("[LAN] IPMIv20 AES decryption failed");
        return -1;
    }

    if(!EVP_DecryptFinal_ex(&ctx, (destination + destination_len), &final_len)) {
        pp_bmc_log_notice("[LAN] IPMIv20 AES decryption failed");
        return -1;
    }
    
    //assert(final_len == 0);
    destination_len += final_len;
    EVP_CIPHER_CTX_cleanup(&ctx);

    // remove pad bytes
    conf_pad_cnt = destination[destination_len-1];
    if (conf_pad_cnt > 15) {
        pp_bmc_log_notice("[LAN] IPMIv20 AES decryption ignored message with strange confidentiality_pad size %d", conf_pad_cnt);
        return -1;
    }
    destination_len = destination_len - (conf_pad_cnt + 1);
    for (i=0; i<conf_pad_cnt; i++) {
        if (destination[destination_len + i] != i+1) {
            pp_bmc_log_notice("[LAN] IPMIv20 AES decryption ignored message with strange confidentiality_pad");
            return -1;
        }
    }
    
    return destination_len;
}

