/**
 * lan_cipher.h
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * Encapsulates the RAKP encryption algorithms and their properties.
 * (authentication, integrity, confidentiality ciphers)
 */

#include <pp/bmc/bmc_imsg.h>

/**
 * Calculate the encryption keys accordingly to the authentication cipher.
 * The new key is written to dest.
 * Requires a source key/password and some data.
 * @returns PP_ERR/PP_SUC
 */
int pp_key_calc(unsigned char auth_cipher,
                unsigned char* key, int key_len,
                unsigned char* source, int source_len,
                unsigned char* dest);

/**
 * Get the length of the hashcode of the authentication cipher for the
 * selected RAKP message. (RAKP message defined as in payload_type).
 * The authcode length may depend on the RAKP message.
 * @returns the authcode length
 * @returns a negative number if cipher is unknown
 */
int pp_authentication_length(unsigned char auth_cipher, unsigned char rakp_message);

/**
 * Calculate the authentication code for the provided block, the provided
 * cipher and the selected RAKP message (as defined in payload_type).
 * The authcode is written to destination.
 * Requires session->password or session->k2
 * @returns PP_ERR/PP_SUC
 */
int pp_authentication_calculate(imsg_session_t* session, unsigned char rakp_message,
                                unsigned char* source, int source_len,
                                unsigned char* destination);

/**
 * Check if the provided data block hashes to the provided authcode for
 * the selected RAKP message and cipher.
 * Requires session->password or session->k2
 * @returns PP_ERR/PP_SUC
 */
int pp_authentication_check(imsg_session_t* session, unsigned char rakp_message,
                            unsigned char* source, int source_len,
                            unsigned char* authcode);


/**
 * Get the length of the integrity hashcode related to the session
 * integrity cipher.
 * @returns the length of the integrity hashcode
 * @returns -1 if cipher is unknown
 */
int pp_integrity_length(unsigned char integrity_cipher);

/**
 * Calculate the integrity code for the provided block and the provided cipher.
 * The authcode is written to destination. Assumes that source is padded according
 * to the spec and that destination has at least pp_integrity_length() bytes.
 * Relies on the existence of session->k1.
 * @returns PP_SUC/PP_ERR
 */
int pp_integrity_calculate(imsg_session_t* session,
                           unsigned char* source, int source_len,
                           unsigned char* destination);

/**
 * Check if the provided data block hashes to the provided authcode. Assumes
 * that source is padded according to the spec and that destination has at
 * least pp_integrity_length() bytes.
 * Relies on the existence of session->k1.
 * @returns PP_SUC/PP_ERR
 */
int pp_integrity_check(imsg_session_t* session,
                       unsigned char* source, int source_len,
                       unsigned char* authcode);


/**
 * Get the length of an encrypted payload (payload_len + conf_header + conf_trailer)
 * depending on the used cipher.
 * @returns a negative number if cipher is unknown
 */
int pp_confidentiality_length(unsigned char conf_cipher, unsigned short payload_len);

/**
 * Encryptes the specified source and writes the ciphertext including header
 * and trailer to destination. Destination must have at lease pp_confidentiality_length()
 * bytes in size. Relies on the existence of session->k2.
 * @returns the length of the encrypted block if successfull (max unsigned short),
 *          same value as pp_confidentiality_length()
 * @returns -1 if unsuccessfull
 */
int pp_confidentiality_encrypt(imsg_session_t* session,
                               unsigned char* source, unsigned short source_len,
                               unsigned char* destination);

/**
 * Decrypts the specified source data block and writes the plaintext to destination.
 * Destination should have source_len bytes and can overlap with source.
 * Relies on the existence of session->k2.
 * @returns the length of the decrypted data if successfull
 * @returns -1 if unsuccessfull
 */
int pp_confidentiality_decrypt(imsg_session_t* session,
                               unsigned char* source, unsigned short source_len,
                               unsigned char* destination);




