/**
 * lan_cmd.c
 *
 * commands dedicated to RMCP+ (Support and Payload)
 * 
 * (c) 2005 Peppercon AG, 2005/05/08, tbr@peppecon.de
 */
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/xdefs.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/utils.h>
#include <pp/bmc/topo_classes.h>

#include <pp/bmc/user_manager.h>
#include <pp/bmc/lan_serv.h>
#include "lan_cmds.h"
#include "lan_sol.h"
#include "lan_sol_cmds.h"
#include "lan_ipmi20.h"

/*
 * private definitions
 * ----------------------------------
 */
struct activate_payload_rq_s {
    unsigned char payload_type;
    unsigned char payload_inst;
    BITFIELD6(unsigned char,        \
	      sol_rsvd1 : 1,        \
	      sol_startup : 1,      \
	      sol_alert : 2,        \
	      sol_rsvd2 : 2,        \
	      sol_authenticate : 1, \
	      sol_encrypt : 1);
    unsigned char aux_rsvd2;
    unsigned char aux_rsvd3;
    unsigned char aux_rsvd4;
} __attribute__ ((packed));
typedef struct activate_payload_rq_s activate_payload_rq_t;

struct activate_payload_rs_s {
    unsigned char  aux[4];                    /* 4 bytes aux */
    unsigned short max_inbound_size_le16;       /* cpu_to_le16 */
    unsigned short max_outbound_size_le16;
    unsigned short port_no_le16;
    unsigned short vlan_no_le16; // 0xffff if not used
} __attribute__ ((packed));
typedef struct activate_payload_rs_s activate_payload_rs_t;

struct deactivate_payload_rq_s {
    unsigned char payload_type;
    unsigned char payload_inst;
    unsigned char aux[4];
} __attribute__ ((packed));
typedef struct deactivate_payload_rq_s deactivate_payload_rq_t;

/* deactivation response has response code only */

/* 
 * private methods 
 * -----------------
 */
static int activate_payload(imsg_t* imsg);
static inline void act_pl_init_err_rs(activate_payload_rs_t* rs);
static inline void act_pl_init_rs(activate_payload_rs_t* rs);
static int deactivate_payload(imsg_t* imsg);

/*
 * implementations
 * ----------------
 */
static int activate_payload(imsg_t* imsg) {
    activate_payload_rq_t* rq = (void*)imsg->data;
    unsigned char rs_code;
    unsigned short max_inbound_size;
    unsigned short max_outbound_size;
    activate_payload_rs_t* rs;
    int r;

    switch (rq->payload_type) {

    case IPMI_PAYLOAD_SOL: {
	char *val;
	int priv_level;
	
        pp_cfg_get(&val, bmc_lan_sol_min_priv_level_str);
        priv_level = pp_bmc_get_privlevel_value(val);
        if (priv_level < IPMI_PRIV_USER) {
	    pp_bmc_log_debug("[LAN] activate_payload: reading minimum privilege level <%s> from configuration. Ignoring and using \"USER\" instead.", val);
	    priv_level =  IPMI_PRIV_USER;
	}
	free(val);

	// msg has to come from lanplus channel
	if (imsg->chan != IPMI_CHAN_LAN ||
	    ((lan_addr_t*)imsg->buf)->ipmi_version != 0x02) {
	    rs_code = IPMI_ERR_PAYLOAD_ACT_DISABLED;
	    r = PP_ERR;
	} else if (imsg->priv_level  < priv_level) {
	    rs_code = IPMI_ERR_INSUFFICIENT_PRIVILEGE_LEVEL;
	    r = PP_ERR;
	} else if (!pp_bmc_user_may_use_sol(imsg->session->uid) ) {
	    rs_code = IPMI_ERR_PAYLOAD_ACT_DISABLED;
	    r = PP_ERR;
	} else if (rq->payload_inst == 1) { // only single instance supported
	    
	    // TODO: rq->sol_startup ignored, currently
	    // TODO: rq->sol_alert ignored, currently
	    // TODO: rq->sol_authenticate ignored, currently
	    // TODO: rq->sol_encrypt ignored, currently

	    // TODO: how to check that msg really comes from lan-channel
	    
	    r = pp_bmc_sol_activate(&rs_code, &max_inbound_size,
				    &max_outbound_size, imsg->session);
	} else {
	    rs_code = IPMI_ERR_PAYLOAD_ACT_LIMIT_REACHED;
	    r = PP_ERR;
	}
	rs = pp_bmc_imsg_resp(imsg, rs_code, sizeof(activate_payload_rs_t));
	if (r == PP_SUC) {
	    unsigned short port;
	    if (PP_FAILED(pp_cfg_get_short_nodflt(&port, "bmc.lan.port"))) {
	        port = IPMI_LAN_PORT;
	    }

	    act_pl_init_rs(rs);
	    rs->max_inbound_size_le16 = cpu_to_le16(max_inbound_size);
	    rs->max_outbound_size_le16 = cpu_to_le16(max_outbound_size);
	    rs->port_no_le16 = cpu_to_le16(port);
	} else {
	    act_pl_init_err_rs(rs);
	} }
	break; 

    default:
	if (rq->payload_type > IPMI_PAYLOAD_OEM) {
	    rs_code = IPMI_ERR_PARAM_OUT_OF_RANGE;
	} else {
	    rs_code = IPMI_ERR_PAYLOAD_ACT_DISABLED;
	}
	rs = pp_bmc_imsg_resp(imsg, rs_code, sizeof(activate_payload_rs_t));
	act_pl_init_err_rs(rs);
    }
    return pp_bmc_router_send_msg(imsg);
}

static inline void act_pl_init_err_rs(activate_payload_rs_t* rs) {
    act_pl_init_rs(rs);
    rs->max_inbound_size_le16 = 0;
    rs->max_outbound_size_le16 = 0;
    rs->port_no_le16 = 0;
}

static inline void act_pl_init_rs(activate_payload_rs_t* rs) {
    rs->aux[0] = rs->aux[1] = rs->aux[2] = rs->aux[3] = 0;
    rs->vlan_no_le16 = 0xffff; /* as per spec 24.1 */
}

static int deactivate_payload(imsg_t* imsg) {
    deactivate_payload_rq_t* rq = (void*)imsg->data;
    unsigned char rs_code;

    switch (rq->payload_type) {

    case IPMI_PAYLOAD_SOL: 
	// msg has to come from lanplus channel
	// According to the IPMI spec deactivate_payload 
	// should also work from other interfaces (with privilege level ADMIN)
	if (imsg->chan != IPMI_CHAN_LAN ||
	    ((lan_addr_t*)imsg->buf)->ipmi_version != 0x02) {
	    rs_code = IPMI_ERR_PAYLOAD_DEACT_DISABLED;
	} else if (  (imsg->priv_level < IPMI_PRIV_ADMIN) &&
		     ((imsg->session->sol_handler == NULL) || 
		      (imsg->session->sol_deactivate == NULL)) ) {
	    // The client cannot deactivate the payload if he hasn't
	    // got admin level and is not the owner of the payload
	    rs_code = IPMI_ERR_INSUFFICIENT_PRIVILEGE_LEVEL;
	} else if (rq->payload_inst == 1) { // only single instance supported
	    pp_bmc_sol_deactivate(&rs_code, imsg->session);
	} else {
	    rs_code = IPMI_ERR_PAYLOAD_DEACT_NOT_ACTIVE;
	}
	break;

    default:
	if (rq->payload_type > IPMI_PAYLOAD_OEM) {
	    rs_code = IPMI_ERR_PARAM_OUT_OF_RANGE;
	} else {
	    rs_code = IPMI_ERR_PAYLOAD_DEACT_NOT_ACTIVE;
	}
    }

    pp_bmc_imsg_resp(imsg, rs_code, 0);
    return pp_bmc_router_send_msg(imsg);
}

struct get_payload_activation_status_rq_s {
    unsigned char payload_type;
} __attribute__ ((packed));
typedef struct get_payload_activation_status_rq_s
               get_payload_activation_status_rq_t;

struct get_payload_activation_status_rs_s {
    BITFIELD2(unsigned char,              \
	      instance_capacity : 4,      \
	      reserved : 4);
    BITFIELD8(unsigned char,              \
	      activation_instance_1 : 1,  \
	      activation_instance_2 : 1,  \
	      activation_instance_3 : 1,  \
	      activation_instance_4 : 1,  \
	      activation_instance_5 : 1,  \
	      activation_instance_6 : 1,  \
	      activation_instance_7 : 1,  \
	      activation_instance_8 : 1);
    BITFIELD8(unsigned char,              \
	      activation_instance_9  : 1, \
	      activation_instance_10 : 1, \
	      activation_instance_11 : 1, \
	      activation_instance_12 : 1, \
	      activation_instance_13 : 1, \
	      activation_instance_14 : 1, \
	      activation_instance_15 : 1, \
	      activation_instance_16 : 1);
} __attribute__ ((packed));
typedef struct get_payload_activation_status_rs_s
               get_payload_activation_status_rs_t;

static int get_payload_activation_status(imsg_t *imsg) {
    get_payload_activation_status_rq_t *rq = (void*)imsg->data;
    get_payload_activation_status_rs_t *rs;
    unsigned char payload_type = rq->payload_type;

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 
			  sizeof(get_payload_activation_status_rs_t));
    switch (payload_type) {
    case IPMI_PAYLOAD_IPMI:           // Fallthrough
    case IPMI_PAYLOAD_RAKP_OPEN_SESSION_REQUEST:  // Fallthrough
    case IPMI_PAYLOAD_RAKP_OPEN_SESSION_RESPONSE: // Fallthrough
    case IPMI_PAYLOAD_RAKP_MESSAGE1:  // Fallthrough
    case IPMI_PAYLOAD_RAKP_MESSAGE2:  // Fallthrough
    case IPMI_PAYLOAD_RAKP_MESSAGE3:  // Fallthrough
    case IPMI_PAYLOAD_RAKP_MESSAGE4:
	// We say "we support 15 instances and none is active"
	rs->instance_capacity = 0x0F;
	break;
    case IPMI_PAYLOAD_SOL:
	rs->instance_capacity = 1;
	if (pp_bmc_sol_state_is_active()) {
	    rs->activation_instance_1 = 1;
	}
	break;
    case IPMI_PAYLOAD_OEM: // Fallthrough
    default:
	pp_bmc_imsg_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    return pp_bmc_router_send_msg(imsg);
}

struct get_payload_instance_info_rq_s {
    unsigned char payload_type;
    unsigned char payload_instance;
} __attribute__ ((packed));
typedef struct get_payload_instance_info_rq_s get_payload_instance_info_rq_t;

struct get_payload_instance_info_rs_s {
    unsigned int session_id_le32;
    unsigned char sol_port_nr_le16;
    unsigned char reserved[7];
} __attribute__ ((packed));
typedef struct get_payload_instance_info_rs_s get_payload_instance_info_rs_t;

static int get_payload_instance_info(imsg_t *imsg) {
    get_payload_instance_info_rq_t * rq = (void*)imsg->data;
    get_payload_instance_info_rs_t * rs;
    unsigned char payload_type = rq->payload_type;
    unsigned char payload_instance = rq->payload_instance;

    if (payload_type != IPMI_PAYLOAD_SOL) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_payload_instance_info_rs_t));
    
    if (payload_instance == 1) {
	if (pp_bmc_sol_state_is_active()) {
	    rs->session_id_le32 = cpu_to_le32(pp_bmc_sol_get_session_id());
	    rs->sol_port_nr_le16 = cpu_to_le16(pp_bmc_sol_get_remote_port());
	}
	else {
	    pp_bmc_log_debug("[LAN] get_payload_instance_info on instance 1, but no SOL active");
	    pp_bmc_imsg_err(imsg, IPMI_ERR_UNSPECIFIED);
	}
    }
    return pp_bmc_router_send_msg(imsg);
}

struct set_user_payload_access_rq_s {
    BITFIELD2(unsigned char,   \
	      channel : 4,     \
	      reserved : 4);
    BITFIELD2(unsigned char,   \
	      user_id   : 6,   \
	      operation : 2);
    BITFIELD3(unsigned char,   \
	      reserved2   : 1, \
	      payload_sol : 1, \
	      reserved_standard_payloads : 6);
    unsigned char reserved_standard_payload_enables_2;
    unsigned char oem_payload_enables_1;
    unsigned char reserved_oem_payload_enables_2;;
} __attribute__ ((packed));
typedef struct set_user_payload_access_rq_s set_user_payload_access_rq_t;

static int set_user_payload_access(imsg_t *imsg) {
    set_user_payload_access_rq_t * rq = (void*)imsg->data;
    unsigned char channel = rq->channel;
    
    if (rq->user_id == 0) { //reserved
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    if (channel == IPMI_CHAN_PRESENT) {
	channel = imsg->chan;
    }
    if (channel != IPMI_CHAN_LAN) {
	pp_bmc_log_debug("[LAN] set_user_payload_access on non-lan channel requested");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    if(rq->oem_payload_enables_1 != 0) {
	pp_bmc_log_debug("[LAN] set_user_payload_access for OEM payload requested. We don't have OEM payloads.");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }

    if (rq->payload_sol) {
	if ((rq->operation != 0) && (rq->operation != 1) ) {
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
	}
	if (PP_ERR == pp_bmc_user_set_may_use_sol(rq->user_id, 
						  1 - rq->operation)) {
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
	}
    } // else { this command does nothing at all }
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

struct get_user_payload_access_rq_s {
    BITFIELD2(unsigned char,   \
	      channel   : 4,   \
	      reserved1 : 4);
    BITFIELD2(unsigned char,   \
	      user_id   : 6,   \
	      reserved2 : 2);
} __attribute__ ((packed));
typedef struct get_user_payload_access_rq_s get_user_payload_access_rq_t;

struct get_user_payload_access_rs_s {
    BITFIELD3(unsigned char,   \
	      reserved1   : 1, \
	      payload_sol : 1, \
	      reserved_payloads_2_to_7 : 6);
    unsigned char reserved_standard_payload_enables_2;
    unsigned char oem_1;
    unsigned char oem_2;
} __attribute__ ((packed));
typedef struct get_user_payload_access_rs_s get_user_payload_access_rs_t;

static int get_user_payload_access(imsg_t *imsg) {
    get_user_payload_access_rq_t *rq = (void*)imsg->data;
    unsigned char user_id = rq->user_id;
    unsigned char channel = rq->channel;
    get_user_payload_access_rs_t *rs;

    if (channel == IPMI_CHAN_PRESENT) {
	channel = imsg->chan;
    }
    if (channel != IPMI_CHAN_LAN) {
	pp_bmc_log_debug("[LAN] get_user_payload_access on non-lan channel requested");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    if (user_id == 0) { //reserved
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_user_payload_access_rs_t));

    rs->payload_sol = pp_bmc_user_may_use_sol(user_id);
    return pp_bmc_router_send_msg(imsg);
}

struct get_channel_payload_support_rq_s {
    BITFIELD2(unsigned char,  \
	      channel   : 4,  \
	      reserved1 : 4);
} __attribute__ ((packed));
typedef struct get_channel_payload_support_rq_s
               get_channel_payload_support_rq_t;

struct get_channel_payload_support_rs_s{
    unsigned char standard_payload_type_nr_lower;
    unsigned char standard_payload_type_nr_upper;
    unsigned char sess_setup_payload_type_nr_lower;
    unsigned char sess_setup_payload_type_nr_upper;
    unsigned char payload_type_oem_lower;
    unsigned char payload_type_oem_upper;
    unsigned char reserved1;
    unsigned char reserved2;
} __attribute__ ((packed));
typedef struct get_channel_payload_support_rs_s
               get_channel_payload_support_rs_t;

static int get_channel_payload_support(imsg_t *imsg) {
    get_channel_payload_support_rq_t * rq = (void*)imsg->data;
    unsigned char channel = rq->channel;
    get_channel_payload_support_rs_t * rs;

    if (channel == IPMI_CHAN_PRESENT) {
	channel = imsg->chan;
    }
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_channel_payload_support_rs_t));

    switch (channel) {
    case IPMI_CHAN_LAN:
	rs->standard_payload_type_nr_lower = 0x02;
	// Fallthrough
    case IPMI_CHAN_SERIAL:
    case IPMI_CHAN_SI:
    case IPMI_CHAN_LOOP_ERIC:
    case IPMI_CHAN_LOOP_SNMPD:
	rs->standard_payload_type_nr_lower |= 0x01;
    }
    return pp_bmc_router_send_msg(imsg);
}

struct get_channel_payload_version_rq_s {
    BITFIELD2(unsigned char,  \
	      channel   : 4,  \
	      reserved1 : 4);
    unsigned char payload_type_nr;
} __attribute__ ((packed));
typedef struct get_channel_payload_version_rq_s
               get_channel_payload_version_rq_t;

struct get_channel_payload_version_rs_s {
    unsigned char version;
}  __attribute__ ((packed));
typedef struct get_channel_payload_version_rs_s
               get_channel_payload_version_rs_t;

static int get_channel_payload_version(imsg_t *imsg) {
    get_channel_payload_version_rq_t * rq = (void *)imsg->data;
    get_channel_payload_version_rs_t * rs;
    unsigned char channel = rq->channel;
    int payload_available = 0;
    unsigned char version = 0;

    if (channel == IPMI_CHAN_PRESENT) {
	channel = imsg->chan;
    }
    switch (channel) {
    case IPMI_CHAN_SERIAL:
    case IPMI_CHAN_SI:
    case IPMI_CHAN_LOOP_ERIC:
    case IPMI_CHAN_LOOP_SNMPD:
	if (rq->payload_type_nr == IPMI_PAYLOAD_IPMI) {
	    payload_available = 1;
	    version = 0x02; // though it looks like this command is
			    // only meant for SOL payloads
	}
	break;
    case IPMI_CHAN_LAN:
	switch (rq->payload_type_nr) {
	case 0: // IPMI
	case 1: // SOL
	case 10: // RMCP+ request
	case 11: // RMCP+ response
	case 12: // RAKP 1
	case 13: // RAKP 2
	case 14: // RAKP 3
	case 15: // RAKP 4
	    payload_available = 1;
	    version = 0x10;
	}
    }
    if (!payload_available) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PAYLOAD_TYPE_NOT_AVAILABLE);
    }
    
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_channel_payload_version_rs_t));
    rs->version = version;
    return pp_bmc_router_send_msg(imsg);
}

#define IPMI_ERR_KEY_IS_LOCKED  0x80
#define IPMI_ERR_KEY_TOO_SHORT  0x81
#define IPMI_ERR_KEY_TOO_LONG   0x82
#define IPMI_ERR_KEY_INVALID    0x83
#define IPMI_ERR_KR_IS_NOT_USED 0x84

struct set_channel_security_keys_rq_s {
    BITFIELD2(unsigned char, channel : 4,  : 4);
    BITFIELD2(unsigned char, operation : 2,  : 6);
    unsigned char key_id;
    const char key_data[0];
} __attribute__ ((packed));
typedef struct set_channel_security_keys_rq_s set_channel_security_keys_rq_t;

struct set_channel_security_keys_rs_s {
    BITFIELD2(unsigned char, lock_status : 2, : 6);
    char key_data[0];
} __attribute__ ((packed));
typedef struct set_channel_security_keys_rs_s set_channel_security_keys_rs_t;

static int set_channel_security_keys(imsg_t *imsg)
{
    set_channel_security_keys_rq_t *rq = (void *)imsg->data;
    set_channel_security_keys_rs_t *rs;
    unsigned char channel = rq->channel;
    size_t key_size, i;
    char *key_data;

    if (channel == IPMI_CHAN_PRESENT) {
	channel = imsg->chan;
    }
    if (channel != IPMI_CHAN_LAN) {
	pp_bmc_log_debug("[LAN] set_channel_security_keys on non-lan channel requested");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }

    if (rq->key_id == 0x00) {
	/* K_R is not required for our random generator */
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_KR_IS_NOT_USED);
    } else if (rq->key_id != 0x01) {
	/* neither K_G nor K_R; unknown key_id */
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }

    switch (rq->operation) {
	case 0: // read key
	    key_data = NULL;
	    if (PP_SUCCED( pp_cfg_get_binary(&key_data, &key_size, PP_BMC_CHANNEL_ACCESS_KEYGEN_KEY, channel) )) {
		if (key_size != 20) {
		    free(key_data);
		    return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
		}
	    }
	    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(set_channel_security_keys_rs_t) + 20);
	    rs->lock_status = 0x00; // key is not lockable
	    if (key_data) {
		memcpy(rs->key_data, key_data, key_size);
		free(key_data);
	    }
	    return pp_bmc_router_send_msg(imsg);
	case 1: // set key
	    key_size = imsg->data_size - sizeof(set_channel_security_keys_rq_t);
	    if (key_size < 20) {
		return pp_bmc_router_resp_err(imsg, IPMI_ERR_KEY_TOO_SHORT);
	    } else if (key_size > 20) {
		return pp_bmc_router_resp_err(imsg, IPMI_ERR_KEY_TOO_LONG);
	    }
	    for (i = 0; i < key_size; i++) {
		if (rq->key_data[i]) break;
	    }
	    if (i < key_size) {
		pp_cfg_set_binary(rq->key_data, key_size, PP_BMC_CHANNEL_ACCESS_KEYGEN_KEY, channel);
	    } else {
		// all-zero key; delete from config
		pp_cfg_remove(PP_BMC_CHANNEL_ACCESS_KEYGEN_KEY, channel);
	    }
	    pp_cfg_save(DO_FLUSH);
	    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(set_channel_security_keys_rs_t));
	    rs->lock_status = 0x00; // key is not lockable
	    return pp_bmc_router_send_msg(imsg);
	case 2: // lock key
	    /* K_R is not supported, K_G cannot be locked => fail */
	default:
	    ;
    }

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
}

/*
 * lan channel cmds
 * ------------------  
 */
static const dev_cmd_entry_t lan_cmd_tab[] = {
    { netfn:          IPMI_NETFN_APP,
      cmd:            IPMI_CMD_ACTIVATE_PAYLOAD,
      cmd_hndlr:      activate_payload,
      min_data_size:  sizeof(activate_payload_rq_t),
      min_priv_level: IPMI_PRIV_CALLBACK,
    },
    { netfn:          IPMI_NETFN_APP,
      cmd:            IPMI_CMD_DEACTIVATE_PAYLOAD,
      cmd_hndlr:      deactivate_payload,
      min_data_size:  sizeof(deactivate_payload_rq_t),
      min_priv_level: IPMI_PRIV_CALLBACK,
    },
    { netfn:          IPMI_NETFN_APP,
      cmd:            IPMI_CMD_GET_PAYLOAD_ACTIVATION_STATUS,
      cmd_hndlr:      get_payload_activation_status,
      min_data_size:  sizeof(get_payload_activation_status_rq_t),
      min_priv_level: IPMI_PRIV_USER,
    },
    { netfn:          IPMI_NETFN_APP,
      cmd:            IPMI_CMD_GET_PAYLOAD_INSTANCE_INFO,
      cmd_hndlr:      get_payload_instance_info,
      min_data_size:  sizeof(get_payload_instance_info_rq_t),
      min_priv_level: IPMI_PRIV_USER,
    },
    { netfn:          IPMI_NETFN_APP,
      cmd:            IPMI_CMD_SET_USER_PAYLOAD_ACCESS,
      cmd_hndlr:      set_user_payload_access,
      min_data_size:  sizeof(set_user_payload_access_rq_t),
      min_priv_level: IPMI_PRIV_ADMIN,
    },
    { netfn:          IPMI_NETFN_APP,
      cmd:            IPMI_CMD_GET_USER_PAYLOAD_ACCESS,
      cmd_hndlr:      get_user_payload_access,
      min_data_size:  sizeof(get_user_payload_access_rq_t),
      min_priv_level: IPMI_PRIV_OPERATOR,
    },
    { netfn:          IPMI_NETFN_APP,
      cmd:            IPMI_CMD_GET_CHANNEL_PAYLOAD_SUPPORT,
      cmd_hndlr:      get_channel_payload_support,
      min_data_size:  sizeof(get_channel_payload_support_rq_t),
      min_priv_level: IPMI_PRIV_USER,
    },
    { netfn:          IPMI_NETFN_APP,
      cmd:            IPMI_CMD_GET_CHANNEL_PAYLOAD_VERSION,
      cmd_hndlr:      get_channel_payload_version,
      min_data_size:  sizeof(get_channel_payload_version_rq_t),
      min_priv_level: IPMI_PRIV_USER,
    },
    { netfn:          IPMI_NETFN_APP,
      cmd:            IPMI_CMD_SET_CHANNEL_SECURITY_KEYS,
      cmd_hndlr:      set_channel_security_keys,
      min_data_size:  sizeof(set_channel_security_keys_rq_t),
      min_priv_level: IPMI_PRIV_USER,
    },
    { cmd_hndlr:      NULL,
    }
};

/* c'tor / d'tor */
int pp_lan_cmd_init() {
    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(lan_cmd_tab))
	return PP_ERR;
    return PP_SUC;
}

void pp_lan_cmd_cleanup() {
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(lan_cmd_tab);
}
