/**
 * lan_ipmi15.c
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * Handles the IPMIv15 messages for the ipmi class of RMCP
 */


#include "lan_ipmi15.h"
#include "rmcp.h"
#include <pp/bmc/lan_serv.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/session_manager.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/utils.h>

#include <pp/base.h>
#include <openssl/md5.h>
#include <openssl/md2.h>

#include "lan_serv_intern.h"


/**
 * Handle rmcp v1.5 session headers.
 */
int pp_ipmi15_receive(unsigned char* data, int len, lan_addr_t* addr) {
    unsigned char authtype;
    unsigned int session_seq_no;
    unsigned int session_id;
    int data_off;
    // blocks within *data
    unsigned char* authcode;       // length = 16
    unsigned char* ipmi_msg;       // start of ipmi message, length = ipmi_msg_len
    unsigned char ipmi_msg_len;    // length of the ipmi message

    imsg_session_t* session;
    static unsigned char hash[16]; // our version of the hash 
    int allowed_anyway;
    int userlevel_only;
    int wrap_add;
    
    int rv;

    //pp_bmc_log_debug("[LAN] IPMIv15 receiving message");

    // determine session header length, extract session header data
    // rmcp.c already checked that we have the authtype - no need to do that again
    authtype = data[0] & 0x0F;
    if (authtype == IPMI_AUTH_NONE) {
        // auth none
        if (len < 10) {
            pp_bmc_log_notice("[LAN] IPMIv15 message to short - ignored");
            return PP_ERR;
        }
        ipmi_msg_len = data[9];
        authcode = NULL;
        data_off = 10;
    } else {
        // 16 bytes more (authcode)
        if (len < 26) {
            pp_bmc_log_notice("[LAN] IPMIv15 message to short - ignored");
            return PP_ERR;
        }
        ipmi_msg_len = data[25];
        authcode = data + 9;
        data_off = 26;
    }
    if (len < (data_off + ipmi_msg_len)) {
        pp_bmc_log_notice("[LAN] IPMIv15 message to short - ignored");
        return PP_ERR;
    }

    session_seq_no = le32_to_cpu(*((unsigned int*)(data+1)));
    session_id = le32_to_cpu(*((unsigned int*)(data+5)));
    ipmi_msg = data + data_off;

    // remaining length should be 0 or 1 (legacy pad, can be ignored)
    if (((len - (ipmi_msg_len + data_off)) != 0) && 
        ((len - (ipmi_msg_len + data_off)) != 1)) {
        pp_bmc_log_notice("[LAN] IPMIv15 message length wrong - ignored");
        return PP_ERR;
    }

    // get session for message
    // check if selected authtype is allowed for this message
    userlevel_only = 0;
    addr->msg_unauth = 0;
    if (session_id == 0) {
        if (authtype != IPMI_AUTH_NONE) {
            pp_bmc_log_notice("[LAN] IPMIv15 cannot authenticate sessionless message");
            return PP_ERR;
        }
        session = NULL;
    } else {
        session = pp_bmc_get_session(session_id);
        if (session == NULL) {
            // very bad, the requested session does not exist. Ignore the message
            pp_bmc_log_notice("[LAN] IPMIv15 requested sessionID does not exist - message ignored");
            return PP_ERR;
        }
        if (session->password == NULL) {
            // should not happen but theoretically possible (ipmi20 creation without rakp message1)
            pp_bmc_log_notice("[LAN] IPMIv15 could not authenticate session without password");
            pp_bmc_session_delete(session);
            return PP_ERR;
        }
        
        // check if the selected authtype is allowed for this session
        if (session->authtype != authtype) {
            allowed_anyway = 0;

            /* sessionbased messages can be unauthenticated *
             * if one of these two options is on            */
            if (authtype == IPMI_AUTH_NONE) {
                if ((get_lan_channel_config())->user_level_authentication_disable == 1) {
                    /* accept unauthenticated message if it is a user level command */
                    /* (mark this message, decision is done later) */
                    allowed_anyway = 1;
                    userlevel_only = 1;
                }
                if ((get_lan_channel_config())->per_message_authentication_disable == 1) {
                    /* accept unauthenticated message, only user login needs authentication */
                    allowed_anyway = 1;
                }
            }
            
            if (allowed_anyway == 0) {
                pp_bmc_log_debug("[LAN] IPMIv15 ignored message where authtype differs from session");
                pp_bmc_session_delete(session);
                return PP_ERR;
            } else {
                addr->msg_unauth = 1;
            }
        }
    }


    // authenticate message (if necessary) with session->authtype
    if (authtype == IPMI_AUTH_NONE) {
        // everything okay, nothing to authenticate, authtype none is allowed
        // even if session != NULL
    } else {
        if ((session->password == NULL) ||
            (session->challenge == NULL))
        {
            pp_bmc_log_debug("[LAN] IPMIv15 ignored authenticated session without password or challenge");
            pp_bmc_session_delete(session);
            return PP_ERR;
        }
        
        // authtype matches session->authtype, session exists
        MD5_CTX md5_ctx;
        MD2_CTX md2_ctx;

        switch (authtype) {
      //case IPMI_AUTH_NONE cannot happen, see above
        case IPMI_AUTH_MD2:
            // calculate md2
            MD2_Init(&md2_ctx);
            MD2_Update(&md2_ctx, session->password, 16);
            MD2_Update(&md2_ctx, data+5, 4);  // session_id
            MD2_Update(&md2_ctx, ipmi_msg, ipmi_msg_len);
            MD2_Update(&md2_ctx, data+1, 4);  // session_seq_no
            MD2_Update(&md2_ctx, session->password, 16);
            MD2_Final(hash, &md2_ctx);
            if (memcmp(hash, authcode, 16) != 0) {
                pp_bmc_log_notice("[LAN] IPMIv15 md2 authentication failed - message ignored");
                pp_bmc_session_delete(session);
                return PP_ERR;
            }
            break;
        case IPMI_AUTH_MD5:
            // calculate md5
            MD5_Init(&md5_ctx);
            MD5_Update(&md5_ctx, session->password, 16);
            MD5_Update(&md5_ctx, data+5, 4);  // session_id
            MD5_Update(&md5_ctx, ipmi_msg, ipmi_msg_len);
            MD5_Update(&md5_ctx, data+1, 4);  // session_seq_no
            MD5_Update(&md5_ctx, session->password, 16);
            MD5_Final(hash, &md5_ctx);
            if (memcmp(hash, authcode, 16) != 0) {
                pp_bmc_log_notice("[LAN] IPMIv15 md5 authentication failed - message ignored");
                pp_bmc_session_delete(session);
                return PP_ERR;
            }
            break;
        case IPMI_AUTH_STRAIGHT:
            if (memcmp(authcode, session->password, 16) != 0) {
                pp_bmc_log_notice("[LAN] IPMIv15 straight authentication failed - message ignored");
                pp_bmc_session_delete(session);
                return PP_ERR;
            }
            break;
        default:
            // cannot happen but who knows
            pp_bmc_log_error("[LAN] IPMIv15 serious internal error, unknown authtype - message ignored");
            pp_bmc_session_delete(session);
            return PP_ERR;
        }
    }
     
    /* check sequence number, unactivated sessions have an invalid   *
     * sequence (seq=0) right before activate_session is called      *
     * simple window algorithm                                       */
    if ((session != NULL) && (session->inbound_sequence != 0)) {
        /* track inbound session sequence number         *
         * very simple implementation, see spec 6.12.16  */

        /* if one of the sequence numbers could be wrapped add a fixed  *
         * amount to all variables. This will not alter the results but *
         * assure that all numbers are wrapped equally                  */
        if (session_seq_no <= 0x00000008) {
            /* wrapping possible */
            wrap_add = 10;
        } else {
            wrap_add = 0;
        }
        if ((session_seq_no + wrap_add) <= (session->inbound_sequence + wrap_add)) {
            /* new sequence no is smaller than stored sequence no - ignore msg */
            pp_bmc_log_notice("[LAN] IPMIv15 received msg with invalid (old) sequence number");
            pp_bmc_session_delete(session);
            return PP_ERR;
        }
        if ((session_seq_no + wrap_add) > (session->inbound_sequence + 8 + wrap_add)) {
            /* new sequence no is more than 8 ahead - ignore msg */
            /* we might even close the session here              */
            pp_bmc_log_notice("[LAN] IPMIv15 received msg with invalid sequence number");
            pp_bmc_session_delete(session);
            return PP_ERR;
        }
        
        session->inbound_sequence = session_seq_no;
        if (session->inbound_sequence == 0) {
            /* client sent invalid sequence number (wrong wraparound at client) */
            pp_bmc_log_warn("[LAN] wrong sequence number wraparound at client detected");
            session->inbound_sequence = 1;
            pp_bmc_session_delete(session);
            return PP_ERR;
        }
    }

    // call lanserver to handle the message
    rv = pp_bmc_lanserv_ipmi_handle_receive(ipmi_msg, ipmi_msg_len, session, userlevel_only, addr);
    pp_bmc_session_delete(session);
    
    return rv;
}

/**
 * Send message data wrapped with an IPMIv15 header.
 */
int pp_ipmi15_send_msg(unsigned char* data, int datalen, imsg_session_t* session, lan_addr_t* addr) {
    unsigned char* buf;
    int header_len;
    int len;

    unsigned char authtype;
    unsigned int *uip;
    unsigned int ui;
    
    int rv;
    
    if (session == NULL) {
        authtype = 0x00;  // sessionless messages are always unauthenticated
    } else {
        authtype = session->authtype;
        
        if (addr->msg_unauth == 1) {
            /* either no_user_auth or no_msg_auth is active and this message was not authenticated */
            authtype = IPMI_AUTH_NONE;
        }

        if (session->password == NULL) {
            // should not happen as we came in with the same password, but who knows
            pp_bmc_session_delete(session);
            pp_bmc_log_error("[LAN] IPMIv15 could not authenticate outgoing message without password");
            return PP_ERR;
        }
    }
    //pp_bmc_log_debug("[LAN] IPMIv15 sending message");
    
    // determine session header length / data start pointer
    if (authtype == 0x00) {
        header_len = 10;
    } else {
        header_len = 26;
    }
    
    // get buffer (plus one byte for legacy pad)
    buf = malloc(datalen + header_len + 1);
    // copy message
    memcpy(buf + header_len, data, datalen);
    len = header_len + datalen;

    // construct ipmi session header
    buf[0] = authtype;
    if (session == NULL) {
        // both session_id and session_seq = 0;
        memset(buf+1, 0, 8);
    } else {
        session->outbound_sequence++;
        if (session->outbound_sequence == 0) session->outbound_sequence++;
        uip = (unsigned int*)(((unsigned char*)buf)+1);
        *uip = cpu_to_le32(session->outbound_sequence);
        uip = (unsigned int*)(((unsigned char*)buf)+5);
        *uip = cpu_to_le32(session->session_id);
    }
    
    // build authcode if necessary
    if (authtype == 0x00) {
        buf[9] = datalen;
        // authcode does not exist
    } else {
        MD5_CTX md5_ctx;
        MD2_CTX md2_ctx;

        buf[25] = datalen;

        switch (authtype) {
        case IPMI_AUTH_MD2:
                // calculate md2
                MD2_Init(&md2_ctx);
                MD2_Update(&md2_ctx, session->password, 16);
                ui = cpu_to_le32(session->session_id);
                MD2_Update(&md2_ctx, (unsigned char*)(&ui), 4);
                MD2_Update(&md2_ctx, data, datalen);
                ui = cpu_to_le32(session->outbound_sequence);
                MD2_Update(&md2_ctx, (unsigned char*)(&ui), 4);
                MD2_Update(&md2_ctx, session->password, 16);
                MD2_Final(buf+9, &md2_ctx);
                break;
        case IPMI_AUTH_MD5:
                // calculate md5
                MD5_Init(&md5_ctx);
                MD5_Update(&md5_ctx, session->password, 16);
                ui = cpu_to_le32(session->session_id);
                MD5_Update(&md5_ctx, (unsigned char*)(&ui), 4);
                MD5_Update(&md5_ctx, data, datalen);
                ui = cpu_to_le32(session->outbound_sequence);
                MD5_Update(&md5_ctx, (unsigned char*)(&ui), 4);
                MD5_Update(&md5_ctx, session->password, 16);
                MD5_Final(buf+9, &md5_ctx);
                break;
        case IPMI_AUTH_STRAIGHT:
                memcpy(buf+9, session->password, 16);
                break;
        default:
                pp_bmc_log_debug("[LAN] IPMIv15 invalid authtype for sending message");
                free(buf);
                pp_bmc_session_delete(session);
                return PP_ERR;
        }
    }
    
    /* Legacy pad: I think we dont need it         *
     * see spec 13.6, footnote[1] for details      *
     * if (need_legacy_pad) { len++; buf=0x00; }   */

    rv = pp_send_rmcp_msg(buf, len, addr, RMCP_CLASS_IPMI);
    // pp_bmc_session_delete(session); // session is deleted with imsg
    free(buf);
    
    if (rv == PP_ERR) {
        pp_bmc_log_notice("[LAN] IPMIv15 error sending message");
    }

    return rv;
}

