/**
 * adc_kira_sensor.c
 *
 * implements the ADC KIRA sensors
 * 
 * (c) 2006 Peppercon AG, mkl@peppecon.de
 */

#include <malloc.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/ipmi_bits.h>
#include <pp/bmc/ipmi_cmd_sensor.h>

#include <pp/bmc/sensor.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_comdev.h>
#include <pp/bmc/tp_scan_sensdev.h>

#include <adc_kira.h>

#include "drivers/adc_kira_sensor.h"


/**
 * private structures
 */

/**
 * The instance of a ADC KIRA sensor.
 */
typedef struct {
    pp_tp_scan_sensdev_t base;	/* basic sensor description, must be first */
    int dev_fd;			/* ADC KIRA device handle */
    u_char v_channel;           /* voltage rail */
    u_short sdr_mtol;
    signed char sdr_rexp;
} adc_kira_sensor_t;

#define ADC_KIRA_SDR_MTOL  PP_BMC_SDR_M_TOL_2_MTOL(129, 1)

/**
 * private function prototypes
 */
static int  adc_kira_sensor_init(adc_kira_sensor_t* this,
				 const char* id,
				 u_int sensor_idx, int corr_factor,
				 pp_tp_cond_t* scan_cond);
static void adc_kira_cleanup(adc_kira_sensor_t* this);
static void adc_kira_dtor(pp_tp_obj_t* this);
static ipmi_sdr_header_t* adc_kira_default_sdr(pp_tp_sensdev_t* s);
static void adc_kira_update_voltage_reading(pp_sensor_scannable_t* s);


/**
 * public functions
 */

pp_tp_obj_t*
pp_sensor_adc_kira_ctor(const char* id, vector_t* args)
{
    const char* errmsg = "[ADC KIRA] %s: '%s' failed (%s)";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    adc_kira_sensor_t* adc_kira = NULL;
    pp_tp_cond_t* scan;
    u_int corr_factor;
    u_int snum;
    
    if (pp_tp_arg_scanf(args, 0, &err, "dd|o<sc>",
			&snum, &corr_factor, &scan) != 3) {
        pp_bmc_log_error("%s(): '%s' failed: %s (%s)",
            ___F, id, strerror(errno), pp_strstream_buf(&err));
	goto bail;
    } else if (snum > 7) { /* we have 8 sensors (0-7) */
        pp_bmc_log_error("%s(): '%s' failed: invalid sensor number: %d",
	       ___F, id, snum);
	goto bail;
    }
    
    adc_kira = calloc(1, sizeof(adc_kira_sensor_t));
    if (PP_FAILED(adc_kira_sensor_init(adc_kira, id, snum, corr_factor, scan))) {
	pp_bmc_log_error(errmsg, ___F, id, "init");
	free(adc_kira);
	adc_kira = NULL;
	goto bail;
    }

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)adc_kira;
}

static void
adc_kira_dtor(pp_tp_obj_t* this)
{
    adc_kira_cleanup((adc_kira_sensor_t*)this);
    free(this);
}

/**
 * private functions
 */
static int
adc_kira_sensor_init(adc_kira_sensor_t* this, const char* id, u_int sensor_idx, int corr_factor, pp_tp_cond_t* scan_cond)
{
#define DEV_FILE "/dev/adc_kira"
    const char* errmsg = "[ADC KIRA] %s: failed (%s)";
    u_int corr_factor_abs = abs(corr_factor);
    int m, tol;
    
    /* globals */
    if (PP_SUCCED(pp_tp_scan_sensdev_init(&this->base, PP_TP_SCAN_SENS_DEV, id,
					  adc_kira_dtor,
					  adc_kira_default_sdr,
					  adc_kira_update_voltage_reading, scan_cond))) {

	this->dev_fd = open(DEV_FILE, O_RDWR);

	if (this->dev_fd < 0) {
	    pp_bmc_log_perror(errmsg, ___F, " open(" DEV_FILE ") ");
	    adc_kira_cleanup(this);
	    return PP_ERR;
	}

	this->v_channel = sensor_idx;
	this->sdr_rexp = -4;

	m = PP_BMC_SDR_MTOL_2_M(ADC_KIRA_SDR_MTOL);
	tol = PP_BMC_SDR_MTOL_2_TOL(ADC_KIRA_SDR_MTOL);

	if (corr_factor_abs > 1 && corr_factor_abs <= 10) {
	    m = (m * corr_factor_abs + 5) / 10;
	    ++this->sdr_rexp;
	}
	if (corr_factor < 0) m = -m;

	this->sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(m, tol);
	
	/* register for scanning  */
	pp_tp_scan_sensdev_register(&this->base, 0);
    }
    
    return PP_SUC;
}

static ipmi_sdr_header_t*
adc_kira_default_sdr(pp_tp_sensdev_t* s)
{
    adc_kira_sensor_t* this = (adc_kira_sensor_t*)s;
    ipmi_sdr_full_sensor_t* sdr;

    sdr = ipmi_sdr_full_part_create(IPMI_SENSOR_TYPE_VOLTAGE,
			       IPMI_EVENT_READING_TYPE_THRESHOLD,
			       IPMI_UNIT_TYPE_VOLTS,
			       IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
			       this->sdr_mtol, 
			       PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
                               0, this->sdr_rexp,
			       ipmi_sdr_max_by_form(IPMI_ANALOG_DATA_FORMAT_UNSIGNED),
			       ipmi_sdr_min_by_form(IPMI_ANALOG_DATA_FORMAT_UNSIGNED));
    
    return (ipmi_sdr_header_t*)sdr;
}

static void
adc_kira_update_voltage_reading(pp_sensor_scannable_t* s)
{
    adc_kira_sensor_t* adc_kira = PP_TP_SCANNABLE_2_OBJ_CAST(s,adc_kira_sensor_t);
    adc_kira_io_t adc_data;
    int value = -2;
    
    if (pp_tp_scan_sensdev_do_scan(&adc_kira->base)) {
	adc_data.channel = adc_kira->v_channel;
	if (ioctl(adc_kira->dev_fd, ADC_KIRA_IOC_GET_READING, &adc_data) < 0) {
	    pp_bmc_log_perror("%s(): ioctl(ADC_KIRA_IOC_GET_READING)", ___F);
	} else {
	    /* adc_reading is a 10 bit value */
	    value = (adc_data.reading & 0x3FC) >> 2;
	}
    }
    pp_tp_scan_sensdev_update_reading(&adc_kira->base, value);
}

static void
adc_kira_cleanup(adc_kira_sensor_t* this)
{
    pp_tp_scan_sensdev_unregister(&this->base);
    if (this->dev_fd >= 0) close(this->dev_fd);
    pp_tp_scan_sensdev_cleanup(&this->base);
}
