/**
 * adm1031_sensor.c
 *
 * implements the ADM1031 sensors
 * 
 * (c) 2005 Peppercon AG, rwa@peppecon.de
 */

#include <malloc.h>

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/ipmi_bits.h>
#include <pp/bmc/ipmi_cmd_sensor.h>

#include <pp/bmc/sensor.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_comdev.h>
#include <pp/bmc/tp_scan_sensdev.h>

#include "drivers/adm1031_sensor.h"

#define ADM_REG_CONFIG1		0x00

/**
 * private structures
 */

/**
 *  The template of a ADM1031 sensor.
 */
typedef struct {
    /* the sensor type */
    unsigned char sensor_type;
    unsigned char unit_type;
    unsigned char analog_type;
    /* configuration */
    u_char reg;
    /* sdr configuration */
    u_short sdr_mtol;
    u_short sdr_bacc;
    char sdr_bexp;
    char sdr_rexp;
} sensor_config_t;

/**
 * The instance of a ADM1031 sensor.
 */
typedef struct {
    pp_tp_scan_sensdev_t base;	/* basic sensor description, must be first */
    pp_tp_i2c_comdev_t* i2cdev;	/* the I2C device */
    u_char i2caddr;		/* The I2C slave address */
    sensor_config_t* cfg;
} adm1031_sensor_t;


/**
 * Initialize ADM1031 sensor templates.
 */
static sensor_config_t sensor_config[] = {
    { /* 0 - local temperature */
        .sensor_type = IPMI_SENSOR_TYPE_TEMPERATURE,
	.unit_type = IPMI_UNIT_TYPE_DEGREES_C,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
	.reg = 0x0a,
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0,
	.sdr_rexp = 0,
    }, { /* 1 - remote temperature 1 */
        .sensor_type = IPMI_SENSOR_TYPE_TEMPERATURE,
	.unit_type = IPMI_UNIT_TYPE_DEGREES_C,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
	.reg = 0x0b,
	.sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0,
	.sdr_rexp = 0,
    }, { /* 2 - remote temperature 2 */
        .sensor_type = IPMI_SENSOR_TYPE_TEMPERATURE,
	.unit_type = IPMI_UNIT_TYPE_DEGREES_C,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
	.reg = 0x0c,
	.sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0,
	.sdr_rexp = 0,
    }, { /* 3 - fan speed 1 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.reg = 0x08,
	.sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0,
	.sdr_rexp = 0,
    }, { /* 4 - fan speed 2 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_type = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.reg = 0x09,
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
	.sdr_bexp = 0,
	.sdr_rexp = 0,
    }
};


/**
 * private function prototypes
 */
static adm1031_sensor_t*
adm1031_sensor_create(const char* id, pp_tp_i2c_comdev_t* i2cdev,
		      u_char i2caddr, u_int sensor_idx, pp_tp_cond_t* scan);
static ipmi_sdr_header_t* adm1031_default_sdr(pp_tp_sensdev_t* s);
static void adm1031_update_temp_reading(pp_sensor_scannable_t* s);
static void adm1031_destroy(adm1031_sensor_t* sensor);

/**
 * read one register of the ADM1031
 * @return PP_SUC on success
 *         PP_ERR on failure
 */
static inline int
adm1031_read_register(pp_tp_i2c_comdev_t* i2cdev, u_char i2caddr,
		      u_char reg, u_char* data) {
    return i2cdev->rx_byte_data(i2cdev, i2caddr, reg, data);
}

/**
 * write one register of the ADM1031
 * @return PP_SUC on success
 *         PP_ERR on failure
 */
static inline int
adm1031_write_register(pp_tp_i2c_comdev_t* i2cdev, u_char i2caddr,
		       u_char reg, u_char data) {
    return i2cdev->tx_byte_data(i2cdev, i2caddr, reg, data);
}

/**
 * public functions
 */

pp_tp_obj_t*
pp_sensor_adm1031_ctor(const char* id, vector_t* args)
{
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    adm1031_sensor_t* o;
    pp_tp_i2c_comdev_t* i2cdev;
    pp_tp_cond_t* scan;
    int i2caddr;
    u_int snum;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>dd|o<sc>",
			&i2cdev, &i2caddr, &snum, &scan) != 4) {
        pp_bmc_log_error("%s(): '%s' failed: %s (%s)",
            ___F, id, strerror(errno), pp_strstream_buf(&err));
        o = NULL;
    } else if (snum > 2) {
	/* we have 3 temperature sensors (0-2) - fans not supported yet */
        pp_bmc_log_error("%s(): '%s' failed: invalid sensor number: %d",
	       ___F, id, snum);
        o = NULL;
    } else {
        o = adm1031_sensor_create(id, i2cdev, i2caddr, snum, scan);
    }
    pp_strstream_free(&err);

    return (pp_tp_obj_t*)o;
}

/**
 * private functions
 */
static adm1031_sensor_t*
adm1031_sensor_create(const char* id, pp_tp_i2c_comdev_t* i2cdev,
		      u_char i2caddr, u_int sensor_idx, pp_tp_cond_t* scan)
{
    sensor_config_t *sc = &sensor_config[sensor_idx];
    adm1031_sensor_t* adm1031;

    if (sc->sensor_type != IPMI_SENSOR_TYPE_TEMPERATURE
	/*&& sc->sensor_type != IPMI_SENSOR_TYPE_FAN*/) {
	assert(0);
    }

    adm1031 = calloc(1, sizeof(adm1031_sensor_t));

    /* globals */
    pp_tp_scan_sensdev_init(&adm1031->base, PP_TP_SCAN_SENS_DEV, id,
			    (void(*)(pp_tp_obj_t*))adm1031_destroy,
			    adm1031_default_sdr,
			    adm1031_update_temp_reading, scan);
    adm1031->i2cdev = pp_tp_i2c_comdev_duplicate(i2cdev);

    /* adm1031 configuration */
    adm1031->i2caddr = i2caddr;
    adm1031->cfg = sc;

    /* register for scanning  */
    pp_tp_scan_sensdev_register(&adm1031->base, 0);

    return adm1031;
}

static ipmi_sdr_header_t*
adm1031_default_sdr(pp_tp_sensdev_t* s)
{
    adm1031_sensor_t* this = (adm1031_sensor_t*)s;
    ipmi_sdr_full_sensor_t* sdr;
    sensor_config_t* cfg = this->cfg;
    
    sdr = ipmi_sdr_full_part_create(cfg->sensor_type,
				    IPMI_EVENT_READING_TYPE_THRESHOLD,
				    cfg->unit_type, cfg->analog_type,
				    cfg->sdr_mtol, cfg->sdr_bacc,
				    cfg->sdr_bexp, cfg->sdr_rexp,
			            ipmi_sdr_max_by_form(cfg->analog_type),
   		                    ipmi_sdr_min_by_form(cfg->analog_type));
    return (ipmi_sdr_header_t*)sdr;
}


static void
adm1031_update_temp_reading(pp_sensor_scannable_t* s)
{
    adm1031_sensor_t* adm1031 = PP_TP_SCANNABLE_2_OBJ_CAST(s,adm1031_sensor_t);
    int ret = -2;
    pp_tp_i2c_comdev_t* i2cdev = adm1031->i2cdev;
    u_char i2caddr = adm1031->i2caddr;
    u_char value;

    if (pp_tp_scan_sensdev_do_scan(&adm1031->base)) {
	if (PP_SUCCED(ret = pp_tp_i2c_comdev_pre_com(i2cdev, i2caddr))) {

	    /* activate measuring of temperature
	       TODO(miba) doesn't seem to be enough if we do this once
	       during sensor creation/init, why? Check again. */
	    adm1031_write_register(adm1031->i2cdev, adm1031->i2caddr, 
				   ADM_REG_CONFIG1, 0x01);

	    ret = adm1031_read_register(i2cdev, i2caddr,
					adm1031->cfg->reg, &value);
	    pp_tp_i2c_comdev_post_com(i2cdev, i2caddr);
	    if (PP_SUCCED(ret)) ret = value;
	}
    }
    pp_tp_scan_sensdev_update_reading(&adm1031->base, ret);
}

static void
adm1031_destroy(adm1031_sensor_t* sensor)
{
    pp_tp_scan_sensdev_unregister(&sensor->base);
    pp_tp_i2c_comdev_release(sensor->i2cdev);
    pp_tp_scan_sensdev_cleanup(&sensor->base);
    free(sensor);
}
