/**
 * some file header - todo
 * 
 */ 

#include <pp/selector.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>

#include "drivers/adt7463_chip.h"


/* ADT7463 registers */
#define CONFIGURATION_REGISTER_1       0x40
#define CONFIGURATION_REGISTER_3       0x78
#define FAN_PULSES_PER_REVOLUTION      0x7b

#define PWM1_CONFIGURATION_REGISTER    0x5c
#define PWM2_CONFIGURATION_REGISTER    0x5d
#define PWM3_CONFIGURATION_REGISTER    0x5e
  // some bit positions for PWMx configration registers
  #define PWM_CFG_SPIN 0
  #define PWM_CFG_BHVR 5

#define PWM1_DUTY_CYCLE_REGISTER       0x30
#define PWM2_DUTY_CYCLE_REGISTER       0x31
#define PWM3_DUTY_CYCLE_REGISTER       0x32

#define PWM1_MIN_DUTY_CYCLE_REGISTER   0x64
#define PWM2_MIN_DUTY_CYCLE_REGISTER   0x65
#define PWM3_MIN_DUTY_CYCLE_REGISTER   0x66


/* states of the driver */
#define ADT_STATE_OFF            0
#define ADT_STATE_ON_WAITING1    1
#define ADT_STATE_ON_WAITING2    2
#define ADT_STATE_ON_READY       3


/* internal prototypes */
static
pp_tp_obj_t* adt7463_chip_init(const unsigned char* id,
                               pp_tp_i2c_comdev_t* i2cdev,
                               unsigned char i2caddr,
                               pp_tp_cond_t* scan);
static void pp_tp_adt7463_chip_cleanup(pp_tp_adt7463_chip_t* this);
static void power_on_init(pp_tp_adt7463_chip_t* this);
static int  power_on_configure(const int item_id UNUSED, void* ctx);
static int  power_on_initialize_final(const int item_id, void* ctx);
static void power_off_cleanup(pp_tp_adt7463_chip_t* this);


pp_tp_obj_t* pp_adt7463_chip_ctor(const char* id, vector_t* args) {
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;

    pp_tp_i2c_comdev_t* i2cdev;
    int i2caddr;
    pp_tp_cond_t* scan;
      
    if (pp_tp_arg_scanf(args, 0, &err, "o<i>do<sc>",
                        &i2cdev, &i2caddr, &scan) != 3)
    {
        pp_bmc_log_error("'%s' failed: %s (%s)",
                        id, strerror(errno), pp_strstream_buf(&err));
        return NULL;
    }
    
    return adt7463_chip_init(id, i2cdev, i2caddr, scan);
}

static
pp_tp_obj_t* adt7463_chip_init(const unsigned char* id,
                               pp_tp_i2c_comdev_t* i2cdev,
                               unsigned char i2caddr,
                               pp_tp_cond_t* scan)
{   
    pp_tp_adt7463_chip_t* this;

    this = malloc(sizeof(pp_tp_adt7463_chip_t));
    memset(this, 0, sizeof(pp_tp_adt7463_chip_t));
    
    this->adt_state = ADT_STATE_OFF;
    this->init_to_hndl = -1;
    
    pp_tp_i2c_chip_init((pp_tp_i2c_chip_t*)this, PP_TP_I2C_CHIP,
                        id, (void*)pp_tp_adt7463_chip_cleanup,
                        "adt7463", i2cdev, 
                        i2caddr, scan,
                        (void*)power_on_init,
                        (void*)power_off_cleanup);

    pp_bmc_log_info("[ADT7463] '%s' initialized", id);
    
    return (pp_tp_obj_t*)this;
}                                          

static void pp_tp_adt7463_chip_cleanup(pp_tp_adt7463_chip_t* this) {
    pp_bmc_log_info("[ADT7463] cleaning up chip '%s'", ((pp_tp_obj_t*)this)->id);

    if (this->init_to_hndl != -1) {
        pp_select_remove_to(this->init_to_hndl);
    }
    pp_tp_i2c_chip_cleanup((pp_tp_i2c_chip_t*) this);

    free (this);
}

// called on chip_poweron
static void power_on_init(pp_tp_adt7463_chip_t* this) {
    pp_bmc_log_debug("[ADT7463] power_on_init");
    
    /* the chip needs some time to fully power up */
    this->init_to_hndl = pp_select_add_to(1000, 0, power_on_configure, (void*)this);
    this->adt_state = ADT_STATE_ON_WAITING1;
}

static int power_on_configure(const int item_id UNUSED, void* ctx) {
    int ret;
    pp_tp_adt7463_chip_t* this = (pp_tp_adt7463_chip_t*)ctx;
    
    /* assume that the chip power is on long enough to
     * ensure that CONFIGURATION_REGISTER_1.2 is set */
    
    if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(&this->base))) {
        // Not very nice but we have no way to recover if i2c fails ...

        /* set global chip enable */
        if (ret == PP_SUC) ret = pp_tp_i2c_chip_tx_byte_data(&this->base, CONFIGURATION_REGISTER_1, 0x01);
    
        /* fan pulses per revolution (default values ?) */
        if (ret == PP_SUC) ret = pp_tp_i2c_chip_tx_byte_data(&this->base, FAN_PULSES_PER_REVOLUTION, 0x55);
        
        /* set fan continuous reading + fast tacho reading */
        if (ret == PP_SUC) ret = pp_tp_i2c_chip_tx_byte_data(&this->base, CONFIGURATION_REGISTER_3, 0xf8);

        /* set pwm control mode to manual (pwm duty cycle is 100% by default, that's okay) */
        if (ret == PP_SUC) ret = pp_tp_i2c_chip_tx_byte_data(&this->base, PWM1_CONFIGURATION_REGISTER, 0xe0);
        if (ret == PP_SUC) ret = pp_tp_i2c_chip_tx_byte_data(&this->base, PWM2_CONFIGURATION_REGISTER, 0xe0);
        if (ret == PP_SUC) ret = pp_tp_i2c_chip_tx_byte_data(&this->base, PWM3_CONFIGURATION_REGISTER, 0xe0);

        /* minimum pwm frequency for all 3 pwm registers */
        if (ret == PP_SUC) ret = pp_tp_i2c_chip_tx_byte_data(&this->base, PWM1_MIN_DUTY_CYCLE_REGISTER, 0x20);
        if (ret == PP_SUC) ret = pp_tp_i2c_chip_tx_byte_data(&this->base, PWM2_MIN_DUTY_CYCLE_REGISTER, 0x20);
        if (ret == PP_SUC) ret = pp_tp_i2c_chip_tx_byte_data(&this->base, PWM3_MIN_DUTY_CYCLE_REGISTER, 0x20);
        
        /* write lock bit to prevent the BIOS from changing our settings */
        if (ret == PP_SUC) ret = pp_tp_i2c_chip_tx_byte_data(&this->base, CONFIGURATION_REGISTER_1, 0x03);

        pp_tp_i2c_chip_post_com(&this->base);
    }
    
    if (ret != PP_SUC) {
        /* init failed, very bad, reschedule init in 10 seconds */
        this->init_to_hndl = pp_select_add_to(10000, 0, power_on_configure, (void*)this);
        pp_bmc_log_debug("[ADT7463] init failed, retrying in 10s");
    } else {
        /* now wait some time to ensure chip registers contain values */
    
        this->init_to_hndl = pp_select_add_to(2000, 0, power_on_initialize_final, (void*)this);
        this->adt_state = ADT_STATE_ON_WAITING2;
    }
    
    return PP_SUC;
}

static int power_on_initialize_final(const int item_id UNUSED, void* ctx) {
    pp_tp_adt7463_chip_t* this = (pp_tp_adt7463_chip_t*)ctx;
    
    this->init_to_hndl = -1;
    
    /* chip is now initialized and ready to monitor */
    this->adt_state = ADT_STATE_ON_READY;
    
    pp_bmc_log_debug("[ADT7463] chip now ready for monitoring");
    return PP_SUC;
}

static void power_off_cleanup(pp_tp_adt7463_chip_t* this) {
    /* remove pending inits (should not happen) */
    if (this->init_to_hndl != -1) {
        pp_select_remove_to(this->init_to_hndl);
    }
    
    pp_bmc_log_debug("[ADT7463] power_off");
    this->adt_state = ADT_STATE_OFF;
}


/* returns 1 if ready, 0 if not ready */
int pp_tp_adt7463_chip_is_ready(pp_tp_i2c_chip_t* this) {
    if (((pp_tp_adt7463_chip_t*)this)->adt_state == ADT_STATE_ON_READY)
        return 1;
    else
        return 0;
}







