/**
 * adt7463_sensor.c
 *
 * implements all ADT7463 sensors
 * 
 * (c) 2005 Peppercon AG, 02/16/2005, thre@peppecon.de
 */

#include <malloc.h>

#include <pp/base.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/ipmi_bits.h>

#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_comdev.h>
#include <pp/bmc/tp_scan_sensdev.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/debug.h>

#include "drivers/adt7463_sensor.h"
#include "drivers/adt7463_chip.h"



/*
 * ADT7463 sensor structure
 */
typedef struct sensor_config_s {
    /* the sensor type */
    unsigned char sensor_type;
    unsigned char unit_type;
    unsigned char analog_format;
    /* update function */
    void (*update)(pp_sensor_scannable_t*);
    /* configuration */
    union {
        struct {
            unsigned char reg;
        } volt;
        struct {
            unsigned char pulses_shift;
            unsigned char low_reg;
            unsigned char high_reg;
        } fan;
        struct {
            unsigned char reg;
        } temp;
    } config; 
    /* sdr configuration */
    unsigned short sdr_mtol;
    unsigned short sdr_bacc;
    char sdr_bexp;
    char sdr_rexp;
} sensor_config_t;

typedef struct {
    /* the basic sensor description, must be first */
    pp_tp_scan_sensdev_t base;

    /* which adt chip to use */
    pp_tp_i2c_chip_t* i2cchip;

    /* sensor configuration */
    sensor_config_t* cfg;
} adt7463_sensor_t;

/* internal prototypes */
static adt7463_sensor_t* adt7463_sensor_create(/* globals */
					       const char* id,
					       pp_tp_i2c_chip_t* i2cchip,
					       int sensor_idx,
					       pp_tp_cond_t* scan);
//static char* adt7463_to_string(adt7463_sensor_t* sensor);
static ipmi_sdr_header_t* adt7463_default_sdr(pp_tp_sensdev_t* s);
static void adt7463_update_voltage_reading(pp_sensor_scannable_t* s);
static void adt7463_update_tacho_reading(pp_sensor_scannable_t* s);
static void adt7463_update_temp_reading(pp_sensor_scannable_t* s);
static void adt7463_destroy(adt7463_sensor_t* sensor);


/*
 * private structures
 ************************/
static sensor_config_t sensor_config[] = {
    { /* 0 - 5V */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = adt7463_update_voltage_reading,
        .config = { .volt = {
		 .reg = 0x23,
	     },	},
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(26, 1),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 1 - 2.5V */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = adt7463_update_voltage_reading,
        .config = { .volt = {
		 .reg = 0x20,
	     }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(13, 1),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 2 - 12V */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = adt7463_update_voltage_reading,
        .config = { .volt = {
		 .reg = 0x24,
	     }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(63, 1),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 3 - Vcc */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = adt7463_update_voltage_reading,
        .config = { .volt = {
		 .reg = 0x22,
	     }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(17, 1),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
	.sdr_bexp = 0, .sdr_rexp = -3,
    }, { /* 4 - Vccp */
        .sensor_type = IPMI_SENSOR_TYPE_VOLTAGE,
	.unit_type = IPMI_UNIT_TYPE_VOLTS,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = adt7463_update_voltage_reading,
        .config = { .volt = {
		 .reg = 0x21,
	     }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(117, 1),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
	.sdr_bexp = 0, .sdr_rexp = -4,
    }, { /* 5 - fan speed 1 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = adt7463_update_tacho_reading,
        .config = { .fan = {
		 .pulses_shift = 0, .low_reg = 0x28, .high_reg = 0x29,
	     }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(50, 1),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 6 - fan speed 2 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = adt7463_update_tacho_reading,
        .config = { .fan = {
		 .pulses_shift = 2, .low_reg = 0x2a, .high_reg = 0x2b,
	     }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(50, 1),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 7 - fan speed 3 */
        .sensor_type = IPMI_SENSOR_TYPE_FAN,
	.unit_type = IPMI_UNIT_TYPE_RPM,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
	.update = adt7463_update_tacho_reading,
        .config = { .fan = {
		 .pulses_shift = 4, .low_reg = 0x2c, .high_reg = 0x2d,
	     }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(50, 1),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 8 - local temperature */
        .sensor_type = IPMI_SENSOR_TYPE_TEMPERATURE,
	.unit_type = IPMI_UNIT_TYPE_DEGREES_C,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
	.update = adt7463_update_temp_reading,
        .config = { .temp = { .reg = 0x26, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 9 - remote temperature 1 */
        .sensor_type = IPMI_SENSOR_TYPE_TEMPERATURE,
	.unit_type = IPMI_UNIT_TYPE_DEGREES_C,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
	.update = adt7463_update_temp_reading,
        .config = { .temp = { .reg = 0x25, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
	.sdr_bexp = 0, .sdr_rexp = 0,
    }, { /* 10 - remote temperature 2 */
        .sensor_type = IPMI_SENSOR_TYPE_TEMPERATURE,
	.unit_type = IPMI_UNIT_TYPE_DEGREES_C,
	.analog_format = IPMI_ANALOG_DATA_FORMAT_2_COMPL,
	.update = adt7463_update_temp_reading,
        .config = { .temp = { .reg = 0x27, }, },
        .sdr_mtol = PP_BMC_SDR_M_TOL_2_MTOL(1, 0),
	.sdr_bacc = PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
	.sdr_bexp = 0, .sdr_rexp = 0,
    }
};



/*
 * implementations
 */

pp_tp_obj_t* pp_adt7463_sensor_ctor(const char* id, vector_t* args) {
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    adt7463_sensor_t* o;

    pp_tp_i2c_chip_t* i2cchip;
    int snum;
    pp_tp_cond_t* scan;
    
    if (pp_tp_arg_scanf(args, 0, &err, "o<h>d|o<sc>",
			&i2cchip, &snum, &scan) != 3)
    {
	pp_bmc_log_error("[ADT7463] sensor '%s' initialization failed: %s",
	   id, pp_strstream_buf(&err));
	o = NULL;
    } else if (snum > PP_SENSOR_ADT7463_MAX_SENSOR_NUM) {
	pp_bmc_log_error("[ADT7463] sensor '%s' initialization: invalid sensor number: %d",
            id, snum);
	o = NULL;
    } else if (strcmp((i2cchip)->model, "adt7463") != 0) {
        pp_bmc_log_error("[ADT7463] sensor '%s' needs adt7463 chip to initialize",
            id);
        o = NULL;
    } else{
        o = adt7463_sensor_create(id, i2cchip, snum, scan);
    }
    return (pp_tp_obj_t*)o;
}						

/*
 * private functions
 ********************************************************************/

static adt7463_sensor_t* adt7463_sensor_create(/* globals */
					      const char* id,
					      pp_tp_i2c_chip_t* i2cchip,
					      int sensor_idx,
					      pp_tp_cond_t* scan)
{
    sensor_config_t *s = &sensor_config[sensor_idx];
    adt7463_sensor_t* adts = malloc(sizeof(adt7463_sensor_t));

    /* globals */
    pp_tp_scan_sensdev_init(&adts->base, PP_TP_SCAN_SENS_DEV, id,
			    (void(*)(pp_tp_obj_t*))adt7463_destroy,
			    adt7463_default_sdr, s->update, scan);
    adts->i2cchip = pp_tp_i2c_chip_duplicate(i2cchip);
    adts->cfg = s;
	
    /* register for scanning  */
    pp_tp_scan_sensdev_register(&adts->base, 0);

    return adts;
}

static ipmi_sdr_header_t* adt7463_default_sdr(pp_tp_sensdev_t* s) {
    sensor_config_t* cfg = ((adt7463_sensor_t*)s)->cfg;
    ipmi_sdr_full_sensor_t* mysdr;

    mysdr = ipmi_sdr_full_part_create(cfg->sensor_type,
			       IPMI_EVENT_READING_TYPE_THRESHOLD,
			       cfg->unit_type,
			       cfg->analog_format,
			       cfg->sdr_mtol, cfg->sdr_bacc,
			       cfg->sdr_bexp, cfg->sdr_rexp,
			       ipmi_sdr_max_by_form(cfg->analog_format),
			       ipmi_sdr_min_by_form(cfg->analog_format));
				      
    return (ipmi_sdr_header_t*)mysdr;
}

static void adt7463_update_voltage_reading(pp_sensor_scannable_t* s) {
    adt7463_sensor_t* adts = PP_TP_SCANNABLE_2_OBJ_CAST(s, adt7463_sensor_t);
    pp_tp_i2c_chip_t* i2cchip = adts->i2cchip;
    sensor_config_t* cfg = adts->cfg;
    int ret = -2;
    unsigned char value;

    if ( (pp_tp_scan_sensdev_do_scan(&adts->base)) &&
         (pp_tp_adt7463_chip_is_ready(i2cchip)) ) {
        // we should read (scan condition on) and we can read (adt7463 ready)
        
	if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(i2cchip))) {
            ret = pp_tp_i2c_chip_rx_byte_data(i2cchip, cfg->config.volt.reg,
					      &value);
	    pp_tp_i2c_chip_post_com(i2cchip);
           
            /* return value if read successful*/
	    if (ret == PP_SUC) ret = value;
	}
    }
    
    pp_tp_scan_sensdev_update_reading(&adts->base, ret);
}

static void adt7463_update_tacho_reading(pp_sensor_scannable_t* s) {
    adt7463_sensor_t* adts = PP_TP_SCANNABLE_2_OBJ_CAST(s, adt7463_sensor_t);
    sensor_config_t* cfg = adts->cfg;
    pp_tp_i2c_chip_t* i2cchip = adts->i2cchip;
    int ret = -2;
    u_char conf_reg, pulses;
    u_char low, high;
    u_int pulses_per_minute;
    unsigned char value;
    unsigned int cycles;

    if ( (pp_tp_scan_sensdev_do_scan(&adts->base)) &&
         (pp_tp_adt7463_chip_is_ready(i2cchip)) ) {
        // we should read (scan condition on) and we can read (adt7463 ready)

	if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(i2cchip))) {

            // read all relevant registers
	    (   (PP_SUC == (ret = pp_tp_i2c_chip_rx_byte_data(i2cchip, 0x7B,
							      &conf_reg)))
             && (PP_SUC == (ret = pp_tp_i2c_chip_rx_byte_data(i2cchip,
					 cfg->config.fan.low_reg, &low)))
             && (PP_SUC == (ret = pp_tp_i2c_chip_rx_byte_data(i2cchip,
					 cfg->config.fan.high_reg, &high))) );
	
	    pp_tp_i2c_chip_post_com(i2cchip);
	    
            /* return value if read successful*/
	    if (ret == PP_SUC) {
		
		// how many clock cycles have been counted?
		cycles = (high << 8) | low;
	    
		if (cycles == 0xffff || cycles == 0) {
		    // fan is not moving, set value to 0 and return
		    value = 0;
		} else {
		    // how many fan pulses are counted?
		    pulses = ((conf_reg >> cfg->config.fan.pulses_shift) & 3)
			+ 1;
		    
		    // the counter tells us the number of clock cycles (90 kHz)
		    // for one fan revolution
		    pulses_per_minute = (90000 * 60 * 2) / (pulses * cycles);
		    
		    // convert the count into a byte value
		    value = min((u_int)255, pulses_per_minute / 50);
		}
		ret = value;
	    }
	}
    }

    pp_tp_scan_sensdev_update_reading(&adts->base, ret);
}

static void adt7463_update_temp_reading(pp_sensor_scannable_t* s) {
    adt7463_sensor_t* adts = PP_TP_SCANNABLE_2_OBJ_CAST(s, adt7463_sensor_t);
    sensor_config_t* cfg = adts->cfg;
    pp_tp_i2c_chip_t* i2cchip = adts->i2cchip;
    int ret = -2;
    unsigned char value;

    if ( (pp_tp_scan_sensdev_do_scan(&adts->base)) &&
         (pp_tp_adt7463_chip_is_ready(i2cchip)) ) {
        // we should read (scan condition on) and we can read (adt7463 ready)

	if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(i2cchip))) {
            ret = pp_tp_i2c_chip_rx_byte_data(i2cchip, cfg->config.temp.reg,
					      &value);
            
	    pp_tp_i2c_chip_post_com(i2cchip);
	    
	    if (ret == PP_SUC) {
		// check for correct values --> 0x80 means no value
		if (value == 0x80) {
		    errno = ENODATA;
		    ret = PP_ERR;
		} else {
		    ret = value;
		}
	    } else errno = EIO;
	} else errno = EIO;
    }

    pp_tp_scan_sensdev_update_reading(&adts->base, ret);
}

static void adt7463_destroy(adt7463_sensor_t* sensor) {
    /* unregister from scanning  */
    pp_tp_scan_sensdev_unregister(&sensor->base);
    
    pp_tp_i2c_chip_release(sensor->i2cchip);
    pp_tp_scan_sensdev_cleanup(&sensor->base);
    free(sensor);
}

