/*
 * LM75 sensor
 * 
 * roka@peppercon.de
 */


#include <malloc.h>

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_comdev.h>
#include <pp/bmc/tp_scan_sensdev.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/ipmi_bits.h>
#include <pp/bmc/sensor.h>

#include "drivers/lm75_sensor.h"


/*
 * private structures
 ************************/


typedef struct {
    /* the basic sensor description, must be first */
    pp_tp_scan_sensdev_t base;
    
    /* internal variables */
    pp_tp_i2c_comdev_t* i2cdev;

    /* which LM75 to use? */
    unsigned char i2caddr;
} lm75_sensor_t;

/*
 * private function prototypes
 ***************************************/

static lm75_sensor_t*
lm75_sensor_create(const char* id, pp_tp_i2c_comdev_t* i2cdev,
		   unsigned char i2caddr, pp_tp_cond_t* scan);
static ipmi_sdr_header_t* lm75_default_sdr(pp_tp_sensdev_t* s);
static void lm75_update_temp_reading(pp_sensor_scannable_t* s);
static void lm75_destroy(lm75_sensor_t* sensor);

/**
 * read one register of the LM75
 * @return PP_SUC on success
 *         PP_ERR on failure
 */
static inline int lm75_read_register(pp_tp_i2c_comdev_t* i2cdev,
					unsigned char i2caddr,
					unsigned char* data) {
    return i2cdev->rx_byte_data(i2cdev, i2caddr, 0x0, data);
}

/**
 * write one register of the LM75
 * @return PP_SUC on success
 *         PP_ERR on failure
 */
static inline int lm75_write_register(pp_tp_i2c_comdev_t* i2cdev,
					 unsigned char i2caddr,
					 unsigned char reg,
					 unsigned char data) {
    return i2cdev->tx_byte_data(i2cdev, i2caddr, reg, data);
}

/**
 * LM75 constructor
 */
pp_tp_obj_t* pp_sensor_lm75_ctor(const char *id, vector_t *args) {
    const char* fn = __FUNCTION__;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    lm75_sensor_t* lm75_object;
    pp_tp_i2c_comdev_t* i2cdev;
    pp_tp_cond_t* scan;
    int i2caddr;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>d|o<sc>",
			&i2cdev, &i2caddr, &scan) != 3) {
	pp_bmc_log_error("%s: '%s' failed: %s (%s)",
		fn, id, strerror(errno), pp_strstream_buf(&err));
	lm75_object = NULL;
    } 
    else {
        lm75_object = lm75_sensor_create(id, i2cdev, i2caddr, scan);
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*) lm75_object;
}


/*
 * private functions
 ********************************************************************/

static lm75_sensor_t* lm75_sensor_create(const char* id,
					 pp_tp_i2c_comdev_t* i2cdev,
					 unsigned char i2caddr,
					 pp_tp_cond_t* scan) {
    lm75_sensor_t* lm_sensor = malloc(sizeof(lm75_sensor_t));
    memset(lm_sensor, 0, sizeof(lm75_sensor_t));

    /* globals */
    pp_tp_scan_sensdev_init(&lm_sensor->base, PP_TP_SCAN_SENS_DEV, id,
			    (void(*)(pp_tp_obj_t*))lm75_destroy,
			    lm75_default_sdr,
			    lm75_update_temp_reading, scan);

    lm_sensor->i2cdev = pp_tp_i2c_comdev_duplicate(i2cdev);
    
    /* lm75 configuration */
    lm_sensor->i2caddr = i2caddr;

    /* register for scanning  */
    pp_tp_scan_sensdev_register(&lm_sensor->base, 0);

    return lm_sensor;
}

static ipmi_sdr_header_t* lm75_default_sdr(pp_tp_sensdev_t* s UNUSED) {
    ipmi_sdr_full_sensor_t* sdr;
    
    sdr = ipmi_sdr_full_part_create(IPMI_SENSOR_TYPE_TEMPERATURE,
				    IPMI_EVENT_READING_TYPE_THRESHOLD,
				    IPMI_UNIT_TYPE_DEGREES_C,
				    IPMI_ANALOG_DATA_FORMAT_2_COMPL,
				    PP_BMC_SDR_M_TOL_2_MTOL(1, 0), 
				    PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
                                    0, 0,
		   ipmi_sdr_max_by_form(IPMI_ANALOG_DATA_FORMAT_2_COMPL),
   		   ipmi_sdr_min_by_form(IPMI_ANALOG_DATA_FORMAT_2_COMPL));
    return (ipmi_sdr_header_t*)sdr;
}

/*
  return: ret holds a 8-bit signed value. If the 32-bit ret value is
  negative, this indicates an error -- just guessing from what tbr told me.
*/
static void lm75_update_temp_reading(pp_sensor_scannable_t* s) {
    lm75_sensor_t* lm75_sensor = PP_TP_SCANNABLE_2_OBJ_CAST(s, lm75_sensor_t);
    pp_tp_i2c_comdev_t* i2cdev = lm75_sensor->i2cdev;
    unsigned char i2caddr = lm75_sensor->i2caddr;
    int ret = -2;
    u_char value;

    if (pp_tp_scan_sensdev_do_scan(&lm75_sensor->base)) {
	if (PP_SUC == (ret = pp_tp_i2c_comdev_pre_com(i2cdev, i2caddr))) {
	    ret = lm75_read_register(i2cdev, i2caddr, &value);
	    pp_tp_i2c_comdev_post_com(i2cdev, i2caddr);
	    
	    if (ret == PP_SUC) {
		// TODO: check for correct values --> 0x80 means no value
		if (value == 0x80) {
		    errno = ENODATA;
		    ret = PP_ERR;
		} else {
		    ret = value;
		}
	    }
	}
    }
    pp_tp_scan_sensdev_update_reading(&lm75_sensor->base, ret);
}

static void lm75_destroy(lm75_sensor_t* sensor) {
    pp_tp_scan_sensdev_unregister(&sensor->base);
    pp_tp_i2c_comdev_release(sensor->i2cdev);
    pp_tp_scan_sensdev_cleanup(&sensor->base);
    free(sensor);
}

