/**
 * fpga_gpio_dev.c
 *
 * Device driver class for the ext_gpio in the KIMs FPGA
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <unistd.h>
#include <fcntl.h>

#include <pp/base.h>
#include <pp/selector.h>
#include <pp_gpio.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>

#include "drivers/pp_fpga_gpio_dev.h"

/*
 * If POLLING is defined, the driver does not use the HW interrupt feature of
 * FPGA GPIO core. Instead, the GPIOs are monitored by polling each 50 msecs.
 */
#define noPOLLING

static int pp_fpga_gpio_get_val(pp_tp_gpio_dev_t* d, unsigned long mask) {
    int value = ((pp_fpga_gpio_dev_t*)d)->value;
    if (value >= 0) value &= mask;
    return value;
}

static int pp_fpga_gpio_set_val(pp_tp_gpio_dev_t* d, unsigned long mask,
				unsigned long val, unsigned long tristate) {
    assert((mask & ~d->valid_mask) == 0);
    gpio_ioctl_out(((pp_fpga_gpio_dev_t*)d)->fd_gpio,
                   mask & ~tristate & val,
		   mask & ~tristate & ~val,
		   mask & tristate);

    return PP_SUC;
}

static int pp_fpga_gpio_init_gpio(pp_tp_gpio_dev_t* d, unsigned long mask)
{
    if (gpio_ioctl_alt_func(((pp_fpga_gpio_dev_t*)d)->fd_gpio,
			    0, mask) != 0) {
	pp_bmc_log_error("[EGPIO] '%s', alternate function ioctl failed", ((pp_tp_obj_t*)d)->id);
	return PP_ERR;
    }

    return PP_SUC;
}

static void pp_fpga_gpio_subscriber_changed(pp_tp_gpio_dev_t* d)
{
    pp_fpga_gpio_dev_t *o = (pp_fpga_gpio_dev_t*)d;

    /* determine new resulting bit mask */
    int i, s = vector_size(d->gpio_subscribers);
    o->mask = 0;
    for (i = 0; i < s; i++) {
        pp_tp_gpio_subscriber_t *sub = vector_get(d->gpio_subscribers, i);
        o->mask |= sub->get_gpio_mask(sub);
    }

#ifndef POLLING
    /* set mask to edge triggered for raising and falling
       and switch IRQ on */
    if (0 > gpio_ioctl_irqconf(o->fd_gpio, 0, o->mask, o->mask,o->mask) ||
	0 > gpio_ioctl_watch(o->fd_gpio, o->mask)) {
	pp_bmc_log_error("[EGPIO] '%s', pp_fpga_gpio_subscriber_changed: "
			 "IRQ config failed");
    }
#endif
}

#ifdef POLLING
static int read_sensor_to(const int item_id UNUSED, void* ctx)
{
    pp_fpga_gpio_dev_t *o = ctx;
    unsigned long old_value = o->value;
    unsigned long new_value;
    unsigned long change_mask;
    int ret;

    // read values
    ret = gpio_ioctl_in(o->fd_gpio, &new_value);

    // notify subscribers
    if (new_value != old_value) {
	o->value = new_value;
	if (ret < 0) change_mask = 0;
	else change_mask = o->valid < 0 ? (unsigned long)(old_value ^ new_value) :
					  o->base.valid_mask;
	o->valid = ret < 0 ? 0 : 1;
	pp_tp_gpio_dev_notify_subscribers(&o->base, new_value, change_mask, o->valid);
    }
    return PP_SUC;
}
#else
static int read_sensor_cb(const int item_id UNUSED, const int fd UNUSED,
                          const short event UNUSED, void* ctx)
{
    pp_fpga_gpio_dev_t *o = ctx;
    int old_valid = o->valid;
    int new_valid;
    unsigned long old_value = o->value;
    unsigned long new_value;
    unsigned long change_mask;
    int ret;

    // check, what really has been changed at least once (reset modified mask)
    // (since some values could have been changed back to origin value)
    ret = gpio_ioctl_get_modified(o->fd_gpio, &change_mask);
    if (ret < 0) change_mask = 0; // not supported by device :(

    // read values (ack intr)
    ret = gpio_ioctl_in(o->fd_gpio, &new_value);
    new_valid = ret < 0 ? 0 : 1;

    if (new_valid != old_valid) {
        // validity changed: inform all subscriber
        change_mask = o->base.valid_mask;
        if (new_valid) o->valid = new_valid;
    } if (new_valid && new_value != old_value) {
        // value changed: inform affected subscriber
        // (change_mask might already be pre-set by gpio_ioctl_get_modified!)
        change_mask = change_mask | (old_value ^ new_value);
        o->value = new_value;
    }

    // notify subscribers
    if (change_mask) {
        pp_tp_gpio_dev_notify_subscribers(&o->base, o->value, change_mask, o->valid);
    }
    return PP_SUC;
}
#endif

static void startup(pp_sensor_startable_t* s)
{
    pp_fpga_gpio_dev_t* d =
	PP_TP_INTF_2_OBJ_CAST(s, pp_fpga_gpio_dev_t, startable);

    // the sensor scanner has started and all sensors are there:
    //   send initial values
#ifdef POLLING
    read_sensor_to(d->cb_hndl, d);
#else
    read_sensor_cb(d->cb_hndl, d->fd_gpio, POLLIN, d);
#endif
}

static void pp_fpga_gpio_dev_cleanup(pp_tp_obj_t* d)
{
    pp_fpga_gpio_dev_t *o = (pp_fpga_gpio_dev_t*)d;
    pp_bmc_log_debug("[EGPIO] cleanup");

    pp_sensor_scanner_remove_startable(&o->startable);

#ifdef POLLING
    pp_select_remove_to(o->cb_hndl);
#else
    pp_select_remove_fd(o->cb_hndl);
#endif
    close(o->fd_gpio);
    pp_tp_gpio_dev_cleanup((pp_tp_gpio_dev_t*)d);
}

int pp_fpga_gpio_dev_init(pp_fpga_gpio_dev_t* d, char* gpio_devname)
{
    char dev[32], name[32];

    snprintf(dev, sizeof(dev), "/dev/%s", gpio_devname);
    snprintf(name, sizeof(name), "GPIO '%s' device", gpio_devname);
    pp_bmc_log_debug("[EGPIO] init (%s)", dev);

    d->fd_gpio = open(dev, O_RDWR);
    if (d->fd_gpio < 0)
        return PP_ERR;

    pp_tp_gpio_dev_init((pp_tp_gpio_dev_t*)d,
                           PP_FPGA_GPIO_DEV,
                           name,
                           0xffffffff, /* potentially support all 32 bits */
                           pp_fpga_gpio_dev_cleanup,
			   pp_fpga_gpio_init_gpio,
                           pp_fpga_gpio_get_val,
                           pp_fpga_gpio_set_val,
                           pp_fpga_gpio_subscriber_changed);

    d->value = -1;
    d->mask = 0;

    /* TODO check gpio ioctl for success/error once low-level functions support it */

#ifdef POLLING
    d->cb_hndl = pp_select_add_to(50, 1, read_sensor_to, d);
#else
    d->cb_hndl = pp_select_add_fd(d->fd_gpio, POLLIN, read_sensor_cb, d);
#endif

    // register for initial startup
    d->startable.start = startup;
    pp_sensor_scanner_add_startable(&d->startable);

    return PP_SUC;
}

pp_tp_obj_t* pp_fpga_gpio_dev_ctor(const char* id UNUSED, vector_t* args)
{
    const char* fn = __FUNCTION__;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_fpga_gpio_dev_t* egpio;

    char *gpio_devname;

    if (vector_size(args) > 1) {
	pp_bmc_log_error("ERROR: pp_fpga_gpio_dev_ctor '%s' failed: too many arguments",
            id);
        return NULL;
    }

    if (pp_tp_arg_scanf(args, 0, &err, "s", &gpio_devname) != 1) {
        pp_bmc_log_error("%s: '%s' failed: %s (%s)",
            fn, id, strerror(errno), pp_strstream_buf(&err));
        return NULL;
    }

    egpio = malloc(sizeof(pp_fpga_gpio_dev_t));
    if (pp_fpga_gpio_dev_init(egpio, gpio_devname) == PP_ERR) {
        free(egpio);
        pp_bmc_log_debug("[EGPIO] initialization failed");
        return NULL;
    }

    return (pp_tp_obj_t*)egpio;
}
