/**
 * rpc_cbk_sensor.c
 *
 * Circuit breaker sensor for RPC, this is a 'virtual' sensor
 * determining voltage measurement to decide that circuit breaker
 * was released
 * 
 * (c) 2006 Peppercon AG, 2006/10/23, tbr@raritan.com
 */
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_sensdev.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/selector.h>

#include "drivers/rpc_cbk_sensor.h"

typedef struct cbk_sens_s {
    pp_tp_sensdev_t base;
    /* Circuit sensors for safety */
    pp_tp_sensdev_t* volt_sens;
    pp_tp_sensdev_subscriber_t sens_subscr;
} cbk_sens_t;

static void cbk_sens_dtor(pp_tp_obj_t* o);
static ipmi_sdr_header_t* cbk_sens_default_sdr(struct pp_tp_sensdev_s* s);
static void cbk_sens_recv_reading(pp_tp_sensdev_subscriber_t*,
				  pp_tp_sensdev_t* source, int reading);


pp_tp_obj_t* pp_rpc_cbk_sens_ctor(const char* id, vector_t* args) {
    const char* errmsg = "[CBKSens] %s c'tor: %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    cbk_sens_t* this = NULL;
    pp_tp_sensdev_t* volt_sens = NULL;

    if (pp_tp_arg_scanf(args, 0, &err, "o<s>", &volt_sens) != 1) {
	pp_bmc_log_perror(errmsg, id, pp_strstream_buf(&err));
    } else {
	this = malloc(sizeof(cbk_sens_t));
	pp_tp_sensdev_init(&this->base,  PP_TP_SENS_DEV, id,
			   cbk_sens_dtor, cbk_sens_default_sdr);
	this->volt_sens = pp_tp_sensdev_duplicate(volt_sens);
	this->sens_subscr.recv_reading = cbk_sens_recv_reading;    
	pp_bmc_tp_sensdev_subscribe(this->volt_sens, &this->sens_subscr);
	pp_bmc_log_debug(errmsg, id, "initialized");
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void cbk_sens_dtor(pp_tp_obj_t* o) {
    cbk_sens_t* this = (cbk_sens_t*)o;
    assert(this != NULL);
    pp_bmc_log_debug("[CBKSens] %s d'tor", pp_tp_obj_to_string(o));
    pp_bmc_tp_sensdev_unsubscribe(this->volt_sens, &this->sens_subscr);
    pp_tp_sensdev_release(this->volt_sens);
    pp_tp_sensdev_cleanup(&this->base);
    free(this);
}

static ipmi_sdr_header_t*
cbk_sens_default_sdr(struct pp_tp_sensdev_s* s UNUSED) {
    ipmi_sdr_compact_sensor_t* sdr;
    sdr = ipmi_sdr_compact_part_create(IPMI_SENSOR_TYPE_POWER_SUPPLY,
				       IPMI_EVENT_READING_TYPE_DISCRETE_STATE);
    return (ipmi_sdr_header_t*)sdr;      
}
  
static void cbk_sens_recv_reading(pp_tp_sensdev_subscriber_t* o , 
				  pp_tp_sensdev_t* source UNUSED, int reading) {
    cbk_sens_t *this = PP_TP_INTF_2_OBJ_CAST(o, cbk_sens_t, sens_subscr);
    int new_reading;
    
    new_reading = (reading < 0) ? -1 : (reading < 10) ? 1 : 0;
    if (this->base.reading != new_reading) {
	this->base.reading = new_reading;
	pp_bmc_tp_sensdev_notify_subscribers(&this->base, new_reading);
    }
}
