/**
 * rpc_olled_ctrl.c 
 *
 * Implements control of LED associated with each outlet
 *
 * (c) 2006 Peppercon AG, 2006/10/10 tbr@raritan.com
 */

#include <pp/base.h>
#include <pp/selector.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/tp_gpio_sens.h>
#include <pp/bmc/tp_ipmi_sens.h>
#include <pp/bmc/tp_gpio_act.h>
#include <pp/bmc/tp_ctrl.h>
#include <pp/bmc/rpc_olled_ctrl.h>

typedef struct rc_olled_ctrl_s {
    pp_tp_ctrl_t base;

    pp_tp_gpio_sens_t* olstate;
    pp_tp_gpio_act_t* red;
    pp_tp_gpio_act_t* green;
    pp_tp_sensdev_t* cbreak;
    pp_tp_ipmi_sens_t* curr;

    char overcurrent;
    char red_on;
    char green_on;
    char blinking;
    char period;   /* whether currently on or off during blinking */
    int  blink_to_id;

    pp_tp_sensdev_subscriber_t sens_subsc;
    pp_tp_ipmi_sens_subscriber_t isens_subsc;
} rc_olled_ctrl_t;

static void rpc_olled_ctrl_dtor(pp_tp_obj_t* o);
static void rpc_olled_ctrl_isens_recv_reading(pp_tp_ipmi_sens_subscriber_t* s,
					      pp_tp_ipmi_sens_t* source, int reading);
static void rpc_olled_ctrl_sens_recv_reading(pp_tp_sensdev_subscriber_t* s,
					     pp_tp_sensdev_t* source, int reading);
static void rpc_olled_ctrl_set_state(rc_olled_ctrl_t* this);
static void rpc_olled_set_led(rc_olled_ctrl_t* this);
static int rpc_olled_blink_to_hndl(int item_id UNUSED, void* ctx);

#define RPC_OLLED_BLINK_PERIOD_MS 500

pp_tp_obj_t* pp_rpc_olled_ctrl_ctor(const char* id, vector_t* args) {
    const char* fn = "[RPCOllCtrl] ctor";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    rc_olled_ctrl_t* this = NULL;
    pp_tp_gpio_sens_t* olstate;
    pp_tp_sensdev_t* cbreak;
    pp_tp_ipmi_sens_t* curr;
    pp_tp_gpio_act_t* red;
    pp_tp_gpio_act_t* green;

    if (pp_tp_arg_scanf(args, 0, &err, "o<sg>o<p>o<s>o<ag>o<ag>",
			&olstate, &curr, &cbreak, &red, &green) != 5) {
	pp_bmc_log_perror("%s(): '%s' failed: %s", fn, id,
			  pp_strstream_buf(&err));
    } else {
	this = malloc(sizeof(rc_olled_ctrl_t));
	pp_tp_ctrl_init(&this->base, PP_TP_OLLED_CTRL, id,rpc_olled_ctrl_dtor);
	
	this->red_on = this->green_on = this->blinking = this->overcurrent = 0;
	this->olstate = pp_tp_gpio_sens_duplicate(olstate);
	this->red = pp_tp_gpio_act_duplicate(red);
	this->green = pp_tp_gpio_act_duplicate(green);
	this->cbreak = pp_tp_sensdev_duplicate(cbreak);
	this->curr = pp_tp_ipmi_sens_duplicate(curr); 
	this->blink_to_id = 0;
	
	this->sens_subsc.recv_reading = rpc_olled_ctrl_sens_recv_reading;
	pp_tp_gpio_sens_subscribe(olstate, &this->sens_subsc);
	pp_bmc_tp_sensdev_subscribe(cbreak, &this->sens_subsc);
	
	this->isens_subsc.recv_reading = rpc_olled_ctrl_isens_recv_reading;
	pp_tp_ipmi_sens_subscribe(curr, &this->isens_subsc);
	pp_bmc_log_debug("%s(): '%s' initialized", fn, id);
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void rpc_olled_ctrl_dtor(pp_tp_obj_t* o) {
    rc_olled_ctrl_t* this = (rc_olled_ctrl_t*)o;
    assert(this != NULL);
    pp_bmc_log_debug("[RPCOllCtrl] dtor(): '%s' deinit",
		     pp_tp_obj_to_string(o));
    pp_tp_ipmi_sens_unsubscribe(this->curr, &this->isens_subsc);
    pp_bmc_tp_sensdev_unsubscribe(this->cbreak, &this->sens_subsc);
    pp_tp_gpio_sens_unsubscribe(this->olstate, &this->sens_subsc);
    pp_tp_ipmi_sens_release(this->curr);	
    pp_tp_sensdev_release(this->cbreak);
    pp_tp_gpio_act_release(this->green);
    pp_tp_gpio_act_release(this->red);
    pp_tp_gpio_sens_release(this->olstate);
    pp_tp_ctrl_cleanup(&this->base);
    free(this);
}

static void rpc_olled_ctrl_isens_recv_reading(pp_tp_ipmi_sens_subscriber_t* s,
					     pp_tp_ipmi_sens_t* source UNUSED, int reading UNUSED) {
    rc_olled_ctrl_t* this = PP_TP_INTF_2_OBJ_CAST(s, rc_olled_ctrl_t,
						  isens_subsc);
    unsigned char rsp_code;
    ipmi_sensor_get_reading_rs_t icurr_read;
    int ol_overcurrent;
    
    if (pp_tp_ipmi_sens_get_reading(this->curr, &rsp_code, &icurr_read)
	== PP_SUC && !icurr_read.rt.unavailable) {
	ol_overcurrent = icurr_read.rt.above_upper_crit;
    } else {
	ol_overcurrent = 0;
    }

    if (this->overcurrent != ol_overcurrent) {
	this->overcurrent = ol_overcurrent;
	rpc_olled_ctrl_set_state(this);
    }
}

static void rpc_olled_ctrl_sens_recv_reading(pp_tp_sensdev_subscriber_t* s,
					     pp_tp_sensdev_t* source UNUSED, int reading UNUSED) {
    rc_olled_ctrl_t* this = PP_TP_INTF_2_OBJ_CAST(s, rc_olled_ctrl_t,
						  sens_subsc);
    rpc_olled_ctrl_set_state(this);
    
}
 
static void rpc_olled_ctrl_set_state(rc_olled_ctrl_t* this) {
    int ol_overcurrent = this->overcurrent;
    int ol_close;
    int cb_open;
    
    ol_close = pp_tp_gpio_sens_get_reading(this->olstate);
    if (ol_close < 0) ol_close = 0;
    
    cb_open = pp_tp_sensdev_get_reading(this->cbreak);
    if (cb_open < 0) cb_open = 0;

    this->period   = 0;          /* calculate state */
    this->blinking = (cb_open || (ol_close && ol_overcurrent));
    this->red_on   = (ol_close);
    this->green_on = (!ol_close || cb_open);

    if (this->blink_to_id != 0) { /* cancel old timer, if any */
	pp_select_remove_to(this->blink_to_id);
    }
    if (this->blinking) {         /* set new timer, if required */
	this->blink_to_id = pp_select_add_to(RPC_OLLED_BLINK_PERIOD_MS, 1,
					     rpc_olled_blink_to_hndl, this);
    }
    rpc_olled_set_led(this);
}

static void rpc_olled_set_led(rc_olled_ctrl_t* this) {
    if (this->blinking && this->period) {
	if (this->red_on) pp_tp_gpio_act_set_gpio(this->red, 0);
	if (this->green_on) pp_tp_gpio_act_set_gpio(this->green, 0);
    } else {
	pp_tp_gpio_act_set_gpio(this->red, this->red_on);
	pp_tp_gpio_act_set_gpio(this->green, this->green_on);
    }
}

static int rpc_olled_blink_to_hndl(int item_id UNUSED, void* ctx) {
    rc_olled_ctrl_t* this = (rc_olled_ctrl_t*)ctx;
    this->period = this->period ? 0 : 1;
    rpc_olled_set_led(this);
    return PP_SUC;
}

unsigned char pp_bmc_rpc_olled_ctrl_get_led_state(pp_tp_ctrl_t *ctrl) {
    rc_olled_ctrl_t *this;
    unsigned char led_state;

    assert(PP_TP_OBJ_IS_TYPE(PP_TP_OLLED_CTRL, ctrl));
    
    this = (rc_olled_ctrl_t*)ctrl;
    
    led_state =  (this->green_on & 0x01) |
                ((this->red_on   & 0x01) << 1) |
                ((this->blinking & 0x01) << 2);
                
    return led_state;
}
