/**
 * rpc_pwr_sens.c
 *
 * implements the PDU Sensor / Receptacle controller chip
 * This file includes all related objects (chip, sensors, pwm, gpio)
 * 
 * (c) 2006 Peppercon AG, 2006/3/27, george@raritan.com
 */
#include <math.h>
#include <pp/bmc/tp_scan_sensdev.h>
#include <pp/bmc/ipmi_sdr.h>
#include "drivers/rpc_pwr_sens.h"


/*
 * Sensor total current
 **************************/
typedef struct curr_subsc_entry_s {
    pp_rpc_curr_sens_t* obj;
    pp_tp_sensdev_t* sens;
    pp_tp_sensdev_subscriber_t subscriber;
    size_t                   index;	
} curr_subsc_entry_t;

pp_tp_obj_t* pp_rpc_curr_sens_ctor(const char* id, vector_t* args) {
    const char* errmsg = "[RPCCurrSens] c'tor: '%s' failed: %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_rpc_curr_sens_t* this = NULL;
    int i, arg_cnt;
    vector_t * sens = NULL;
	
    arg_cnt = vector_size(args);

    if (arg_cnt < 2) {
	pp_bmc_log_error(errmsg, id, "missing argument(s)");
	goto bail;
    } else if (arg_cnt > 24) {
	pp_bmc_log_error(errmsg, id, "too many arguments");
	goto bail;
    }
    sens = vector_new(NULL, arg_cnt,
		      (vector_elem_del_func_simple)pp_tp_obj_release);
	
    for (i = 0; i <arg_cnt; ++i) {
	pp_tp_sensdev_t * curr_sens;
        if (pp_tp_arg_scanf(args, i, &err, "o<s>", &curr_sens) != 1) {
	    pp_bmc_log_error(errmsg, id, pp_strstream_buf(&err));
	    vector_delete(sens);
            goto bail;
        }
	vector_add(sens, pp_tp_sensdev_duplicate(curr_sens));
    }
    this = calloc(1, sizeof(pp_rpc_curr_sens_t));
    if (PP_FAILED(pp_rpc_curr_sens_init(this,
					PP_TP_SENS_DEV, id,
					pp_rpc_curr_sens_dtor,
					sens))) {
	pp_bmc_log_error(errmsg, id, "init");
	vector_delete(sens);
	free(this);
	this = NULL;
    }   

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;

}


/* internal prototypes */
static void rpc_curr_sens_recv_reading(pp_tp_sensdev_subscriber_t* subsc,
				       pp_tp_sensdev_t* source, int reading);

static ipmi_sdr_header_t* curr_sens_default_sdr(pp_tp_sensdev_t* s UNUSED) {
    ipmi_sdr_full_sensor_t* sdr;
	
    //pp_bmc_log_debug("%s M=%d B=%d",this->base.base.base.id,cfg->sdr_mtol,cfg->sdr_bacc);		//george
    sdr = ipmi_sdr_full_part_create(IPMI_SENSOR_TYPE_CURRENT,
				    IPMI_EVENT_READING_TYPE_THRESHOLD,
				    IPMI_UNIT_TYPE_AMPS,
				    IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
				    PP_BMC_SDR_M_TOL_2_MTOL(2, 0), 
				    PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
				    0, -1,
				    ipmi_sdr_max_by_form(IPMI_ANALOG_DATA_FORMAT_UNSIGNED),
				    ipmi_sdr_min_by_form(IPMI_ANALOG_DATA_FORMAT_UNSIGNED));
    return (ipmi_sdr_header_t*)sdr;
}

int pp_rpc_curr_sens_init(pp_rpc_curr_sens_t* this, pp_tp_obj_type_t type,
			  const char* id, pp_tp_obj_dtor_func_t dtor,
			  vector_t* sens)
{
    pp_tp_sensdev_t* curr_sens;
    curr_subsc_entry_t* subsc_entry;
    size_t i, sens_cnt= vector_size(sens);

    if(PP_FAILED(pp_tp_sensdev_init(&this->base, type, id,
				    dtor, curr_sens_default_sdr))) {
	pp_bmc_log_perror("%s() : initial fail", id);
        return PP_ERR;
    }
	
    // duplicate 
    this->sens = sens; /* sens are already duplicates */
    // subscribe	
    this->sens_subscribers = vector_new(NULL, sens_cnt, free);
    for (i = 0; i < sens_cnt; ++i) {
	this->curr[i]=0;
	curr_sens = (pp_tp_sensdev_t*)vector_get(this->sens, i);
	subsc_entry = calloc(1, sizeof(curr_subsc_entry_t));
	subsc_entry->obj = this;
	subsc_entry->sens = curr_sens;
	subsc_entry->index = i;
	subsc_entry->subscriber.recv_reading =rpc_curr_sens_recv_reading;
	vector_add(this->sens_subscribers, subsc_entry);   
	pp_bmc_tp_sensdev_subscribe(curr_sens, &subsc_entry->subscriber);  
    }  

    return PP_SUC;
}

static void
rpc_curr_sens_recv_reading(pp_tp_sensdev_subscriber_t* subsc,
			   pp_tp_sensdev_t* source UNUSED, int reading)
{
    unsigned int i, j;
    int sum=0, sum1=0;
    // find my index
    curr_subsc_entry_t* subsc_entry =
	PP_TP_INTF_2_OBJ_CAST(subsc, curr_subsc_entry_t, subscriber);
    pp_rpc_curr_sens_t* this = subsc_entry->obj;
    size_t sens_cnt= vector_size(this->sens);
	
    i = (int)subsc_entry->index;
    this->curr[i]=reading;
    for(j=0; j<sens_cnt; j++)
        sum += this->curr[j];    
    sum1 = sum /2;
    if(this->base.reading != sum1) {
	//pp_bmc_log_debug("RPC unit current =%d",sum);
	this->base.reading = sum1;
        pp_bmc_tp_sensdev_notify_subscribers(&this->base, sum1);
    }
}

void pp_rpc_curr_sens_dtor(pp_tp_obj_t* o)
{
    pp_tp_sensdev_t* curr_sens;
    pp_tp_sensdev_subscriber_t* subscriber;

    pp_rpc_curr_sens_t* this = (pp_rpc_curr_sens_t*)o;
    size_t i, sens_cnt = vector_size(this->sens);

    for (i = 0; i < sens_cnt; ++i) {
	curr_sens = (pp_tp_sensdev_t*)vector_get(this->sens, i);
	subscriber =
	    (pp_tp_sensdev_subscriber_t*)vector_get(this->sens_subscribers, i);
	pp_bmc_tp_sensdev_unsubscribe(curr_sens, subscriber);
    }
    vector_delete(this->sens_subscribers);
    vector_delete(this->sens);	   
   
    pp_tp_sensdev_cleanup(&this->base);
    free(this);
}

/*
 * Sensor Max current
 **************************/
pp_tp_obj_t* pp_rpc_max_curr_sens_ctor(const char* id, vector_t* args) {

    const char* errmsg = "[RPCPowerSens] c'tor: '%s' failed: %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_rpc_max_curr_sens_t* this ;
    pp_tp_sensdev_t* curr_sens = NULL;
	
    if (pp_tp_arg_scanf(args, 0, &err, "o<s>", &curr_sens)!=1) {
	pp_bmc_log_perror(errmsg, id, pp_strstream_buf(&err));
	return NULL;
    }

    this = malloc(sizeof(pp_rpc_max_curr_sens_t));
    if (PP_FAILED(pp_rpc_max_curr_sens_init(this, PP_TP_SENS_DEV, id,
					    rpc_max_curr_sens_dtor,
					    curr_sens))) {
	pp_bmc_log_error(errmsg, id, "init");
	free(this);
	this = NULL;
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

/* internal prototypes */
static void rpc_max_curr_sens_recv_reading(pp_tp_sensdev_subscriber_t* subscriber,
					   pp_tp_sensdev_t* source, int reading);

int pp_rpc_max_curr_sens_init(pp_rpc_max_curr_sens_t* this,
			      pp_tp_obj_type_t type,
			      const char *id, pp_tp_obj_dtor_func_t dtor,
			      pp_tp_sensdev_t* curr_sens) {
    if(PP_FAILED(pp_tp_sensdev_init(&this->base, type, id, dtor,
				    curr_sens_default_sdr))) {
	pp_bmc_log_perror("%s() : initial fail", id);
        return PP_ERR;
    }
    
    // duplicate 
    this->curr_sens = pp_tp_sensdev_duplicate(curr_sens);
    // subscribe	
    this->sens_subscr.recv_reading = rpc_max_curr_sens_recv_reading;
    pp_bmc_tp_sensdev_subscribe(this->curr_sens, &this->sens_subscr);	
    return PP_SUC;			    
}			    

void rpc_max_curr_sens_recv_reading(pp_tp_sensdev_subscriber_t* subsc,
				    pp_tp_sensdev_t* source UNUSED, int reading)
{	
    pp_rpc_max_curr_sens_t* this =
	PP_TP_INTF_2_OBJ_CAST(subsc, pp_rpc_max_curr_sens_t,
			      sens_subscr);	

    /* TODO use ACPI state information to check for fault only in S0 state */
    
    if((this->base.reading < (reading/2))) {
	
	this->base.reading = (reading/2);		 
	pp_bmc_tp_sensdev_notify_subscribers(&this->base, (reading/2));
    }
}

void rpc_max_curr_sens_dtor(pp_tp_obj_t* o) {
    pp_rpc_max_curr_sens_t* this = (pp_rpc_max_curr_sens_t*)o;
    if (this != NULL) {
	pp_tp_sensdev_release(this->curr_sens);
	pp_tp_sensdev_cleanup(&this->base);    	
    	free(this);
    }    
}

/*
 * Sensor total power
 **************************/
typedef struct pwr_subsc_entry_s {
    pp_rpc_pwr_sens_t* obj;
    pp_tp_sensdev_t* sens;
    pp_tp_sensdev_subscriber_t subscriber;
    size_t                   index;	
} pwr_subsc_entry_t;

pp_tp_obj_t* pp_rpc_pwr_sens_ctor(const char* id, vector_t* args) {
    const char* errmsg = "[RPCPwrSens] c'tor: '%s' failed: %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_rpc_pwr_sens_t* this = NULL;
    int i, arg_cnt;
    vector_t * sens = NULL;
	
    arg_cnt = vector_size(args);

    if (arg_cnt < 2) {
	pp_bmc_log_error(errmsg, id, "missing argument(s)");
	goto bail;
    } else if (arg_cnt > 24) {
	pp_bmc_log_error(errmsg, id, "too many arguments");
	goto bail;
    }
    sens = vector_new(NULL, arg_cnt,
		      (vector_elem_del_func_simple)pp_tp_obj_release);
	
    for (i = 0; i <arg_cnt; ++i) {
	pp_tp_sensdev_t * pwr_sens;
        if (pp_tp_arg_scanf(args, i, &err, "o<s>", &pwr_sens) != 1) {
	    pp_bmc_log_error(errmsg, id, pp_strstream_buf(&err));
	    vector_delete(sens);
            goto bail;
        }
	vector_add(sens, pp_tp_sensdev_duplicate(pwr_sens));
    }
    this = calloc(1, sizeof(pp_rpc_pwr_sens_t));
    if (PP_FAILED(pp_rpc_pwr_sens_init(this,
				       PP_TP_SENS_DEV, id,
				       pp_rpc_pwr_sens_dtor,
				       sens))) {
	pp_bmc_log_error(errmsg, id, "init");
	vector_delete(sens);
	free(this);
	this = NULL;
    }   

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;

}


/* internal prototypes */
static void rpc_pwr_sens_recv_reading(pp_tp_sensdev_subscriber_t* subsc,
				      pp_tp_sensdev_t* source, int reading);

static ipmi_sdr_header_t* pwr_sens_default_sdr(pp_tp_sensdev_t* s UNUSED) {
    ipmi_sdr_full_sensor_t* sdr;
	
    sdr = ipmi_sdr_full_part_create(IPMI_SENSOR_TYPE_POWER_UNIT,
				    IPMI_EVENT_READING_TYPE_THRESHOLD,
				    IPMI_UNIT_TYPE_WATTS,
				    IPMI_ANALOG_DATA_FORMAT_UNSIGNED,
				    PP_BMC_SDR_M_TOL_2_MTOL(30, 0), 
				    PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0),
				    0, 0,
		      ipmi_sdr_max_by_form(IPMI_ANALOG_DATA_FORMAT_UNSIGNED),
		      ipmi_sdr_min_by_form(IPMI_ANALOG_DATA_FORMAT_UNSIGNED));
    return (ipmi_sdr_header_t*)sdr;
}

int pp_rpc_pwr_sens_init(pp_rpc_pwr_sens_t* this, pp_tp_obj_type_t type,
			 const char* id, pp_tp_obj_dtor_func_t dtor,
			 vector_t* sens)
{
    pp_tp_sensdev_t* pwr_sens;
    pwr_subsc_entry_t* subsc_entry;
    size_t i, sens_cnt= vector_size(sens);

    if(PP_FAILED(pp_tp_sensdev_init(&this->base, type, id,
				    dtor, pwr_sens_default_sdr))) {
	pp_bmc_log_perror("%s() : initial fail", id);
        return PP_ERR;
    }
	
    // duplicate 
    this->sens = sens; /* sens are already duplicates */
    // subscribe	
    this->sens_subscribers = vector_new(NULL, sens_cnt, free);
    for (i = 0; i < sens_cnt; ++i) {
	this->pwr[i]=0;
	pwr_sens = (pp_tp_sensdev_t*)vector_get(this->sens, i);
	subsc_entry = calloc(1, sizeof(pwr_subsc_entry_t));
	subsc_entry->obj = this;
	subsc_entry->sens = pwr_sens;
	subsc_entry->index = i;
	subsc_entry->subscriber.recv_reading =rpc_pwr_sens_recv_reading;
	vector_add(this->sens_subscribers, subsc_entry);   
	pp_bmc_tp_sensdev_subscribe(pwr_sens, &subsc_entry->subscriber);  
    }  

    return PP_SUC;
}

static void
rpc_pwr_sens_recv_reading(pp_tp_sensdev_subscriber_t* subsc,
			  pp_tp_sensdev_t* source UNUSED, int reading)
{
    unsigned int i, j;
    int sum=0, sum1=0;
    // find my index
    pwr_subsc_entry_t* subsc_entry =
	PP_TP_INTF_2_OBJ_CAST(subsc, pwr_subsc_entry_t, subscriber);
    pp_rpc_pwr_sens_t* this = subsc_entry->obj;
    size_t sens_cnt= vector_size(this->sens);
	
    i = (int)subsc_entry->index;
    this->pwr[i]=reading;
    for(j=0; j<sens_cnt; j++)
        sum += this->pwr[j];    
    sum1 = sum /30;
    if(this->base.reading != sum1) {
	this->base.reading = sum1;
        pp_bmc_tp_sensdev_notify_subscribers(&this->base, sum1);
    }
}

void pp_rpc_pwr_sens_dtor(pp_tp_obj_t* o)
{
    pp_tp_sensdev_t* pwr_sens;
    pp_tp_sensdev_subscriber_t* subscriber;

    pp_rpc_pwr_sens_t* this = (pp_rpc_pwr_sens_t*)o;
    size_t i, sens_cnt = vector_size(this->sens);

    for (i = 0; i < sens_cnt; ++i) {
	pwr_sens = (pp_tp_sensdev_t*)vector_get(this->sens, i);
	subscriber =
	    (pp_tp_sensdev_subscriber_t*)vector_get(this->sens_subscribers, i);
	pp_bmc_tp_sensdev_unsubscribe(pwr_sens, subscriber);
    }
    vector_delete(this->sens_subscribers);
    vector_delete(this->sens);	   
   
    pp_tp_sensdev_cleanup(&this->base);
    free(this);
}	
/*
 * Sensor Apparent powr
 ***************************/
pp_tp_obj_t* pp_rpc_apt_pwr_sens_ctor(const char* id, vector_t* args) {
    const char* errmsg = "[RPCPowerSens] c'tor: '%s' failed %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_rpc_apt_pwr_sens_t* this ;
    pp_tp_sensdev_t* volt_sens = NULL;
    pp_tp_sensdev_t* cur_sens = NULL;
    if (pp_tp_arg_scanf(args, 0, &err, "o<s>o<s>",
			&volt_sens, &cur_sens)!=2) {
	pp_bmc_log_perror(errmsg, id, pp_strstream_buf(&err));
	return NULL;
    }

    this = malloc(sizeof(pp_rpc_apt_pwr_sens_t));
    if (PP_FAILED(pp_rpc_apt_pwr_sens_init(this, PP_TP_SENS_DEV, id,
					   rpc_apt_pwr_sens_dtor,
					   volt_sens, cur_sens))) {
	pp_bmc_log_error(errmsg, id, "init");
	free(this);
	this = NULL;
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}
/* internal prototypes */
static void rpc_apt_pwr_sens_recv_reading(pp_tp_sensdev_subscriber_t* subscriber,
					  pp_tp_sensdev_t* source, int reading);
int pp_rpc_apt_pwr_sens_init(pp_rpc_apt_pwr_sens_t* this,
			     pp_tp_obj_type_t type,
			     const char *id, pp_tp_obj_dtor_func_t dtor,
			     pp_tp_sensdev_t* volt_sens,
			     pp_tp_sensdev_t* cur_sens)			    
{
    if(PP_FAILED(pp_tp_sensdev_init(&this->base, type, id, dtor,
				    pwr_sens_default_sdr))) {
	pp_bmc_log_perror("%s() : initial fail", id);
        return PP_ERR;
    }
    
    // duplicate 
    this->volt_sens = pp_tp_sensdev_duplicate(volt_sens);
    this->cur_sens = pp_tp_sensdev_duplicate(cur_sens);	

    // subscribe	
    this->sens_subscr.recv_reading = rpc_apt_pwr_sens_recv_reading;
    pp_bmc_tp_sensdev_subscribe(this->volt_sens, &this->sens_subscr);	
    pp_bmc_tp_sensdev_subscribe(this->cur_sens, &this->sens_subscr);
    return PP_SUC;			    
}	

void rpc_apt_pwr_sens_recv_reading(pp_tp_sensdev_subscriber_t* subsc,
				   pp_tp_sensdev_t* source UNUSED, int reading UNUSED)
{
    int read_volt=0, read_cur=0;
    int new_reading = 0;
	
    pp_rpc_apt_pwr_sens_t* this =
	PP_TP_INTF_2_OBJ_CAST(subsc, pp_rpc_apt_pwr_sens_t,
			      sens_subscr);	
    read_volt = pp_tp_sensdev_get_reading(this->volt_sens);
    read_cur = pp_tp_sensdev_get_reading(this->cur_sens);	
    new_reading = (read_volt * read_cur)/100;
    if((this->base.reading != new_reading)) {
	/*   pp_bmc_log_debug("%s : %d x %d=%d",
	     (pp_tp_obj_t*)this->base.base.id, 
	     read_volt, read_cur, new_reading); */
	this->base.reading = new_reading;		 
	pp_bmc_tp_sensdev_notify_subscribers(&this->base, new_reading);
    }
}

void rpc_apt_pwr_sens_dtor(pp_tp_obj_t* o) {
    pp_rpc_apt_pwr_sens_t* this = (pp_rpc_apt_pwr_sens_t*)o;
    if (this != NULL) {
	pp_tp_sensdev_release(this->volt_sens);
	pp_tp_sensdev_release(this->cur_sens);	
	pp_tp_sensdev_cleanup(&this->base);     	
    	free(this);
    }    
}

/*
 * Sensor Real RMS powr
 **************************/
pp_tp_obj_t* pp_rpc_rms_pwr_sens_ctor(const char* id, vector_t* args) {

    const char* errmsg = "[RPCPowerSens] c'tor: '%s' failed %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_rpc_rms_pwr_sens_t* this ;
    pp_tp_sensdev_t* pwr_sens = NULL;
    pp_tp_sensdev_t* phase_sens = NULL;
    int period=0;
	
    if (pp_tp_arg_scanf(args, 0, &err, "o<s>o<s>d",
			&pwr_sens, &phase_sens, &period)< 2) {
	pp_bmc_log_perror(errmsg, id, pp_strstream_buf(&err));
	return NULL;
    }

    this = malloc(sizeof(pp_rpc_rms_pwr_sens_t));
    if (PP_FAILED(pp_rpc_rms_pwr_sens_init(this, PP_TP_SENS_DEV, id,
					   rpc_rms_pwr_sens_dtor,
					   pwr_sens, phase_sens, period))) {
	pp_bmc_log_error(errmsg, id, "init");
	free(this);
	this = NULL;
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

/* internal prototypes */
static void rpc_rms_pwr_sens_recv_reading(pp_tp_sensdev_subscriber_t* subscriber,
					  pp_tp_sensdev_t* source, int reading);

int pp_rpc_rms_pwr_sens_init(pp_rpc_rms_pwr_sens_t* this,
			     pp_tp_obj_type_t type,
			     const char *id, pp_tp_obj_dtor_func_t dtor,
			     pp_tp_sensdev_t* pwr_sens,
			     pp_tp_sensdev_t* phase_sens,
			     int period)			    
{
    if(PP_FAILED(pp_tp_sensdev_init(&this->base, type, id, dtor,
				    pwr_sens_default_sdr))) {
	pp_bmc_log_perror("%s() : initial fail", id);
        return PP_ERR;
    }
    int i;
    for(i=0;i<20;i++)
        this->avrgpwr[i] = 0;

    this->period = period;	
    // duplicate 
    this->pwr_sens = pp_tp_sensdev_duplicate(pwr_sens);
    this->phase_sens = pp_tp_sensdev_duplicate(phase_sens);
    // subscribe	
    this->sens_subscr.recv_reading = rpc_rms_pwr_sens_recv_reading;
    pp_bmc_tp_sensdev_subscribe(this->pwr_sens, &this->sens_subscr);	
    pp_bmc_tp_sensdev_subscribe(this->phase_sens, &this->sens_subscr);	
    return PP_SUC;			    
}			    

void rpc_rms_pwr_sens_recv_reading(pp_tp_sensdev_subscriber_t* subsc,
				   pp_tp_sensdev_t* source UNUSED, int reading UNUSED)
{
    int read_pwr=0, read_phase=0, i;
    int new_reading = 0, sum=0;
	
    pp_rpc_rms_pwr_sens_t* this =
	PP_TP_INTF_2_OBJ_CAST(subsc, pp_rpc_rms_pwr_sens_t,
			      sens_subscr);	

    /* TODO use ACPI state information to check for fault only in S0 state */

    read_pwr = pp_tp_sensdev_get_reading(this->pwr_sens);
    read_phase = pp_tp_sensdev_get_reading(this->phase_sens);
    new_reading = (read_pwr * cos(read_phase));

    if((this->period & 0xff) != 0) {
        this->avrgpwr[((this->period&0xff00)>>8)/3] = new_reading;
        this->period += 0x300;
	if(((this->period&0xff00)>>8) == (this->period&0xff))
            this->period &=0xff;
        for(i=0;i<20;i++)
	    sum += this->avrgpwr[i];
        sum = sum/20;
    } else 
        sum = new_reading;    
    
    if((this->base.reading != sum)) {
	/*   pp_bmc_log_debug("%s : %d*%d*cos(%d)=%d",
	     (pp_tp_obj_t*)this->base.base.id, 
	     read_volt, read_cur, read_phase, sum); */
	this->base.reading = sum;		 
	pp_bmc_tp_sensdev_notify_subscribers(&this->base, sum);
    }
}

void rpc_rms_pwr_sens_dtor(pp_tp_obj_t* o) {
    pp_rpc_rms_pwr_sens_t* this = (pp_rpc_rms_pwr_sens_t*)o;
    if (this != NULL) {
	pp_tp_sensdev_release(this->pwr_sens);
	pp_tp_sensdev_release(this->phase_sens);	
	pp_tp_sensdev_cleanup(&this->base);    	
    	free(this);
    }    
}

/*
 * Sensor Max RMS powr
 **************************/
pp_tp_obj_t* pp_rpc_max_pwr_sens_ctor(const char* id, vector_t* args) {

    const char* errmsg = "[RPCPowerSens] c'tor: '%s' failed: %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_rpc_max_pwr_sens_t* this ;
    pp_tp_sensdev_t* pwr_sens = NULL;
	
    if (pp_tp_arg_scanf(args, 0, &err, "o<s>", &pwr_sens)!=1) {
	pp_bmc_log_perror(errmsg, id, pp_strstream_buf(&err));
	return NULL;
    }

    this = malloc(sizeof(pp_rpc_max_pwr_sens_t));
    if (PP_FAILED(pp_rpc_max_pwr_sens_init(this, PP_TP_SENS_DEV, id,
					   rpc_max_pwr_sens_dtor, pwr_sens))) {
	pp_bmc_log_error(errmsg, id, "init");
	free(this);
	this = NULL;
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

/* internal prototypes */
static void rpc_max_pwr_sens_recv_reading(pp_tp_sensdev_subscriber_t* subscriber,
					  pp_tp_sensdev_t* source, int reading);

int pp_rpc_max_pwr_sens_init(pp_rpc_max_pwr_sens_t* this,
			     pp_tp_obj_type_t type,
			     const char *id, pp_tp_obj_dtor_func_t dtor,
			     pp_tp_sensdev_t* pwr_sens)			    
{
    if(PP_FAILED(pp_tp_sensdev_init(&this->base, type, id, dtor,
				    pwr_sens_default_sdr))) {
	pp_bmc_log_perror("%s() : initial fail", id);
        return PP_ERR;
    }
    
    // duplicate 
    this->pwr_sens = pp_tp_sensdev_duplicate(pwr_sens);
    // subscribe	
    this->sens_subscr.recv_reading = rpc_max_pwr_sens_recv_reading;
    pp_bmc_tp_sensdev_subscribe(this->pwr_sens, &this->sens_subscr);	
    return PP_SUC;			    
}			    

void rpc_max_pwr_sens_recv_reading(pp_tp_sensdev_subscriber_t* subsc,
				   pp_tp_sensdev_t* source UNUSED, int reading)
{	
    pp_rpc_max_pwr_sens_t* this =
	PP_TP_INTF_2_OBJ_CAST(subsc, pp_rpc_max_pwr_sens_t,
			      sens_subscr);	

    /* TODO use ACPI state information to check for fault only in S0 state */
    
    if((this->base.reading < reading)) {
	
	this->base.reading = reading;		 
	pp_bmc_tp_sensdev_notify_subscribers(&this->base, reading);
    }
}

void rpc_max_pwr_sens_dtor(pp_tp_obj_t* o) {
    pp_rpc_max_pwr_sens_t* this = (pp_rpc_max_pwr_sens_t*)o;
    if (this != NULL) {
	pp_tp_sensdev_release(this->pwr_sens);
	pp_tp_sensdev_cleanup(&this->base);    	
    	free(this);
    }    
}

