/*
 * tmp101.c
 *
 * TMP101 temperature sensor
 * 
 * (c) 2006 Peppercon AG, 2006/10/25 tbr@raritan.com
 *
 * Implementation note:
 * ---------------------
 * Currently we are happy with all the power-on defaults in TMP101,
 * so the sensor is directly implemented via i2c-comdev. If you do need
 * initialization of the chip, i.e. in order to increase resolution,
 * do implement according chip object and add initialization code there!!!
 */

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_comdev.h>
#include <pp/bmc/tp_scan_sensdev.h>

#include "drivers/tmp101.h"


/*
 * private structures
 ************************/

#define TMP101_REG_TEMP   0
#define TMP101_REG_CFG    1
#define TMP101_REG_TLOW   2
#define TMP101_REG_THIGH  3

typedef struct tmp101_sens_s {
    pp_tp_scan_sensdev_t base;
    pp_tp_i2c_comdev_t* i2cdev;
    unsigned char i2caddr;
} tmp101_sens_t;

/*
 * private function prototypes
 ***************************************/

static ipmi_sdr_header_t* tmp101_default_sdr(pp_tp_sensdev_t* s);
static void tmp101_update_temp_reading(pp_sensor_scannable_t* s);
static void tmp101_dtor(tmp101_sens_t* sensor);


/**
 * TMP101 constructor
 */
pp_tp_obj_t* pp_tmp101_sens_ctor(const char *id, vector_t *args) {
    const char* fn = __FUNCTION__;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    tmp101_sens_t* this = NULL;
    pp_tp_i2c_comdev_t* i2cdev;
    int i2caddr;
    pp_tp_cond_t* scond;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>d|o<sc>",
			&i2cdev, &i2caddr, &scond) != 3) {
	pp_bmc_log_error("%s: '%s' failed: %s (%s)",
		fn, id, strerror(errno), pp_strstream_buf(&err));
    } else {
	this = malloc(sizeof(tmp101_sens_t));
	pp_tp_scan_sensdev_init(&this->base, PP_TP_SCAN_SENS_DEV, id,
			    (void(*)(pp_tp_obj_t*))tmp101_dtor,
			    tmp101_default_sdr,
			    tmp101_update_temp_reading, scond);
	this->i2cdev = pp_tp_i2c_comdev_duplicate(i2cdev);
	this->i2caddr = i2caddr;
	pp_tp_scan_sensdev_register(&this->base, 0);
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}


/*
 * private functions
 ********************************************************************/

static void tmp101_dtor(tmp101_sens_t* sensor) {
    pp_tp_scan_sensdev_unregister(&sensor->base);
    pp_tp_i2c_comdev_release(sensor->i2cdev);
    pp_tp_scan_sensdev_cleanup(&sensor->base);
    free(sensor);
}

static ipmi_sdr_header_t* tmp101_default_sdr(pp_tp_sensdev_t* s UNUSED) {
    ipmi_sdr_full_sensor_t* sdr;
    sdr = ipmi_sdr_full_part_create(IPMI_SENSOR_TYPE_TEMPERATURE,
				    IPMI_EVENT_READING_TYPE_THRESHOLD,
				    IPMI_UNIT_TYPE_DEGREES_C,
				    IPMI_ANALOG_DATA_FORMAT_2_COMPL,
				    PP_BMC_SDR_M_TOL_2_MTOL(1, 0), 
				    PP_BMC_SDR_B_ACCURACY_2_BACC(0, 0), 
                                    0, 0,
		   ipmi_sdr_max_by_form(IPMI_ANALOG_DATA_FORMAT_2_COMPL),
   		   ipmi_sdr_min_by_form(IPMI_ANALOG_DATA_FORMAT_2_COMPL));
    return (ipmi_sdr_header_t*)sdr;
}

/*
  return: ret holds a 8-bit signed value. If the 32-bit ret value is
  negative, this indicates an error -- just guessing from what tbr told me.
*/
static void tmp101_update_temp_reading(pp_sensor_scannable_t* s) {
    tmp101_sens_t* this = PP_TP_SCANNABLE_2_OBJ_CAST(s, tmp101_sens_t);
    pp_tp_i2c_comdev_t* i2cdev = this->i2cdev;
    unsigned char i2caddr = this->i2caddr;
    int ret = -2;
    unsigned char value;

    if (pp_tp_scan_sensdev_do_scan(&this->base)) {
	if (PP_SUC == (ret = pp_tp_i2c_comdev_pre_com(i2cdev, i2caddr))) {
	    ret = pp_tp_i2c_comdev_rx_byte_data(i2cdev, i2caddr,
						TMP101_REG_TEMP, &value);
	    pp_tp_i2c_comdev_post_com(i2cdev, i2caddr);
	    if (ret == PP_SUC) {
		ret = value;
	    } else {
		errno = ENODATA;
		ret = PP_ERR;
	    }
	} else errno = ENODATA;
    }
    pp_tp_scan_sensdev_update_reading(&this->base, ret);
}

