/**
 * w83791d_pwm.c
 *
 * implements W83791D pwm device
 * 
 * (c) 2005 Peppercon AG, 2005/11/08, thomas@peppercon.de
 */

#include <malloc.h>

#include <pp/base.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_chip.h>
#include <pp/bmc/tp_scan_sensdev.h>
#include <pp/bmc/tp_pwm_act.h>
#include <pp/bmc/tp_gpio_dev.h>

#include "drivers/w83791d.h"
#include "drivers/w83791d_pwm.h"

/*
 * pwms
 **************************/

typedef struct w83791d_pwm_cfg_s {
    unsigned char duty_cycle_reg;
} w83791d_pwm_cfg_t;

typedef struct w83791d_pwm_s {
    pp_tp_pwm_act_t base;
    pp_tp_i2c_chip_t* chip;
    w83791d_pwm_cfg_t* cfg;
} w83791d_pwm_t;

w83791d_pwm_cfg_t w83791d_pwm_cfg[W83791D_PWM_MAX_NUMBER] = {
    /* 0 */ { W83791D_PWM_DUTY_CYCLE_REG_FAN_1 },
    /* 1 */ { W83791D_PWM_DUTY_CYCLE_REG_FAN_2 },
    /* 2 */ { W83791D_PWM_DUTY_CYCLE_REG_FAN_3 },
    /* 3 */ { W83791D_PWM_DUTY_CYCLE_REG_FAN_4 },
    /* 4 */ { W83791D_PWM_DUTY_CYCLE_REG_FAN_5 },
};

static void w83791d_pwm_dtor(pp_tp_obj_t* o);
static int w83791d_pwm_set_mode(pp_tp_pwm_act_t* o, pp_tp_pwm_mode_t mode);
static int w83791d_pwm_get_mode(pp_tp_pwm_act_t* o, pp_tp_pwm_mode_t*);
static int w83791d_pwm_set_duty_cycle(pp_tp_pwm_act_t* o, unsigned char );
static int w83791d_pwm_get_duty_cycle(pp_tp_pwm_act_t* o, unsigned char* );


pp_tp_obj_t* pp_sensor_w83791d_pwm_ctor(const char* id, vector_t* args) {
    const char* errmsg = "%s(): '%s' failed: %s";
    w83791d_pwm_t* this = NULL;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_i2c_chip_t* chip;
    unsigned int pwm_no;

    if (pp_tp_arg_scanf(args, 0, &err, "o<h>d", &chip, &pwm_no) != 2) {
	pp_bmc_log_perror(errmsg, ___F, id, pp_strstream_buf(&err));
    } else if (strcmp(chip->model, W83791D_MODEL_STRING)) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "incorrect chip model");
    } else if (pwm_no > W83791D_PWM_MAX_NUMBER) {
	errno = EINVAL;
	pp_bmc_log_perror(errmsg, ___F, id, "invalid pwm number");
    } else {
	this = malloc(sizeof(w83791d_pwm_t));
	pp_tp_pwm_act_init(&this->base, PP_TP_PWM_ACT, id, w83791d_pwm_dtor,
			   w83791d_pwm_set_mode,
			   w83791d_pwm_get_mode,
			   w83791d_pwm_set_duty_cycle,
			   w83791d_pwm_get_duty_cycle);
	this->chip    = pp_tp_i2c_chip_duplicate(chip);
	this->cfg     = &w83791d_pwm_cfg[pwm_no];
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

static void w83791d_pwm_dtor(pp_tp_obj_t* o) {
    w83791d_pwm_t* this = (w83791d_pwm_t*)o;
    assert(this != NULL);
    pp_tp_i2c_chip_release(this->chip);
    pp_tp_pwm_act_cleanup(&this->base);
    free(this);
}

/* set and get mode not supported for this chip, currently */
static int w83791d_pwm_set_mode(pp_tp_pwm_act_t* o UNUSED,
				pp_tp_pwm_mode_t mode UNUSED) {
    errno = EINVAL;
    return PP_ERR;
}

static int w83791d_pwm_get_mode(pp_tp_pwm_act_t* o UNUSED,
				pp_tp_pwm_mode_t* mode_ptr) {
    *mode_ptr = PP_TP_PWM_MANUAL;
    return PP_SUC;
}

/*
 * Note:
 * - lower nibble of Fan register codes duty cylce
 * - higher niblle codes max speed, will be set to max
 */
static int w83791d_pwm_set_duty_cycle(pp_tp_pwm_act_t* o,
				      unsigned char duty_cycle) {
    w83791d_pwm_t* this = (w83791d_pwm_t*)o;
    pp_tp_i2c_chip_t* chip = this->chip;
    w83791d_pwm_cfg_t* cfg = this->cfg;
    int ret;

    if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(chip))) {
	ret = pp_tp_i2c_chip_tx_byte_data(chip, cfg->duty_cycle_reg,
					  255 * duty_cycle / 100);
	pp_tp_i2c_chip_post_com(chip);
    }
    return ret;
}

static int w83791d_pwm_get_duty_cycle(pp_tp_pwm_act_t* o,
				      unsigned char* duty_cycle_ptr) {
    w83791d_pwm_t* this = (w83791d_pwm_t*)o;
    pp_tp_i2c_chip_t* chip = this->chip;
    w83791d_pwm_cfg_t* cfg = this->cfg;
    unsigned char duty_cycle;
    int ret;
    
    if (PP_SUC == (ret = pp_tp_i2c_chip_pre_com(chip))) {
	ret = pp_tp_i2c_chip_rx_byte_data(chip, cfg->duty_cycle_reg,
					  &duty_cycle);
	*duty_cycle_ptr = 100 * duty_cycle / 255;
	pp_tp_i2c_chip_post_com(chip);
    }
    return ret;
}

