/**
 * warthog_fan_ctrl.c
 *
 * An object controlling the fan speed on the AMD Warthog board.
 * 
 * (c) 2005 Peppercon AG, Ronald Wahl <rwa@peppercon.de>
 */

#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_pwm_act.h>
#include <pp/bmc/tp_ipmi_sens.h>
#include "drivers/warthog_fan_ctrl.h"

static void warthog_fan_ctrl_temp_recv_reading(pp_tp_ipmi_sens_subscriber_t*, pp_tp_ipmi_sens_t*, int);
static void warthog_fan_ctrl_fan_recv_reading(pp_tp_ipmi_sens_subscriber_t*, pp_tp_ipmi_sens_t*, int);
static void warthog_fan_ctrl_adjust_fans(pp_warthog_fan_ctrl_t* this);

pp_tp_obj_t*
pp_warthog_fan_ctrl_ctor(const char* id, vector_t* args)
{
    const char* errmsg = "[WarthogFanCtrl] %s: '%s' failed (%s)";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;  
    pp_warthog_fan_ctrl_t* instance = NULL;
    vector_t * pwm_actors = NULL;
    vector_t * temp_sensors = NULL;
    vector_t * fan_sensors = NULL;    
    int pwm_count, fan_count, temp_count;
    int i = 0;

    /* get counts */
    if (pp_tp_arg_scanf(args, i, &err, "ddd", &pwm_count, &temp_count,
			&fan_count) != 3) {
	pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
	goto bail;
    }

    /* initialize vectors */
    pwm_actors = vector_new(NULL, pwm_count,
			    (vector_elem_del_func_simple)pp_tp_obj_release);
    temp_sensors = vector_new(NULL, fan_count,
			      (vector_elem_del_func_simple)pp_tp_obj_release);
    fan_sensors = vector_new(NULL, temp_count,
			     (vector_elem_del_func_simple)pp_tp_obj_release);

    for (i = 0; i < pwm_count; ++i) {
	pp_tp_pwm_act_t * pwm_act;
        if (pp_tp_arg_scanf(args, 3 + i, &err, "o<ap>", &pwm_act) != 1) {
	    pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
            goto bail;
        }
	vector_add(pwm_actors, pp_tp_pwm_act_duplicate(pwm_act));
    }

    for (i = 0; i < temp_count; ++i) {
	pp_tp_ipmi_sens_t * ipmi_sens;
        if (pp_tp_arg_scanf(args, 3 + pwm_count + i, &err, "o<p>", &ipmi_sens)
	    != 1) {
	    pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
            goto bail;
        }
	vector_add(temp_sensors, pp_tp_ipmi_sens_duplicate(ipmi_sens));
    }

    for (i = 0; i < fan_count; ++i) {
	pp_tp_ipmi_sens_t * ipmi_sens;
        if (pp_tp_arg_scanf(args, 3 + pwm_count + temp_count + i, &err, "o<p>",
			    &ipmi_sens) != 1) {
	    pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
            goto bail;
        }
	vector_add(fan_sensors, pp_tp_ipmi_sens_duplicate(ipmi_sens));
    }

    instance = calloc(1, sizeof(pp_warthog_fan_ctrl_t));
    if (PP_FAILED(pp_warthog_fan_ctrl_init(instance, PP_TP_FAN_CTRL, id,
					   pp_warthog_fan_ctrl_dtor,
					   pwm_actors, temp_sensors,
					   fan_sensors))) {
	pp_bmc_log_error(errmsg, ___F, id, "init");
	free(instance);
	instance = NULL;
    }

 bail:
    pp_strstream_free(&err);
    if (instance == NULL) {
	vector_delete(temp_sensors);
	vector_delete(fan_sensors);
    }
    return (pp_tp_obj_t*)instance;
}

void
pp_warthog_fan_ctrl_dtor(pp_tp_obj_t* this)
{
    pp_warthog_fan_ctrl_cleanup((pp_warthog_fan_ctrl_t*)this);
    free(this);
}

int
pp_warthog_fan_ctrl_init(pp_warthog_fan_ctrl_t* this, pp_tp_obj_type_t type,
			 const char* id, pp_tp_obj_dtor_func_t dtor,
			 vector_t * pwm_actors, vector_t * temp_sensors,
			 vector_t * fan_sensors) {
    pp_tp_ipmi_sens_t* ipmi_sens;
    
    if (PP_SUCCED(pp_tp_ctrl_init(&this->base, type, id, dtor))) {
	u_int temp_sensor_cnt = vector_size(temp_sensors);
	u_int fan_sensor_cnt = vector_size(fan_sensors);
	u_int i;
	this->warn_cnt = this->old_warn_cnt = 0;
	this->crit_cnt = this->old_crit_cnt = 0;
	this->pwm_actors = pwm_actors;
	this->temp_sensors = temp_sensors;
	this->fan_sensors = fan_sensors;
	this->temp_subscriber.recv_reading =warthog_fan_ctrl_temp_recv_reading;
	this->fan_subscriber.recv_reading = warthog_fan_ctrl_fan_recv_reading;
	for (i = 0; i < temp_sensor_cnt; ++i) {
	    ipmi_sens = (pp_tp_ipmi_sens_t *)vector_get(temp_sensors, i);
	    pp_tp_ipmi_sens_subscribe(ipmi_sens, &this->temp_subscriber);
	}
	for (i = 0; i < fan_sensor_cnt; ++i) {
	    ipmi_sens = (pp_tp_ipmi_sens_t *)vector_get(fan_sensors, i);
	    pp_tp_ipmi_sens_subscribe(ipmi_sens, &this->fan_subscriber);
	}
	return PP_SUC;
    }
    return PP_ERR;
}

void
pp_warthog_fan_ctrl_cleanup(pp_warthog_fan_ctrl_t* this)
{
    u_int temp_sensor_cnt = vector_size(this->temp_sensors);
    u_int fan_sensor_cnt = vector_size(this->fan_sensors);
    pp_tp_ipmi_sens_t * ipmi_sens;
    u_int i;
    for (i = 0; i < temp_sensor_cnt; ++i) {
	ipmi_sens = (pp_tp_ipmi_sens_t *)vector_get(this->temp_sensors, i);
	pp_tp_ipmi_sens_unsubscribe(ipmi_sens, &this->temp_subscriber);
    }
    for (i = 0; i < fan_sensor_cnt; ++i) {
	ipmi_sens = (pp_tp_ipmi_sens_t *)vector_get(this->fan_sensors, i);
	pp_tp_ipmi_sens_unsubscribe(ipmi_sens, &this->fan_subscriber);
    }
    vector_delete(this->pwm_actors);
    vector_delete(this->temp_sensors);
    vector_delete(this->fan_sensors);
    pp_tp_ctrl_cleanup(&this->base);
}

static void warthog_fan_ctrl_temp_recv_reading(pp_tp_ipmi_sens_subscriber_t* subs,
					       pp_tp_ipmi_sens_t* source UNUSED, int reading)
{
    pp_warthog_fan_ctrl_t* this =
	PP_TP_INTF_2_OBJ_CAST(subs, pp_warthog_fan_ctrl_t, temp_subscriber);

    this->old_warn_cnt = this->warn_cnt;
    this->old_crit_cnt = this->crit_cnt;
    
    if (reading & IPMI_SENSOR_EVENT_UPPER_NON_CRIT_GOING_HI) {
	this->warn_cnt += reading & 0x80000000 ? -1 : 1;
    } else if (reading & IPMI_SENSOR_EVENT_UPPER_CRIT_GOING_HI) {
	this->crit_cnt += reading & 0x80000000 ? -1 : 1;
    }
    warthog_fan_ctrl_adjust_fans(this);
}

static void warthog_fan_ctrl_fan_recv_reading(pp_tp_ipmi_sens_subscriber_t* subs,
					      pp_tp_ipmi_sens_t* source UNUSED, int reading)
{
    pp_warthog_fan_ctrl_t* this =
	PP_TP_INTF_2_OBJ_CAST(subs, pp_warthog_fan_ctrl_t, fan_subscriber);

    this->old_warn_cnt = this->warn_cnt;
    this->old_crit_cnt = this->crit_cnt;
    
    if (reading & IPMI_SENSOR_EVENT_LOWER_NON_CRIT_GOING_LO) {
	this->warn_cnt += reading & 0x80000000 ? -1 : 1;
    } else if (reading & IPMI_SENSOR_EVENT_LOWER_CRIT_GOING_LO) {
	this->crit_cnt += reading & 0x80000000 ? -1 : 1;
    }
    warthog_fan_ctrl_adjust_fans(this);
}

static void warthog_fan_ctrl_adjust_fans(pp_warthog_fan_ctrl_t* this)
{
#if PP_BMC_DEBUG_LEVEL>=7
    const char* msg = "[WarthogFanCtrl] %s";
#endif
    int duty_cycle = -1;
    static int initialized = 0;
   
    if (this->warn_cnt < 0) {
	this->warn_cnt = 0;
    }
    if (this->crit_cnt < 0) {
	this->crit_cnt = 0;
    }

    if (this->crit_cnt > 0) {
	if (this->old_crit_cnt == 0) {
	    pp_bmc_log_info(msg, "setting fans to high speed");
	    duty_cycle = 99;
	}
    } else if (this->warn_cnt > 0) {
	if (this->old_warn_cnt == 0 || this->old_crit_cnt > 0) {
	    pp_bmc_log_info(msg, "setting fans to medium speed");
	    duty_cycle = 76;
	}
    } else {
	if (!initialized || this->old_crit_cnt > 0 || this->old_warn_cnt > 0) {
	    pp_bmc_log_info(msg, "setting fans to low speed");
	    duty_cycle = 57;
	}
    }
    if (duty_cycle >= 0) {
	u_int pwm_actor_cnt = vector_size(this->pwm_actors);
	u_int i;
	for (i = 0; i < pwm_actor_cnt; ++i) {
	    pp_tp_pwm_act_t * pwm_act =
		(pp_tp_pwm_act_t *)vector_get(this->pwm_actors, i);
	    pp_tp_pwm_set_duty_cycle(pwm_act, duty_cycle);
	}
	initialized = 1;
    }
}
