/**
 * tp_gpio_sens.h
 *
 * A sensor for one concrete gpio input signal
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

 
#ifndef _PP_BMC_TP_GPIO_SENS_H
#define _PP_BMC_TP_GPIO_SENS_H

#include <pp/bmc/tp_gpio_dev.h>
#include <pp/bmc/tp_sensdev.h>
#include <pp/bmc/topo_factory.h>

typedef struct pp_tp_gpio_sens_s pp_tp_gpio_sens_t;

/**
 * concrete gpio signal input sensor
 */
struct pp_tp_gpio_sens_s {
    pp_tp_sensdev_t base;
    
    /* debounce time in milliseconds, 0 means 'disabled' */
    int debounce;

    /* select handler for debouncing */
    int debounce_hdl;

    /* when debouncing, save raw value to be used after debounce time period */
    int value_pending;
	  
    /* the number of the gpio signal this object represents */
    unsigned char gpio_no;
    
    /* whether this signal is inverted or not */
    unsigned char invert;
   
    /* the gpio this object belongs to */
    pp_tp_gpio_dev_t* gpio_dev;
    
    /* gpio_subscriber interface */
    pp_tp_gpio_subscriber_t gpio_subscriber_interface;
};

/* ppdoc
 * name:   GPIO sensor
 * tpname: GpioSens
 * desc:   A single GPIO signal abstraction. Should operate on logical levels.
 * desc:   Can be inverted and debounced.
 * arg 0:  [OBJ(GPIO_DEV)] gpiodev: Gpio device for this signal
 * arg 1:  [INT] signal_no:         the gpio signal number within the gpio device object
 * arg 2:  [BOOL] invert:           passing true will invert the signal reading
 * arg 3:  [INT] debounce:          debounce time (in ms, 0 to disable)
 * arg 4:  [INT] stype:             (optional) IPMI sensor type
 * arg 4:                           (see IPMI spec tab 42-3, default is 'button')
 * arg 5:  [INT] rtype:             (optional) IPMI reading type
 * arg 5:                           (see IPMI spec tab 42-2, default is 'discrete state')
 * return: [OBJ(GPIO_SENS)]
 */
pp_tp_obj_t* pp_tp_gpio_sens_ctor(const char* id, vector_t* args);

void pp_tp_gpio_sens_dtor(pp_tp_obj_t* this);

int pp_tp_gpio_sens_init(pp_tp_gpio_sens_t* this, pp_tp_obj_type_t type,
			 const char* id, pp_tp_obj_dtor_func_t dtor,
			 pp_tp_sensdev_default_sdr_func_t default_sdr,
			 pp_tp_gpio_dev_t* gpio_dev, u_char gpio_no,
			 u_char invert_signal, int debounce);

void pp_tp_gpio_sens_cleanup(pp_tp_gpio_sens_t* this);

static inline pp_tp_gpio_sens_t*
pp_tp_gpio_sens_duplicate(pp_tp_gpio_sens_t* this)
{
    return (pp_tp_gpio_sens_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)this);
}

static inline void
pp_tp_gpio_sens_release(pp_tp_gpio_sens_t* this)
{
    pp_tp_obj_release((pp_tp_obj_t*)this);
}

static inline int pp_tp_gpio_sens_get_reading(pp_tp_gpio_sens_t* this) {
    return pp_tp_sensdev_get_reading(&this->base);
}

static inline void pp_tp_gpio_sens_subscribe(pp_tp_gpio_sens_t* this,
					     pp_tp_sensdev_subscriber_t* subs) {
    pp_bmc_tp_sensdev_subscribe(&this->base, subs);
}

static inline int pp_tp_gpio_sens_unsubscribe(pp_tp_gpio_sens_t* this,
					 pp_tp_sensdev_subscriber_t* subs) {
    return pp_bmc_tp_sensdev_unsubscribe(&this->base, subs);
}

static inline void pp_tp_gpio_sens_notify_subscribers(pp_tp_gpio_sens_t* this,
						 int reading) {
    pp_bmc_tp_sensdev_notify_subscribers(&this->base, reading);
}

/** Little helper that inverts a single gpio sensor reading */
static inline int pp_tp_gpio_sens_invert(int reading) {
    if (reading < 0) return reading; // don't invert an error
    return !reading;
}



#endif /* _PP_BMC_TP_GPIO_SENS_H */
