/**
 * tp_scan_sensdev.h
 *
 * Implementation support for a scannable sensdev
 * May or may not be used by a sensdev implementation,
 * however typical scannable sensors should use it.
 * 
 * (c) 2005 Peppercon AG, 7/25/2005, tbr@peppecon.de
 */

#ifndef __PP_BMC_SCAN_SENSDEV_H__
#define __PP_BMC_SCAN_SENSDEV_H__

#include <pthread.h>
#include <pp/bmc/tp_cond.h>
#include <pp/bmc/tp_sensdev.h>
#include "sensor_scanner.h"

/**
 * Abstract scannable sensor.
 * Be careful in using it. This defines actually a subclass of PP_TP_SENS_DEV
 * so you can't initialize it with a type that doesn't fit into
 * this class-hierarchy, for instance PP_TP_GPIO_SENS, or things like that.
 * These cases must be solved differently, that is implementing the scannable
 * interface by a 'mix-in' implementation.
 * However, relax, for most cases of i2c-sensor-chips this scannable
 * base class will fit and make things quite easy
 */
typedef struct pp_tp_scan_sensdev_s {
    pp_tp_sensdev_t base;

    /* intented to be scanned by sensor scanner */
    pp_sensor_scannable_t scannable;

    /* scan condition, if false, no update should be executed */
    pp_tp_cond_t* scan_cond;

    /* lock for reading and scheduled function id */
    pthread_mutex_t mtx;

    /* scheduled function id, i.e. determines whether an update is scheduled */
    int update_sf_id;

} pp_tp_scan_sensdev_t;

/**
 * cast a scannable interface pointer of pp_tp_scan_sensdev_t
 * back to the object type
 */
#define PP_TP_SCANNABLE_2_OBJ_CAST(intf_ptr, obj_type)                  \
    ((obj_type*)PP_TP_INTF_2_OBJ_CAST(intf_ptr, pp_tp_scan_sensdev_t,	\
				      scannable))

int pp_tp_scan_sensdev_init(pp_tp_scan_sensdev_t* this, pp_tp_obj_type_t type,
			    const char* id, pp_tp_obj_dtor_func_t dtor,
			    pp_tp_sensdev_default_sdr_func_t default_sdr,
			    void (*update)(pp_sensor_scannable_t*),
			    pp_tp_cond_t* scan_cond);

static inline void pp_tp_scan_sensdev_cleanup(pp_tp_scan_sensdev_t* this) {
    pp_tp_sensdev_cleanup(&this->base);
}

static inline pp_tp_scan_sensdev_t*
pp_tp_scan_sensdev_duplicate(pp_tp_scan_sensdev_t* this) {
    return (pp_tp_scan_sensdev_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)this);
}

static inline void pp_tp_scan_sensdev_release(pp_tp_scan_sensdev_t* this) {
    pp_tp_obj_release((pp_tp_obj_t*)this);
}

static inline int pp_tp_scan_sensdev_do_scan(pp_tp_scan_sensdev_t* this) {
    return this->scan_cond == NULL ? 1 : pp_bmc_tp_cond_is_true(this->scan_cond);
}

/**
 * Update sensor reading and schedule subscriber notification
 * in selector (BMC-main) thread.
 * @return identifier of scheduled function, needed for cleanup
 *
 * ATTENTION: this function is meant to be used by the
 *            scanner thread only! Do not call it from
 *            any other context than the scanner thread
 *            and only if the sensor implements scannable
 */
int pp_tp_scan_sensdev_update_reading(pp_tp_scan_sensdev_t* s, int reading);

/**
 * Registers a scannable sensor with the sensor scanner
 */
static inline void pp_tp_scan_sensdev_register(pp_tp_scan_sensdev_t* s,
					       int scan_group) {
    pp_sensor_scanner_add(&s->scannable, scan_group);
}

/**
 * Unregisters a scannable sensor before destruction.
 * This is not just removeing the scannable interface from
 * the sensor_scanner but also deleting possible scheduled
 * functions from the selector
 */
void pp_tp_scan_sensdev_unregister(pp_tp_scan_sensdev_t* s);

#endif /* __PP_BMC_SCAN_SENSDEV_H__ */
