/**
 * tp_sdr
 *
 * defines various objects for SDR configuration
 * 
 * (c) 2005 Peppercon AG, 2005/06/03, tbr@peppecon.de
 */

#ifndef __PP_BMC_TP_SDR_H__
#define __PP_BMC_TP_SDR_H__

#include <pp/vector.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/topo_factory.h>

/**
 * the base cfg_sdr is a object that configures additional detailed
 * information for a sensors sdr
 * Format of the values is identical to that of the IPMI spec.
 */
typedef struct pp_tp_cfg_sdr_s {
    pp_tp_obj_t base;
    
    union {                            /* sensor capabilities */
	struct {
	    BITFIELD5(unsigned char,
		      event_msg    : 2,   /* event msg control support */
		      threshold    : 2,   /* threshold access support */
		      hysteresis   : 2,   /* hysteresis support */
		      rearm        : 1,   /* auto rearm support */
		      ignore       : 1 );
	} bits;
	unsigned char byteval;            /* for c'tor */
    } caps; 
} pp_tp_cfg_sdr_t;

int pp_tp_cfg_sdr_init(pp_tp_cfg_sdr_t* o, pp_tp_obj_type_t type,
		       const char* id, 
		       void (*dtor)(pp_tp_obj_t*), vector_t* args);

static inline void pp_tp_cfg_sdr_cleanup(pp_tp_cfg_sdr_t* o) {
    pp_tp_obj_cleanup((pp_tp_obj_t*)o);
}

/**
 * the threshold config sdr is meant to configure analog sensors
 */
typedef struct pp_tp_thresh_sdr_s {
    pp_tp_cfg_sdr_t base;

    ipmi_sdr_evt_mask_t mask;           /* threshold mask */
    
    unsigned char nominal_read;         /* nominal reading, raw value */
    unsigned char normal_max;           /* normal maximum, raw value */
    unsigned char normal_min;           /* normal minimum, raw value */

    ipmi_sdr_threshold_t threshold;     /* threshold specification */
} pp_tp_thresh_sdr_t;

int pp_tp_thresh_sdr_init(pp_tp_thresh_sdr_t* o, pp_tp_obj_type_t type, 
			  const char* id, void (*dtor)(pp_tp_obj_t*),
			  vector_t* args);

static inline void pp_tp_thresh_sdr_cleanup(pp_tp_thresh_sdr_t* o) {
    pp_tp_cfg_sdr_cleanup(&o->base);
}

static inline pp_tp_thresh_sdr_t* 
pp_tp_thresh_sdr_duplicate(pp_tp_thresh_sdr_t* o) {
    return (pp_tp_thresh_sdr_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)o);
}

static inline void pp_tp_thresh_sdr_release(pp_tp_thresh_sdr_t* o) {
    pp_tp_obj_release((pp_tp_obj_t*)o);
}

/**
 * Write the values of the object to a SDR structure of type 'threshold sdr'.
 * Add the additional values snum, desc, entity_id, entity_instance.
 * @return PP_SUC if the sdr could be filled
 * @return PP_ERR if the sdr could not be filled (types not compatible)
 */
int pp_tp_thresh_cfg_sdr_fill_sdr(pp_tp_thresh_sdr_t* o, unsigned char snum,
				   const char* desc, unsigned char entity_id,
                                   unsigned char entity_instance, ipmi_sdr_header_t* sdr);

/* ppdoc
 * name:   IPMI Threshold SDR
 * tpname: ThreshSDR
 * desc:   Threshold SDR objects configure the default SDR values for
 * desc:   hreshold sensors. Complete SDRs will be obtained by
 * desc:   combining this information with default sensor information.
 * desc:   Note: the first argument specifies with other arguments are looked for
 * desc:         thresholds support must be set to != 0 in order to read args 1 - 13
 * desc:         hysteresis support must be set to != 0 in order to read args 14 + 15
 * arg 0:  [INT] cap: 1 byte sensor capabilities       (full sdr byte 12)
 * arg 1:  [INT] ltm: 2 bytes lower threshold mask  (full sdr byte 15+16)
 * arg 2:  [INT] utm: 2 bytes upper threshold mask  (full sdr byte 17+18)
 * arg 3:  [INT] stm: 1 byte setable threshold mask    (full sdr byte 19)
 * arg 4:  [INT] rtm: 1 byte readable threshold mask   (full sdr byte 20)
 * arg 5:  [INT] nrd: 1 byte nominal_read              (full sdr byte 32)
 * arg 6:  [INT] nma: 1 byte normal_max                (full sdr byte 33)
 * arg 7:  [INT] nmi: 1 byte normal_min                (full sdr byte 34)
 * arg 8:  [INT] nru: 1 byte non_recover upper thresh  (full sdr byte 37)
 * arg 9:  [INT] cru: 1 byte critical upper thresh     (full sdr byte 38)
 * arg 10: [INT] ncu: 1 byte non_critical upper thresh (full sdr byte 39)
 * arg 11: [INT] nrl: 1 byte non_recover lower thresh  (full sdr byte 40)
 * arg 12: [INT] crl: 1 byte critical lower thresh     (full sdr byte 41)
 * arg 13: [INT] ncl: 1 byte non_critical lower thresh (full sdr byte 42)
 * arg 14: [INT] pos: 1 byte positive hysteresis       (full sdr byte 43)
 * arg 15: [INT] neg: 1 byte negative hysteresis       (full sdr byte 44)
 * return: [OBJ(TRESH_SDR)]
 */
pp_tp_obj_t* pp_tp_thresh_sdr_ctor(const char* id, vector_t* args);


/**
 * the discrete config sdr is meant to configure discrete sensors
 */
typedef struct pp_tp_disc_sdr_s {
    pp_tp_cfg_sdr_t base;
    
    pp_tp_cfg_sdr_t cfg_sdr;

    ipmi_sdr_evt_mask_t mask;           /* de-/assertion and reading mask */
} pp_tp_disc_sdr_t;

int pp_tp_disc_sdr_init(pp_tp_disc_sdr_t* o, pp_tp_obj_type_t type, 
			const char* id, void (*dtor)(pp_tp_obj_t*),
			vector_t* args);

static inline void pp_tp_disc_sdr_cleanup(pp_tp_disc_sdr_t* o) {
    pp_tp_cfg_sdr_cleanup(&o->base);
}

static inline pp_tp_disc_sdr_t* 
pp_tp_disc_sdr_duplicate(pp_tp_disc_sdr_t* o) {
    return (pp_tp_disc_sdr_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)o);
}

static inline void pp_tp_disc_sdr_release(pp_tp_disc_sdr_t* o) {
    pp_tp_obj_release((pp_tp_obj_t*)o);
}

/**
 * Write the values of the object to a SDR structure of type 'threshold sdr'.
 * Add the additional values snum, desc, entity_id, entity_instance.
 * @return PP_SUC if the sdr could be filled
 * @return PP_ERR if the sdr could not be filled (types not compatible)
 */
int pp_tp_disc_cfg_sdr_fill_sdr(pp_tp_disc_sdr_t* o, unsigned char snum,
				 const char* desc, unsigned char entity_id,
                                 unsigned char entity_instance, ipmi_sdr_header_t* sdr);

/* ppdoc
 * name:   IPMI Discrete SDR
 * tpname: DiscSDR
 * desc:   Discrete SDR objects configure the default SDR values for 
 * desc:   discrete sensors. Complete SDRs will be obtained by
 * desc:   combining this information with default sensor information.
 * arg0:   [INT] cap: 1 byte sensor capabilities      (compact sdr byte 12)
 * arg1:   [INT] eam: 2 bytes assertion event mask    (compact sdr byte 15,16)
 * arg2:   [INT] edm: 2 bytes deassertion event mask  (compact sdr byte 17,18)
 * arg3:   [INT] rdm: 2 bytes discrete reading mask   (compact sdr byte 19,20)
 * return: [OBJ(DISC_SDR)]
 */
pp_tp_obj_t* pp_tp_disc_sdr_ctor(const char* id, vector_t* args);

#endif /* __PP_BMC_TP_SDR_H__ */
