/**
 * Sensor configuration functions
 *
 * (c) 2004 Peppercon AG, 12/15/2004, tbr@peppecon.de
 */

#include <pp/vector.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/host_board_id.h>
#include <pp/bmc/sensor.h>
#include <pp/bmc/topo_base_obj.h>
#include <pp/bmc/tp_ipmi_sens.h>

#include "topo_parser_impl.h"
#include "topo_ctors.h"
#include "sensor_scanner.h"

static void obj_inst_cb(pp_tp_obj_t* obj, va_list cb_ap);

static vector_t* objs_vector;
static int initialized = 0;

/*
 * The central sensor configuration function. Reads the sensor configuration
 * topology from a file, creates the object instances and pushes them to
 * interested consumers (bmc_dev_sensor, libpp_hardware, ...)
 * 
 * @return PP_OK or PP_ERR, errno will be set.
 */
int pp_bmc_sensor_topology_parse(const char* topo_file, ...) {
    va_list ap;
    int ret;
    pp_tp_def_t *tp_def;

    pp_bmc_log_notice("[SENSOR-CONFIG]: using topology from %s", topo_file);
    
    if (NULL == (tp_def = pp_tp_parse_topo(topo_file,
					   pp_tp_ctors_basic,
					   pp_tp_ctors_basic_sz))) {
	pp_bmc_log_perror("[SENSOR-CONFIG]: can't parse %s!", topo_file);
	pp_bmc_log_warn("[SENSOR-LIB]: no sensors configured!");
	ret = PP_ERR;
    } else {
        /* We have no context in the moment, although */
        /* it may be used again later on              */
	va_start(ap, topo_file);
	pp_tp_def_incarnate(tp_def, obj_inst_cb, ap);
	va_end(ap);
	pp_tp_def_destroy(tp_def);
	ret = PP_SUC;
    }
    return ret;
}

int pp_bmc_sensor_topology_init() {
    if (initialized) {
	return PP_ERR;
    }

    objs_vector = vector_new(NULL, 8, (void(*)(void*))pp_tp_obj_release);
    pp_sensor_scanner_init();
    
    initialized = 1;
    return PP_SUC;
}

void pp_bmc_sensor_topology_cleanup() {
    pp_bmc_sensor_scanner_stop();
    pp_sensor_scanner_cleanup();
    vector_delete(objs_vector);
    initialized = 0;
}

/**
 * This is the central callback for new objects. Forwards new objects
 * to misc other callback functions.
 */
static void
obj_inst_cb(pp_tp_obj_t* obj, va_list cb_ap)
{
    pp_bmc_object_creation_t callback;
    
    // notify all object creation callbacks
    while ((callback = va_arg(cb_ap, pp_bmc_object_creation_t)) != NULL) {
	callback(obj);
    }

    // some objects we need to keep, as nobody else will do it
    // and we don't want them to be deleted
    // Currently this holds true for all controllers
    if (PP_TP_OBJ_IS_TYPE(PP_TP_CTRL, obj)) {
	vector_add(objs_vector, pp_tp_obj_duplicate(obj));
    }
}
