%{
/**
 * topo_parser.l
 *
 * Topology parser lexical analyzer
 * (for parser debugging set YYDEBUG to 1 in topo_parser_immpl.h)
 * 
 * (c) 2005 Peppercon AG, 3/12/2005, tbr@peppecon.de
 */


#include <string.h>
#include "topo_parser_impl.h"
#include "topo_parser.tab.h"

#define YY_STACK_USED 0
#define YY_ALWAYS_INTERACTIVE 0
#define YY_NEVER_INTERACTIVE 0
#define YY_MAIN 0

#define YY_NO_UNPUT
#define YYERROR_VERBOSE

extern int pp_tp_yylex(void);
extern int pp_tp_yyparse(void);
#if YYDEBUG
extern int pp_tp_yydebug;
#endif

pp_tp_def_t* pp_tp_yy_def = NULL;
unsigned int pp_tp_yy_pos;

typedef struct {
  YY_BUFFER_STATE buffer;
  unsigned int pos;
} incl_state_t;

static char* process_strlit(char* text);
static incl_state_t* create_incl_state(YY_BUFFER_STATE b, unsigned int pos);
static FILE* new_file(const char* fname, unsigned int* pos);
static void push_input(FILE* in, unsigned int inpos);
static int pop_input(void);
static int noerrors;
static vector_t incl_stack;
static vector_t filenames;
%}

%option noyywrap
%x COMMENTLINE
%x INCLUDE

NUM      [-+]?[0-9]+
HEXNUM   [-+]?0x[0-9a-fA-F]+
ID       [_a-zA-Z][_a-zA-Z0-9]*

%%

"include"           { BEGIN(INCLUDE); }

"true"              { pp_tp_yylval.number = 1;
                      return BOOLEAN_LIT; }

"false"             { pp_tp_yylval.number = 0;
                      return BOOLEAN_LIT; }

"?"                 { return PLACEHOLDER; }

{NUM}               { pp_tp_yylval.number = atoi(pp_tp_yytext);
	              return INT_LIT;
	            }

{HEXNUM}            { pp_tp_yylval.number = strtol(pp_tp_yytext, NULL, 16);
                      return INT_LIT;
                    }

{ID}	            { pp_tp_yylval.string = strdup(pp_tp_yytext);
	              return ID;
	            }

";"|":"|","|"("|")" { return pp_tp_yytext[0]; }

\"([^\"\n]|\\\"|\\\n)*\" { pp_tp_yylval.string = process_strlit(pp_tp_yytext);
	              return STRING_LIT;
	            }

\"[^\"\n]*\n        { pp_tp_yylval.string = pp_tp_yytext;
                      pp_tp_yyerror("unterminated string literal");
                      ++pp_tp_yy_pos;
	              return STRING_LIT;
	            }

"#"                 { BEGIN(COMMENTLINE); }

<COMMENTLINE>.      /* eat the comment */

<COMMENTLINE>\n     { BEGIN(INITIAL); 
                      ++pp_tp_yy_pos;
                    }

<INCLUDE>[ \t]*     /* eat up whitespace */

<INCLUDE>[^ \t\n]+  { FILE* in; unsigned int inpos;
                      if (NULL == (in = new_file(pp_tp_yytext, &inpos))) {
                         pp_tp_yyerrorf(pp_tp_yy_pos, 
                                        "can't open include '%s': %s",
                                        pp_tp_yytext, strerror(errno));
                      } else {
                         push_input(in, inpos);
                      }
                      BEGIN(INITIAL);
                    }

\n                  { ++pp_tp_yy_pos; }

[ \t]+              /* eat up whitespace */

.                   { pp_tp_yyerrorf(++pp_tp_yy_pos, "invalid character: '%c'",
			             pp_tp_yytext[0]); }

<<EOF>>             { if (pop_input() == 0)
                         yyterminate();
                    }

%%

PP_SYM_HIDDEN
pp_tp_def_t* pp_tp_parse_topo(const char* filename, 
                              pp_tp_obj_ctor_def_t* ctors,
		              int no_ctors) {
#if YYDEBUG
  pp_tp_yydebug = 1;
#endif
  pp_tp_def_t* ret;
  yy_init = 1;

  if (pp_tp_yy_def != NULL) {
      pp_tp_def_destroy(pp_tp_yy_def);
      pp_tp_yy_def = NULL;
  }

  vector_new(&filenames, 4, free);

  if (NULL != (pp_tp_yyin = new_file(filename, &pp_tp_yy_pos))) {
    vector_new(&incl_stack, 10, free);
    pp_tp_yy_pos = 1;
    pp_tp_yy_def = pp_tp_def_create(ctors, no_ctors);
    noerrors = 0;
    if (0 != pp_tp_yyparse() || noerrors > 0) {
       printf("%s: encountered %d error%s\n", 
	      filename, noerrors, noerrors > 1 ? "s" : "");
       pp_tp_def_destroy(pp_tp_yy_def);
       errno = EINVAL;
       ret = pp_tp_yy_def = NULL;
    } else {
       ret = pp_tp_yy_def;
       pp_tp_yy_def = NULL;
    }
    vector_delete(&incl_stack);
    yy_delete_buffer(YY_CURRENT_BUFFER);
    fclose(pp_tp_yyin);
  } else {
    //perror (filename);
    ret = NULL;
  }
  vector_delete(&filenames);
  return ret;
}

/*
 * error reporting, used by the parser and other functions
 * --------------------------------------------------------
 */

PP_SYM_HIDDEN
int pp_tp_yyerror(const char* text) {
  pp_tp_yyerror_base(pp_tp_yy_pos, text);
  return 0;
}

PP_SYM_HIDDEN
int pp_tp_yyerror_base(const int pos, const char* text) {
  noerrors++;
  fprintf(stderr, "%s:%d: ERROR: %s.\n", 
         (char*)vector_get(&filenames, pos >> 24), pos & 0xffffff, text);
  return 0;
}

PP_SYM_HIDDEN
int pp_tp_yyerrorf(const int pos, const char *format, ...) {
  char* buf;
  va_list ap, aq;

  va_start(ap, format);
  va_copy(aq, ap);
  buf = alloca(vsnprintf(NULL, 0, format, ap) + 1);
  va_end(ap);
  vsprintf(buf, format, aq);
  va_end(aq);
  pp_tp_yyerror_base(pos, buf);
  return 0;
}

/* 
 * processes escapes and cuts off the leading and trailing quotes
 */
static char* process_strlit(char* text) {
  char *str, *s, c;
  int len = strlen(text) -2;
  ++text; text[len] = '\0';
  str = s = malloc(len + 1);
  while((c = *text++) != '\0') {
    if (c == '\\') {
      c = *text++;
      if (c == '\0') break;
      if (c == '\n') continue;
    }
    *s++ = c;
  }
  *s = '\0';
  return str;
}

#define PATH_SEP '/'

static FILE* new_file(const char* filename, unsigned int* pos) {
  FILE* in;
  char new_fname[FILENAME_MAX];
  char *tmp;
  if(*filename != PATH_SEP && vector_size(&filenames) > 0) {
    /* no absolute path, get working dir from current_file */
    tmp = vector_get(&filenames, vector_size(&filenames) - 1);
    strcpy(new_fname, tmp);
    tmp = strrchr(new_fname, PATH_SEP);
    if(tmp) {
        if(tmp - new_fname + strlen(filename) + 2 > FILENAME_MAX)
            return NULL;
        strcpy(tmp + 1, filename);
        filename = new_fname;
    }
  }
  if (NULL != (in = fopen(filename, "r"))) {
    vector_add(&filenames, strdup(filename));
    *pos = ((vector_size(&filenames) - 1) << 24) + 1;
  }
  return in;
}

static void push_input(FILE* in, unsigned int inpos) {
  vector_add(&incl_stack, create_incl_state(YY_CURRENT_BUFFER, pp_tp_yy_pos));
  pp_tp_yy_switch_to_buffer(pp_tp_yy_create_buffer(in, YY_BUF_SIZE));
  pp_tp_yy_pos = inpos;
}

static int pop_input() {
  incl_state_t* is;
  int ss;

  fclose(pp_tp_yyin);
  pp_tp_yy_delete_buffer(YY_CURRENT_BUFFER);
  ss = vector_size(&incl_stack);
  if (ss == 0) return 0;

  is = (incl_state_t*)vector_get(&incl_stack, ss - 1);
  pp_tp_yy_switch_to_buffer(is->buffer);
  pp_tp_yy_pos = is->pos;
  vector_remove(&incl_stack, ss - 1);
  return 1;
}

static incl_state_t* create_incl_state(YY_BUFFER_STATE buffer, 
                                       unsigned int pos) {
  incl_state_t* s = malloc(sizeof(incl_state_t));
  s->buffer = buffer;
  s->pos = pos;
  return s;
}
