%{
/**
 * topo_parser.y
 *
 * Topology parser's grammar description and according productions
 * (for parser debugging set YYDEBUG to 1 in topo_parser_impl.h)
 * 
 * (c) 2005 Peppercon AG, 3/12/2005, tbr@peppecon.de
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "topo_parser_impl.h"
    
#define YYENABLE_NLS 0
#define YYINCLUDED_STDLIB_H 1
#define YYLTYPE_IS_TRIVIAL 0
#define YYERROR_VERBOSE
%}

%union {
    int                      number;
    char*                    string;
    pp_tp_arg_t              arg;
    arg_list_t*              arglist;
    obj_ctor_t*              objctor;
}

%{
extern int pp_tp_yy_pos;
extern pp_tp_def_t* pp_tp_yy_def;
int pp_tp_yylex (void);
#if YYDEBUG
static void print_token_value (FILE *, int, YYSTYPE);
#define YYPRINT(file, type, value) print_token_value (file, type, value)
#endif
%}

%token              PLACEHOLDER
%token <number>     INT_LIT BOOLEAN_LIT
%token <string>     ID STRING_LIT

%type  <arg>        argument alias_argument
%type  <arg>        basic_argument symbol_argument complex_argument
%type  <arglist>    argument_list
%type  <objctor>    obj_constructor
%type  <number>     decls obj_decl alias_decl

%%

input: 
    /* empty */
  | input decls
  ;

decls:
    obj_decl
  | alias_decl
  | ID ':' error ';' { yyerrok; }
  | ID error ';' { yyerrok; }
  ;

obj_decl:
    ID ':' obj_constructor ';' {
      $$ = tp_yy_ctor_register(pp_tp_yy_def, pp_tp_yy_pos, $1, $3);
    }
  | ID ':' obj_constructor error { yyerrok; }
;

alias_decl:
    ID ':' alias_argument ';' {
      $$ = tp_yy_alias_register(pp_tp_yy_def, pp_tp_yy_pos, $1, &$3);
    }
  | ID ':' alias_argument error { yyerrok; }
;

obj_constructor:
    ID '(' argument_list ')' {
      $$ = tp_yy_ctor_create(pp_tp_yy_def, pp_tp_yy_pos, $1, $3);
    }
  | ID '(' ')' {
      $$ = tp_yy_ctor_create(pp_tp_yy_def, pp_tp_yy_pos, $1,
			     arg_list_create());
    }
;

argument_list:
    argument {
      $$ = arg_list_create();
      arg_list_add($$, &$1);
    }
  | argument_list ',' argument {
      arg_list_add($$, &$3);
    }
;

argument:
    basic_argument
  | symbol_argument
  | complex_argument
;

alias_argument:
    basic_argument
  | symbol_argument
;

basic_argument:
    INT_LIT {
    pp_tp_arg_t arg = { .type = PP_TP_ARG_INT, .value.intval = $1 };
      $$ = arg;
    }  
  | BOOLEAN_LIT {
      pp_tp_arg_t arg = { .type = PP_TP_ARG_BOOL, .value.intval = $1 };
      $$ = arg;
    }  
  | STRING_LIT {
      pp_tp_arg_t arg = { .type = PP_TP_ARG_STRING, .value.string = $1 };
      $$ = arg;
    }  
  | PLACEHOLDER {
      pp_tp_arg_t arg = { .type = PP_TP_ARG_PLACEHOLDER };
      $$ = arg;
    }
;

symbol_argument:
    ID {
      $$ = tp_yy_symbol_arg(pp_tp_yy_def, pp_tp_yy_pos, $1);
    }
;

complex_argument:
    obj_constructor {
      char* sym = tp_yy_ctor_auto_ref(pp_tp_yy_def, pp_tp_yy_pos, $1);
      if (sym != NULL) tp_yy_ctor_register(pp_tp_yy_def, pp_tp_yy_pos,
					   strdup(sym), $1);
      $$.type = PP_TP_ARG_SYMBOL;
      $$.value.string = sym;
    }
;

%%

#if YYDEBUG
static void print_token_value (FILE *file, int type, YYSTYPE value) {
    switch (type) {
      case STRING_LIT:
      case ID:
	  fprintf (file, "%s", value.string);
	  break;
      case INT_LIT:
	  fprintf (file, "%d", value.number);
	  break;
      case BOOLEAN_LIT:
	  fprintf (file, "%s", value.number ? "true" : "false");
	  break;
      default:
	  fprintf (file, "na");
    }
}
#endif
