/**
 * topo_parser_impl.h
 *
 * public and internal definitions of the topology parser
 * 
 * (c) 2005 Peppercon AG, 3/9/2005, tbr@peppecon.de
 */

#ifndef __TOPO_PARSER_IMPL_H__
#define __TOPO_PARSER_IMPL_H__

#include <pp/base.h>
#include <pp/vector.h>
#include <pp/bmc/topo_factory.h>


/* 
 * set YYDEBUG to 1 for detailed parser debugging
 */
#ifndef YYDEBUG
#define YYDEBUG 0
#endif

/*
 * Parser Handling, i.e. parsing instantiation, cleanup
 * -----------------------------------------------------
 */

/*
 * predefined object constructors,
 * same as used by the parser, but without arguments
 */
typedef struct {
    const char* name;
    pp_tp_obj_ctor_func_t method;
} pp_tp_obj_ctor_def_t;

/*
 * result returned by the parser
 * contains all constructors, either predefined or parsed
 */
typedef struct {
    vector_t*  objctors;   // all complete c'tors in cronological order
    pp_hash_t* ctorsmap;   // all c'tors for fast lookup by name
    pp_hash_t* aliasmap;   // symbolic argument aliases
    pp_hash_t* unnamedmap; // unnamed object counters
} pp_tp_def_t;

pp_tp_def_t* pp_tp_def_create(pp_tp_obj_ctor_def_t* ctors, int no_ctors);

void pp_tp_def_destroy(pp_tp_def_t* tp_def);

/**
 * Parses a topology file and returns topology definition.
 * Parse errors will be written to stderr, for the time beeing.
 * (maybe we need to collect them later, for presenting them to a user)
 *
 * @param  fname     name of topology file
 * @param  ctors     pointer to array of predefined constructor defs
 * @param  no_ctors  number of constructor defs in ctors
 * @return pointer to pp_tp_def_t with all objs and c'tors
 *         or NULL if error, errno will be set.
 *         Parse errors will be printed to stderr!
 */
pp_tp_def_t* pp_tp_parse_topo(const char* fname, pp_tp_obj_ctor_def_t* ctors,
			      int no_ctors);

/**
 * callback for pp_tp_incarnate_topo
 * will be called for each successfully created object
 */
typedef void (*pp_tp_incarnate_cb_t)(pp_tp_obj_t* obj, va_list ap);

/**
 * topology instantiation funktion
 * tries to instantiate all objects that have a complete c'tor
 * in case a single instantiation fails, others may proceed,
 * i.e. the function will not terminate in this case
 * 
 * @param tp     topology definition, as created by parser
 * @param obj_cb callback, called for each object created
 * @return PP_SUC if no error occured
 *         PP_ERR if an error occured
 */
int pp_tp_def_incarnate(pp_tp_def_t* tp, pp_tp_incarnate_cb_t obj_cb,
			va_list cb_ap);

/*
 * some definitions used by the parser internally,
 * no need to complicate names by applying namespaces
 * ----------------------------------------------------
 */

/*
 * Argument handling
 * ------------------
 */
typedef struct {
    int no_ph; // number of placeholders in args vector
    vector_t args;
} arg_list_t;

#define ARG_LIST_COMPLETE(al) (al && al->no_ph == 0)

arg_list_t* arg_list_create(void);

void arg_list_destroy(arg_list_t*);

/* destroys the argument list and its vector,
 * but doesn't cleanup the arguments            */
void arg_list_destroy_wo_args(arg_list_t*);

/* adds arg and updates no_ph in arg_list_t */
void arg_list_add(arg_list_t* al, pp_tp_arg_t* arg);

/*
 * symbol and alias handling
 * -------------------------
 */

/* register symbol or argument alias */
int tp_yy_alias_register(pp_tp_def_t* tp, int pos, char* id,
			 pp_tp_arg_t* arg);

/* resolves a symbol to its actual argument, taking arliases
 * into account, i.e. either  basic argument or symbol argument */
pp_tp_arg_t tp_yy_symbol_arg(pp_tp_def_t* tp, int pos, char* sym);


/*
 * c'tor Handling
 * ---------------
 */
typedef struct obj_ctor_s {
    char* name;
    pp_tp_obj_ctor_func_t method;
    arg_list_t* arglist;
} obj_ctor_t;

#define CTOR_IS_TEMPLATE(ctor) \
    (ctor->arglist == NULL || ctor->arglist->no_ph > 0)

obj_ctor_t* obj_ctor_create(char* name, pp_tp_obj_ctor_func_t method, 
			    arg_list_t* arglist);

void obj_ctor_destroy(obj_ctor_t* ctor);

/* checks, whether given class name is known, i.e. may return an error */
obj_ctor_t* tp_yy_ctor_create(pp_tp_def_t* tp, int pos, char* class,
			    arg_list_t* arglist);

/* looks up a c'tor by name */
obj_ctor_t* tp_yy_ctor_find(pp_tp_def_t* tp, int pos, const char* id);

/* looks up an instance ctor,                       *
 * i.e. checks whether ctor will create an instance */
obj_ctor_t* tp_yy_ctor_inst_find(pp_tp_def_t* tp, int pos, const char* id);

/* creates an annonymous object ref name, only if ctor is complete */
char* tp_yy_ctor_auto_ref(pp_tp_def_t* tp, int pos, obj_ctor_t* ctor);

int tp_yy_ctor_register(pp_tp_def_t* tp, int pos, char* id,
			obj_ctor_t* ctor);

/*
 * Object Handling
 * -----------------
 */

/* checks whether argument list is complete and returns error if not */
pp_tp_obj_t* obj_create(int pos, obj_ctor_t* ctor);


/*
 * error reporting, used by the parser and other functions
 * (defined in topo_parser.l)
 * --------------------------------------------------------
 */
int pp_tp_yyerror(const char*);
int pp_tp_yyerrorf(const int pos, const char *format, ...);
int pp_tp_yyerror_base(const int pos, const char* text);

#endif
