/**
 * tp_gpio_multi_act.c
 *
 * Actor, setting a collection of gpios at ones
 * currently rudimentary in terms of features.
 * 
 * (c) 2005 Peppercon AG, tbr@peppercon.de
 */

#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_gpio_dev.h>
#include <pp/bmc/tp_actor.h>
#include <pp/bmc/tp_gpio_multi_act.h>

// use some static fields for storing sensor numbers, currently
#define MAX_GPIO_NRS  16

typedef struct {
    pp_tp_gpio_dev_t* gpio_dev;
    int gpio_nr_cnt;
    char gpio_nr[MAX_GPIO_NRS];
    char gpio_initial[MAX_GPIO_NRS];
} gpio_multi_desc_t;

static void gpio_multi_desc_del_func(void* arg);
static void gpio_multi_desc_set_initial(pp_tp_gpio_multi_act_t* this);
static void gpio_multi_act_dtor(pp_tp_obj_t* o);
static void gpio_multi_act_cleanup(pp_tp_gpio_multi_act_t* this);
static int gpio_multi_act_set(pp_tp_gpio_multi_act_t* this,
			      unsigned int val, unsigned int tri);

pp_tp_obj_t* pp_tp_gpio_multi_act_ctor(const char* id, vector_t* args) {
    const char* msg = "[GPIO-MULTI-ACT] failed, arg[%d]: %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;  
    gpio_multi_desc_t* gdesc;
    int arg_idx = 0, darg_count = 0, garg_count = 0;
    pp_tp_gpio_multi_act_t* gma = malloc(sizeof(pp_tp_gpio_multi_act_t));
    
    pp_tp_actor_init(&gma->base, PP_TP_GPIO_MULTI_ACT, id,
		     gpio_multi_act_dtor);
    
    gma->gpio_cnt = 0;
    gma->gpio_desc_cnt = 0;
    gma->last_value = -1;
    gma->set_value = gpio_multi_act_set;

    /* initially alloc space for 2 gpio devs */ 
    gma->gpio_desc = vector_new(NULL, 2, gpio_multi_desc_del_func);

    /* create as many gpio dev structures as needed */
    do {
	pp_tp_gpio_dev_t* gdev;
	
	darg_count = pp_tp_arg_scanf(args, arg_idx, &err, "o<g>", &gdev);
	if (darg_count == 0) {
	    break;
	}
	if (darg_count != 1) {
	    pp_bmc_log_error(msg, ___F, id, pp_strstream_buf(&err));
	    goto bail;
	}
	
	gma->gpio_desc_cnt++;
	gdesc = (gpio_multi_desc_t*) malloc(sizeof(gpio_multi_desc_t));
	gdesc->gpio_dev = pp_tp_gpio_dev_duplicate(gdev);
	gdesc->gpio_nr_cnt = 0;
	vector_add(gma->gpio_desc, gdesc);

	arg_idx++;

	/* now get the gpio numbers for this device */
	do {
	    char gpio_nr, gpio_initial;
	    
	    garg_count = pp_tp_arg_scanf(args, arg_idx, &err, "d<c>d<c>", &gpio_nr, &gpio_initial);
	    if (garg_count == 0) {
		break;
	    }
	    if (garg_count != 2) {
		pp_bmc_log_error(msg, ___F, id, pp_strstream_buf(&err));
		goto bail;
	    }
	    if (gdesc->gpio_nr_cnt == MAX_GPIO_NRS) {
		pp_bmc_log_error(msg, gma->gpio_desc_cnt, "too many gpio numbers");
		goto bail;
	    }

	    if (gpio_initial > 2) {
		pp_bmc_log_error(msg, gma->gpio_desc_cnt,
				 "initial gpio value must be 0,1 or 2 (tristate)");
		goto bail;	
	    }
	    
	    gdesc->gpio_nr[gdesc->gpio_nr_cnt] = gpio_nr;
	    gdesc->gpio_initial[gdesc->gpio_nr_cnt] = gpio_initial;
	    pp_tp_gpio_dev_init_gpio(gdesc->gpio_dev, 1 << gdesc->gpio_nr[gdesc->gpio_nr_cnt]);
	    gdesc->gpio_nr_cnt++;
	    
	    ++gma->gpio_cnt;

	    arg_idx += 2;
	} while (garg_count != 0);
    } while(darg_count != 0);

    if (gma->gpio_cnt < 1) {
	pp_bmc_log_error("[GPIOMultiAct] '%s' too few gpio argments",
			 ((pp_tp_obj_t*)gma)->id);
	goto bail;
    }

    /* calc and set the initial values for output and tristate */
    gpio_multi_desc_set_initial(gma);
       
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)gma;
    
 bail:
    pp_strstream_free(&err);
    gpio_multi_act_dtor((pp_tp_obj_t*)gma);
    return NULL;
}

static void gpio_multi_act_dtor(pp_tp_obj_t* o) {
    pp_tp_gpio_multi_act_t* this = (pp_tp_gpio_multi_act_t*)o;
    assert(o != NULL);
    gpio_multi_act_cleanup(this);
    free(this);
}

static void gpio_multi_act_cleanup(pp_tp_gpio_multi_act_t* this) {
    vector_delete(this->gpio_desc);
    pp_tp_actor_cleanup(&this->base);
}

static void gpio_multi_desc_del_func(void* arg) {
    gpio_multi_desc_t *desc = (gpio_multi_desc_t*) arg;
    
    pp_tp_gpio_dev_release(desc->gpio_dev);
    free(desc);
}

static void
gpio_multi_desc_set_initial(pp_tp_gpio_multi_act_t* this)
{
    int gdcnt = 0, gncnt = 0, i;
    unsigned int val = 0, tri = 0;
    gpio_multi_desc_t *gpio_desc;
    
    for (i = this->gpio_cnt - 1; i >= 0; --i) {
	char state;

	gpio_desc = (gpio_multi_desc_t*) vector_get(this->gpio_desc, gdcnt);

	state = gpio_desc->gpio_initial[gncnt];

	val |= ((state == 2) ? 0 : state) << i;
	tri |= ((state == 2) ? 1 : 0) << i;
	
	if (++gncnt == gpio_desc->gpio_nr_cnt) {
	    gncnt = 0;
	    ++gdcnt;
	}
    }

    if (PP_FAILED(pp_tp_gpio_multi_act_set(this, val, tri))) {
	pp_bmc_log_warn("[GPIOMultiAct] '%s' could not set initial value",
			((pp_tp_obj_t*)this)->id);
    }     
}

static int gpio_multi_act_set(pp_tp_gpio_multi_act_t* this,
			      unsigned int val, unsigned int tri) {
    int gdcnt = 0, gncnt = 0, v = 0, m = 0, t = 0, i, gno;
    gpio_multi_desc_t *gpio_desc;

    for (i = this->gpio_cnt - 1; i >= 0; --i) {
	assert(gdcnt < this->gpio_desc_cnt);
	gpio_desc = (gpio_multi_desc_t*) vector_get(this->gpio_desc, gdcnt);
	gno = gpio_desc->gpio_nr[gncnt];
	m |= (0x1 << gno);
	v |= (((val >> i) & 0x1) << gno);
	t |= (((tri >> i) & 0x1) << gno);
	if (++gncnt == gpio_desc->gpio_nr_cnt) {
	    pp_tp_gpio_dev_set_val(gpio_desc->gpio_dev, m, v, t);
	    gncnt = v = m = 0;
	    ++gdcnt;
	}
    }
    this->last_value = val;
    return PP_SUC;
}

int pp_tp_gpio_multi_act_get_last(pp_tp_gpio_multi_act_t* this) {
    return this->last_value;
}
