/**
 * serial_snoop.c
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * This file implements a serial handler for snooping the
 * connection while the serial client is connected to the
 * host (smux=host_chassis)
 */

#include "serial_snoop.h"
#include <pp/bmc/serial_lowlevel.h>
#include <pp/bmc/debug.h>
#include <pp/selector.h>
#include <pp/base.h>
#include <pp/termios.h>
#include <pp/bmc/utils.h>
#include <pp/bmc/serial_adapter.h>
#include <pp/bmc/ipmi_chan.h>

#include <stdio.h>
#include <sys/poll.h>
#include <unistd.h>
#include <fcntl.h>


static int serial_fd = -1;
static int serial_fd_hndl = -1;

typedef struct {
    unsigned char last_char;
    int in_frame;              // number of chars in current (correct) frame, -1 of not in frame
    unsigned char buf[14];     // buffer, one buffer is enough as all msgs start with a unique char that is escaped in the msg
} snoop_t;
static snoop_t snoop_i;
static snoop_t* snoop = &snoop_i;


/* internal prototype */
static int serial_read(const int item_id UNUSED, const int fd,
                       const short event UNUSED, void* ctx UNUSED);
static void init_snoop(void);
static void snoop_byte(unsigned char);
static void snoop_msg(unsigned char* buf, int len);


void bmc_serial_snoop_set_fd(int fd) {
    serial_fd = fd;
}

/* called on activation due to smux switching */
static void snoop_smux_activate_hndl(pp_bmc_serial_mux_state_t oldstate) {
    init_snoop();
    
    if (oldstate == PP_SMUX_BMC_CHASSIS) {
        // assume that serial_fd is set to correct open fd by serial_adapter
    } else {
        serial_fd = -1;
    }
    if (serial_fd < 0) {
        serial_fd = open(PP_BMC_SERIAL_TTY, O_RDWR | O_NONBLOCK);
        
        if (serial_fd < 0) {
            pp_bmc_log_info("[SER] snoop could not open %s", PP_BMC_SERIAL_TTY);
            return;
        }
    }
    
    if (pp_base_set_tty_params(serial_fd, 115200, "", 8, 1, 0, 0) < 0) {
        pp_bmc_log_pwarn("[SER] failed to set tty params");
        // ignore this and try to go without correct tty params
    }
    serial_fd_hndl = pp_select_add_fd(serial_fd, POLLIN, serial_read, NULL);
    pp_bmc_log_debug("[SER] Now snooping on serial connection");
}

/* called on deactivation due to smux switching */
static void snoop_smux_deactivate_hndl(pp_bmc_serial_mux_state_t newstate) {
    if (newstate == PP_SMUX_BMC_CHASSIS) {
        bmc_serial_adapter_set_fd(serial_fd);
    } else {
        close (serial_fd);
    }
    serial_fd = -1;
    
    pp_select_remove_fd(serial_fd_hndl);
    serial_fd_hndl = -1;
    pp_bmc_log_debug("[SER] Snooping on serial connection stopped");
    pp_bmc_serial_mux_deactivate_finished();
}

static int serial_read(const int item_id UNUSED, const int fd,
                const short event UNUSED, void* ctx UNUSED) {
    char buf[60];
    int n;
    int i;

    if (0 > (n = read(serial_fd, buf, sizeof(buf)))) {
        // read went wrong
        if (errno == EINTR || errno == EAGAIN) {
            return PP_SUC;
        } else {
            pp_bmc_log_perror("[SER] snooping_read(fd=%d)", fd);
            return PP_ERR;
        }
    } else {
        for (i=0; i<n; i++) {
            // snooping works on bytes only
            snoop_byte(buf[i]);
        }
    }
    
    return PP_SUC;
}

static void init_snoop() {
    snoop->last_char = 0;
    snoop->in_frame = -1;
}

static void snoop_byte(unsigned char new_char) {
    unsigned char valid;
    // static const unsigned char* get_ch_cap[20];
    // 0x1b = <ESC>
    
    // snoop for "<ESC>("
    if (new_char == '(') {
        if (snoop->last_char == 0x1b) {
            pp_bmc_log_debug("[SER] snoop detected '<ESC>(' sequence");
            pp_bmc_serial_mux_switch(PP_SMUX_BMC_CHASSIS);
            return;
        }
    }
    
    /* snoop for serial command
     * this is somehow a reimplementation of the serial_adapter. Any way to do it better ?
     * messages always start with a unique character, so snooping can be done with one state machine only
     */

    // always restart if we receive a <START_FRAME>
    valid = 0;
    if (new_char == 0xa0) {
        init_snoop();
        snoop->in_frame = 0;
    } else 
    if (new_char == 0xa5) {
        // stop character
        if (snoop->in_frame > 0) {
            snoop_msg(snoop->buf, snoop->in_frame);
            snoop->in_frame = -1;  // reset snoop buffer
        }
    } else {
        // only receive bytes if we're in a frame, ignore otherwise
        if (snoop->in_frame > -1) {
            if (snoop->last_char == 0xaa) {
                // last byte was escaped
                if ( (new_char == 0xb0) || (new_char == 0xb5) || (new_char == 0xb6) || (new_char == 0xba) ) {
                    valid = 1;
                    new_char = new_char - 0x10;
                } else {
                    // character not escaped correct, restart snooping
                    snoop->in_frame = -1;
                }
            } else {
                // last byte not escaped
                valid = 1;
            }
        } // else: dont snoop out of frame
    }
    
    if (valid == 1) {
        snoop->buf[snoop->in_frame] = new_char;
        snoop->in_frame++;
        if (snoop->in_frame == sizeof(snoop->buf)) {
            // too long for get_channel_capabilities msg, discard
            snoop->in_frame = -1;
        }
    }

    snoop->last_char = new_char;
}

/* snoop for a get_channel_authentication_capabilities msg */
static void snoop_msg(unsigned char* buf, int len) {
    // <0xa0 (start)><rsaddr 0x20><netFn/rsLUN 0x18><checksum c8><rqAddr><rqSeq/rqLun><cmd 0x38><data1><data2><checksum><0xa5 (stop)>
    // size(fix) = 11
    // bis nach der 1.checksum: alles fix
    
    if (len != 9) {
        // wrong size
        return;
    }
    if ( (buf[0] != 0x20) || (buf[1] != 0x18) || (buf[2] != 0xc8)) {
        // byte 0-2 always fix
        return;
    }
    if (buf[8] != pp_bmc_calc_ipmi_checksum(buf, 3, 7)) {
        // 2nd checksum wrong
        return;
    }
    if (buf[5] != 0x38) {
        // command wrong
        return;
    }
    // this seems to be a get_channel_authentication_capabilites message
    pp_bmc_log_debug("[SER] snoop detected get_channel_authentication_capabilities message, switchting to bmc");
    
    pp_bmc_serial_mux_switch(PP_SMUX_BMC_CHASSIS);    
    
    // push this message to the serial channel after activation
    pp_bmc_serial_receive_msg(buf, len);
}

void bmc_serial_snoop_set_access_mode(unsigned char access_mode) {
    if (pp_bmc_serial_mux_get_state() == PP_SMUX_HOST_CHASSIS) {
        switch (access_mode) {
        case IPMI_CHAN_ACCESS_SHARED:
            // activate channel
            if (serial_fd_hndl == -1) {
                snoop_smux_activate_hndl(PP_SMUX_BMC_CHASSIS);
            }
            break;
        case IPMI_CHAN_ACCESS_DISABLE:
            // deactivate
            if (serial_fd_hndl > 0) {
                pp_select_remove_fd(serial_fd_hndl);
                serial_fd_hndl = -1;
                close(serial_fd);
                serial_fd = -1;
            }
            break;
        default:
            // must never happen
            assert(0);
        }
    }
}

int pp_serial_snoop_init() {
    /* register smux-change handler and setup for initial state */
    pp_bmc_serial_mux_register_handler(PP_SMUX_HOST_CHASSIS,
                                       snoop_smux_activate_hndl,
                                       snoop_smux_deactivate_hndl);
    return PP_SUC;
}

void pp_serial_snoop_cleanup() {
    /* register smux-change handler and setup for initial state */
    pp_bmc_serial_mux_clear_handler(PP_SMUX_HOST_CHASSIS);
}

