/**
 * smi_kcs_adapter.c
 *
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 *
 * Uses the bmccore-msgrouter and the lpc.o Kernelmodule
 * to connect the bmccore to the host system interface.
 *
 * Please load the lpc.o module before initializing this adapter.
 *
 * Please create a character device /dev/lpc before initializing.
 * The major number is 246, the minor number doesn't matter.
 */

#include "pp/bmc/debug.h"

#include "pp/base.h"
#include "pp/selector.h"
#include <pp/cfg.h>
#include "pp/bmc/ipmi_msg.h"
#include "pp/bmc/bmc_router.h"
#include "pp/bmc/bmc_core.h"
#include "pp/bmc/ipmi_chan.h"
#include "pp/bmc/ipmi_sess.h"
#include "pp/bmc/smi_adapter.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <linux/ioctl.h>
#include <sys/ioctl.h>

#include "smi_kcs_adapter.h"     // the functions that we offer
#include "lpc.h"                 // the functions that we use

/**
 * the global variables
 */

#define KCS_DEV "/dev/lpc"

typedef struct {
   int fd;            // the file descriptor of our si connection
   int selID;         // our selector entry
} smi_loop_adapter_globals;

static smi_loop_adapter_globals *sla_globals;
static smi_loop_adapter_globals my_sla_globals;

/* KCS message format:
 *     unsigned char netfn : 6;
 *     unsigned char lun   : 2;
 *     unsigned char cmd;
 *     unsigned char data[];
 */
#define KCS_HEADER_SIZE 2


/**
 * Non communication interrupt actor.
 */
static void bmc_set_host_atn(unsigned char value UNUSED) {
    // TODO: implement the interrupt, does not work currently
    
    /*
      ICTS-> "set global enables" Test schlaegt fehl.
      lpc_ioctl_clr_host_atn(sla_globals->fd) kehrt nicht zurueck.
      Der Interrupt wird vmtl. sowieso nicht hardwaremaessig geschaltet.
    */
    
    /*
    if (value == 1) {
        lpc_ioctl_set_host_atn(sla_globals->fd);
    } else {
        lpc_ioctl_clr_host_atn(sla_globals->fd);
    }
    */
}


/**
 * Called by the bmccore/msgrouter to send messages to the smi
 */
static int smi_receive_bmc_msg(imsg_t *imsg) {
    char transmitData[PP_BMC_SMI_MAX_MSG_LEN];
    int rv;
    
    transmitData[0] = (imsg->netfn<<2) | (imsg->rs_lun & 0x03);
    transmitData[1] = imsg->cmd;
    memcpy(transmitData + KCS_HEADER_SIZE, imsg->data, imsg->data_size);

#if PP_BMC_DEBUG_LEVEL >= 8
    {
        unsigned int i;
        printf("Sending <");
        for (i=0; i<(imsg->data_size + KCS_HEADER_SIZE); i++) {
            printf("%2x ", transmitData[i]);
        }
        printf(">\n");
    }
#endif

    rv = write(sla_globals->fd, transmitData, imsg->data_size + KCS_HEADER_SIZE);
    // We cannot recover from problems here, so at least print an error
    if (rv != (int)(imsg->data_size + KCS_HEADER_SIZE)) {
        pp_bmc_log_error("[SMIkcs] could not send message, write failed - %d", rv);
    }
    
    pp_bmc_imsg_delete(imsg);
    
    return PP_SUC;
}

/**
 * called by the selector to indicate that new messages are available on the
 * si. The message is read and forwarded to the core.
 */
static int smi_receive_si_msg(UNUSED const int item_id, const int fd, UNUSED const short event, UNUSED void *context) {
    char receiveData[PP_BMC_SMI_MAX_MSG_LEN];
    int rv;
    imsg_t *msgBMC;
    
    rv = read(fd, receiveData, PP_BMC_SMI_MAX_MSG_LEN);

    if (rv < 0) {
        pp_bmc_log_warn("[SMIkcs] read failed: %d", rv);
        return PP_ERR;
    }

    if (rv == PP_BMC_SMI_MAX_MSG_LEN) {
        pp_bmc_log_warn("[SMIkcs] got oversized message (more than %d bytes)", rv);
    }
    
#if PP_BMC_DEBUG_LEVEL >= 8
    if ((rv > 0) & (rv < PP_BMC_SMI_MAX_MSG_LEN)) {
        int i;
        printf("Received <");
        for (i=0; i<rv; i++) {
            printf("%2x ", receiveData[i]);
        }
        printf(">\n");
    }
#endif
    
    // convert internal message from si data
    if (rv < KCS_HEADER_SIZE) {
        pp_bmc_log_warn("[SMIkcs] ignored too short message");
        return PP_ERR;
    }
    
    msgBMC = pp_bmc_imsg_new(rv - KCS_HEADER_SIZE);
    msgBMC->netfn = receiveData[0] >> 2;
    msgBMC->rq_lun = receiveData[0] & 3;
    msgBMC->cmd = receiveData[1];
    memcpy(msgBMC->data, receiveData + KCS_HEADER_SIZE, msgBMC->data_size);
    // add other information
    msgBMC->chan = IPMI_CHAN_SI;
    msgBMC->priv_level = IPMI_PRIV_ADMIN;
    msgBMC->session = NULL;
    msgBMC->buf = NULL;
    msgBMC->buf_size = 0;
    msgBMC->rs_lun = 0;
    
    pp_bmc_core_handle_msg(msgBMC);
    
    return 0;
}


static const pp_bmc_router_chan_adapter_info_t si_kcs_chan_info = {
    .medium =       IPMI_CHAN_MED_SI,
    .proto =        IPMI_CHAN_PROT_KCS,
    .sess_supp =    IPMI_CHAN_SESS_LESS,
};

int pp_smi_kcs_init() {
    // use malloc here if structure gets too big
    sla_globals = &my_sla_globals;
    
    pp_bmc_log_debug("[SMIkcs] initializing openIpmi SI loop adapter");
    
    if (pp_bmc_router_reg_chan(IPMI_CHAN_SI, smi_receive_bmc_msg, NULL, NULL, &si_kcs_chan_info) == NULL) {
        pp_bmc_log_error("[SMIkcs] could not register channel");
        return PP_ERR;
    }
    
    sla_globals->fd = open(KCS_DEV,O_RDWR);
    if (sla_globals->fd <= 0) {
        pp_bmc_log_error("[SMIkcs] cannot open %s. module load failed", KCS_DEV);
        return PP_ERR;
    }

    unsigned short port;
    pp_cfg_get_short(&port, "bmc.smi_base_port");
    if (lpc_ioctl_init(sla_globals->fd, LPC_MODE_KCS, 3, port)) {
        pp_bmc_log_error("[SMIkcs] cannot switch to kcs mode. module load failed");
        return PP_ERR;
    }

    sla_globals->selID = pp_select_add_fd(sla_globals->fd, POLLIN, smi_receive_si_msg, NULL);
    if (sla_globals->selID < 0) {
        pp_bmc_log_error("[SMIkcs] Could not register %s in selector. module load failed", KCS_DEV);
        return PP_ERR;
    }

    if (pp_bmc_router_register_nc_interrupt(bmc_set_host_atn) == PP_ERR) {
        pp_bmc_log_error("[SMIkcs] could not register sms_atn interrupt actor");
        return PP_ERR;
    }

    pp_bmc_log_info("[SMIkcs] adapter started");
    return PP_SUC;
 }

void pp_smi_kcs_cleanup()  {
    pp_bmc_router_unreg_chan(IPMI_CHAN_SI);
    pp_select_remove_fd(sla_globals->selID);
    pp_bmc_router_register_nc_interrupt(NULL);
    close(sla_globals->fd);
    pp_bmc_log_info("[SMIkcs] adapter shut down");
}
