#!/bin/bash
###############################################################################
# the default operations

pp_clean_default () {
    [ -r Makefile -o -r makefile ] && {
	make ${MAKEOPTS} distclean \
	|| make ${MAKEOPTS} clobber \
	|| make ${MAKEOPTS} clean
    }
    [ -r Makefile.in ]    && rm -f Makefile
    [ -d autom4te.cache ] && rm -rf autom4te.cache
    return 0
}

pp_build_default () {
    [ -r Makefile -o -r makefile ] && make ${MAKEOPTS} install
}

##
# links the contents of a directory and its subdirectores
# to a corresponding directory in the global include directory
# argument #1: where to begin (usually include/pp)
pp_link_recursive () {
    actdir=$1            # relative path to include directory
    curdir=`pwd -P`      # the prefix to $actdir
    
    mkdir -p "$FW_TOPDIR/$actdir"
    for fp in $actdir/*; do
	file="`basename $fp`"
	if [ "$file" != "CVS" ]; then
	    if [ -d "$actdir/$file" ]; then
		pp_link_recursive $actdir/$file
	    else
		rm -f "$FW_TOPDIR/$actdir/$file"
		ln -s "$curdir/$actdir/$file" "$FW_TOPDIR/$actdir/$file"
	    fi
	fi
    done
    return 0
}

pp_linkincl_default () {
    currdir=`pwd -P`
    if [ -n "$PP_PUBLIC_KERNEL_INCLUDES" -o \
	 -n "$PP_PUBLIC_INCLUDES" -o \
	 -n "$RARITAN_INCLUDES" ]; then
	if [ -n "$PP_PUBLIC_KERNEL_INCLUDES" ]; then
	    include_dir="$FW_TOPDIR/include/kernel"
	    include_src=$PP_PUBLIC_KERNEL_INCLUDES
	elif [ -n "$PP_PUBLIC_INCLUDES" ]; then
	    include_dir="$FW_TOPDIR/include"
	    include_src=$PP_PUBLIC_INCLUDES
	elif [ -n "$RARITAN_INCLUDES" ]; then
	    include_dir="$FW_TOPDIR/include/rr"
	    include_src=$RARITAN_INCLUDES
	fi
	mkdir -p "$include_dir"
	for fp in $include_src; do
	    file="`basename $fp`"
	    if [ "$file" != "CVS" ]; then
		# note: in SDKs includes are copied. Add -r to the rm to ensure that
		# the copied directory is deleted and relinked correctly. The -r won't
		# hurt when deleting symlinks for regular building
		rm -rf "$include_dir/$file"
		# note: link referenced directories as whole, do not link contained
		# files separately. this is needed by ucd-snmp
		ln -s "$currdir/$fp" "$include_dir/$file"
	    fi
	done
    fi
    if [ -d src/include/pp ]; then
	pushd src > /dev/null
	# Links include/pp and all its subdirectory-contents
	pp_link_recursive include/pp
	popd > /dev/null
    elif [ -r "src/`basename $currdir`.h" ]; then
	mkdir -p "$FW_TOPDIR/include"
	file="`basename $currdir`.h"
	rm -f "$FW_TOPDIR/include/$file"
	ln -s "$currdir/src/$file" "$FW_TOPDIR/include/$file"
    fi
    return 0
}

pp_copyincl_default () {
    incl_destdir="$1"
    if [ -z "$incl_destdir" ]; then
	echo "pp_copyincl_default(): No include destination directory specified!"
	return 1
    elif ! $(echo "$incl_destdir" | grep -q '^[[:space:]]*/') ; then
	echo "pp_copyincl_default(): The include destination must be absolute!"
	return 1
    fi

    currdir=`pwd -P`
    if [ -n "$PP_PUBLIC_INCLUDES" ]; then
	incl_src="$PP_PUBLIC_INCLUDES"
    elif [ -n "$RARITAN_INCLUDES" ]; then
	incl_src="$RARITAN_INCLUDES"
	incl_destdir="$incl_destdir/rr"
    elif [ -d src/include/pp ]; then
	incl_src="src/include/pp"
    elif [ -r "src/`basename $currdir`.h" ]; then
	incl_src="src/`basename $currdir`.h"
    else
	# no includes - return
	return 0
    fi

    mkdir -p "$incl_destdir"

    for fp in $incl_src; do
	basename=`basename "$fp"`
	dirname=`dirname "$fp"`
	if [ -d "$dirname" ]; then
	    pushd "$dirname" > /dev/null
		e=`find "$basename" '(' -name CVS -o -name '*~' -o -name '.?*' ')' -prune \
		    -o ! -type d \
		    '(' -exec cp --parent --remove-destination "{}" "$incl_destdir" ';' \
			-o -print ')' || echo error`
	    popd > /dev/null
	    [ -n "$e" ] && return 1
	else
	    echo "pp_copyincl_default(): '$dirname' does not exist!"
	    return 1
	fi
    done
    return 0
}

#
###############################################################################
# internal commands

# this function multiplexes between a local or the default clean operation
_pp_clean () {
    rm -f .Build.stamp
    if [ "`type -t pp_clean_local`" = "function" ]; then
	pp_clean_local "$@"
    else
	pp_clean_default "$@"
    fi
    # we do not care about the return value of pp_clean_{local,default}
    return 0
}

# this function multiplexes between a local or the default build operation
_pp_build () {
    if [ ! -f .Build.stamp -o Build -nt .Build.stamp ]; then
	PP_FORCE_FULL_BUILD=1
    else
	PP_FORCE_FULL_BUILD=0
    fi
    touch -r Build .Build.stamp
    if [ "`type -t pp_build_local`" = "function" ]; then
	pp_build_local "$@"
    else
	pp_build_default "$@"
    fi
}

# this function multiplexes between a local or the default linkincl operation
_pp_linkincl () {
    if [ "`type -t pp_linkincl_local`" = "function" ]; then
	pp_linkincl_local "$@"
    else
	pp_linkincl_default "$@"
    fi
}

# this function multiplexes between a local or the default copyincl operation
_pp_copyincl () {
    if [ "`type -t pp_copyincl_local`" = "function" ]; then
	pp_copyincl_local "$@"
    else
	pp_copyincl_default "$@"
    fi
}

#
###############################################################################
# the main multiplexer function

pp_do () {
    operation="${1:-build}"
    shift
    case "$operation" in
	clean)
	    _pp_clean "$@"
	    ;;
	build)
	    # some includes might exist only after the build
	    ( set -e;  _pp_build "$@"; _pp_linkincl "$@" )
	    ;;
	cleanbuild)
	    _pp_clean "$@"
	    # some includes might exist only after the build
	    ( set -e;  _pp_build "$@"; _pp_linkincl "$@" )
	    ;;
	linkincl)
	    ( set -e; _pp_linkincl "$@" )
	    ;;
	copyincl)
	    ( set -e ; _pp_copyincl "$@" )
	    ;;
	*)
	    echo "Invalid operation '$operation' specified!"
	    return 1
    esac
    return $?
}

#
###############################################################################
