/* 
 * Copyright (C) 2000 Manuel Novoa III
 * Copyright (C) 2002 Erik Andersen
 *
 * This is a crude wrapper to use uClibc with gcc.
 * It was originally written to work around ./configure for ext2fs-utils.
 * It certainly can be improved, but it works for me in the normal cases.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/wait.h>

static char static_linking[]	= "-static";
static char nostdinc[]		= "-nostdinc";
static char nostartfiles[]	= "-nostartfiles";
static char nodefaultlibs[]	= "-nodefaultlibs";
static char nostdlib[]		= "-nostdlib";
static char nostdinc_plus[]	= "-nostdinc++";

static char uClibc_include_dir[]= __UCLIBC_DEVEL_PREFIX__ "/include/";
/* when uClibc is not using constructors/destructors then use crt0.o! */
static char crt1_path[]		= __UCLIBC_DEVEL_PREFIX__ "/lib/crt1.o";
static char crti_path[]		= __UCLIBC_DEVEL_PREFIX__ "/lib/crti.o";
static char crtn_path[]		= __UCLIBC_DEVEL_PREFIX__ "/lib/crtn.o";
static char gcrt1_path[]	= __UCLIBC_DEVEL_PREFIX__ "/lib/gcrt1.o";
static char crtbegin_path[]	= __WRAPPER_LIBGCC_DIR__ "/crtbegin.o";
static char crtbeginS_path[]	= __WRAPPER_LIBGCC_DIR__ "/crtbeginS.o";
static char crtend_path[]	= __WRAPPER_LIBGCC_DIR__ "/crtend.o";
static char crtendS_path[]	= __WRAPPER_LIBGCC_DIR__ "/crtendS.o";
static char gcc_include_dir[]	= __WRAPPER_GCC_INCLUDE_DIR__;
static char libc_spec[]		= "-lc";
static char libm_spec[]		= "-lm";
static char libpthread_spec[]	= "-lpthread";
static char libstdcpp_spec[]	= "-lstdc++";
static char libsupcpp_spec[]	= "-lsupc++";
static char libgcc_a_spec[]	= "-lgcc";
static char libgcc_eh_spec[]	= "-lgcc_eh";
static char libgcc_s_spec[]	= __WRAPPER_LIBGCC_S_SPEC__;
static char ldso_spec[]		= __UCLIBC_DEVEL_PREFIX__ "/lib/ld-uClibc.so.0";
static int is_arm = 0;

/* Include a local implementation of basename, since this
 * uses the host system's C lib, and CYGWIN apparently
 * doesn't provide an implementation of basename(). */
char *basename(const char *path)
{
    register const char *s;
    register const char *p;
    p = s = path;
    while (*s) {
	if (*s++ == '/') {
	    p = s;
	}
    }
    return (char *) p;
}


int call_distcc(char *const argv[])
{
    int argc = 0,ret;
    char **temp;

    temp = (char**)argv;
    while (*temp++) ++argc;

    temp = (char**)calloc(argc + 2, sizeof(char*));

    for (; argc >= 0; --argc) {
	temp[argc + 1] = argv[argc];
    }

    temp[0] = "distcc";
    ret = execvp("distcc", temp);
    free(temp);
    return ret;
}


int main(int argc, char **argv)
{
    int linking = 1, use_static_linking = 0;
    int use_stdinc = 1, use_start = 1, use_stdlib = 1, use_pic = 0;
    int source_cnt = 0, library_cnt = 0, libpath_cnt = 0;
    int gcc_argv_cnt = 0, gcc_argument_cnt = 0, verbose = 0;
    int ctor_dtor = 1, cplusplus = 0, use_nostdinc_plus = 0;
    int profile = 0, pthread = 0;
    int i, len;
    char ** gcc_argv;
    char ** gcc_argument;
    char ** libraries;
    char ** libpath;
    char *dlstr = "-Wl,--dynamic-linker," __WRAPPER_DYNAMIC_LINKER__;
    char *application_name;
    char *compiler;

    application_name = basename(argv[0]);
    if (application_name[0] == '-') application_name++;

    len = strlen(application_name);
    if (!strcmp(application_name+len-3, "gcc")
	|| !strcmp(application_name+len-2, "cc")) {
	compiler = __WRAPPER_GCC_BIN__;
    } else if (!strcmp(application_name+len-3, "g++")
	       || !strcmp(application_name+len-3, "c++")) {
	cplusplus = 1;
	use_nostdinc_plus = 1;
	compiler = __WRAPPER_GXX_BIN__;
    } else if (!strcmp(application_name+len-2, "ld")) {
	fprintf(stderr, "%s: The gcc wrapper should not be used as a direct ld wrapper!\n", application_name);
	exit(1);
    } else  {
	fprintf(stderr, "%s: Not supported wrap mode!\n", application_name);
	exit(1);
    }

    if (!strcmp(__WRAPPER_GCC_TARGET_MACHINE__, "arm-linux")) {
	is_arm = 1;
    }
    
    library_cnt = 0;
    libraries = __builtin_alloca(sizeof(char*) * argc * 2);
    libraries[library_cnt] = NULL;

    libpath_cnt = 0;
    libpath = __builtin_alloca(sizeof(char*) * argc * 2);
    libpath[libpath_cnt] = NULL;

    for (i = 1; i < argc; i++) {
	if (argv[i][0] == '-') { /* option */
	    switch (argv[i][1]) {
		static const char * wl_prefix = "-Wl,--rpath-link,";
		static size_t wl_prefix_size = sizeof(wl_prefix); /* incl. 0 */
	      case 'c':			/* compile or assemble */
	      case 'S':			/* generate assembler code */
	      case 'E':			/* preprocess only */
	      case 'M':			/* generate dependencies */
		  linking = 0;
		  break;
	      case 'L': 		/* library path */
		  /* for every "-L..." we add a "-Wl,--rpath-link,..." */
		  libpath[libpath_cnt++] = argv[i];
		  libpath[libpath_cnt] = NULL;
		  if (argv[i][2] == '\0') {
		      argv[i++] = NULL;
		      libpath[libpath_cnt++] = argv[i];
		      libpath[libpath_cnt] = __builtin_alloca(wl_prefix_size + strlen(argv[i]));
		      strcpy(libpath[libpath_cnt], wl_prefix);
		      strcat(libpath[libpath_cnt], argv[i]);
		      libpath[++libpath_cnt] = NULL;
		  } else {
		      libpath[libpath_cnt] = __builtin_alloca(wl_prefix_size + strlen(argv[i]) - 2);
		      strcpy(libpath[libpath_cnt], wl_prefix);
		      strcat(libpath[libpath_cnt], argv[i] + 2);
		      libpath[++libpath_cnt] = NULL;
		  }
		  argv[i] = NULL;
		  break;
	      case 'l': 		/* library */
		  libraries[library_cnt++] = argv[i];
		  libraries[library_cnt] = NULL;
		  argv[i] = NULL;
		  break;
	      case 'v':			/* verbose */
		  if (argv[i][2] == 0) verbose = 1;
		  printf("Invoked as %s\n", argv[0]);
		  break;
	      case 'n':
		  if (!strcmp(nostdinc, argv[i])) {
		      use_stdinc = 0;
		  } else if (!strcmp(nostartfiles, argv[i])) {
		      ctor_dtor = 0;
		      use_start = 0;
		  } else if (!strcmp(nodefaultlibs, argv[i])) {
		      use_stdlib = 0;
		      argv[i] = NULL;
		  } else if (!strcmp(nostdlib, argv[i])) {
		      ctor_dtor = 0;
		      use_start = 0;
		      use_stdlib = 0;
		  } else if (!strcmp(nostdinc_plus, argv[i])) {
		      if (cplusplus) use_nostdinc_plus = 0;
		  }
		  break;
	      case 's':
		  if (strstr(argv[i], static_linking) != NULL) {
		      use_static_linking = 1;
		  } else if (!strcmp("-shared", argv[i])) {
		      use_start = 0;
		      use_pic = 1;
		  }
		  break;
	      case 'W':			/* -static could be passed directly to ld */
		  if (!strncmp("-Wl,", argv[i], 4)) {
		      if (strstr(argv[i], static_linking) != NULL) {
			  use_static_linking = 1;
		      }
		      if (strstr(argv[i], "--dynamic-linker") != NULL) {
			  dlstr = NULL;
		      }
		  }
		  break;
	      case 'p':
		  if (!strcmp("-pg", argv[i])) {
		      profile = 1;
		  } else if (!strcmp("-pthread", argv[i])) {
		      pthread = 1;
		  }
		  break;
	      case 'f':			/* Check if we are doing PIC */
		  if (!strcmp("-fPIC", argv[i])
		      || !strcmp("-fpic", argv[i])) {
		      use_pic = 1;
		  } else if (!strcmp("-fprofile-arcs",argv[i])) {
		      profile = 1;
		  }
		  break;
	    }
	} else {				/* assume it is an existing source file */
	    ++source_cnt;
	}
    }

    gcc_argv = __builtin_alloca(sizeof(char*) * (argc + 64));
    gcc_argument = __builtin_alloca(sizeof(char*) * (argc + 20));

    gcc_argv[gcc_argv_cnt++] = compiler;
	
    for (i = 1; i < argc; i++) {
	if (argv[i] != NULL) {
	    gcc_argument[gcc_argument_cnt++] = argv[i];
	    gcc_argument[gcc_argument_cnt] = NULL;
	}
    }

    if (use_stdinc && source_cnt) {
	gcc_argv[gcc_argv_cnt++] = nostdinc;
	if (cplusplus) {
	    if (use_nostdinc_plus) gcc_argv[gcc_argv_cnt++] = nostdinc_plus;
	    gcc_argv[gcc_argv_cnt++] = "-isystem";
	    gcc_argv[gcc_argv_cnt++] = __WRAPPER_CXX_INCLUDE_DIR__;
	    gcc_argv[gcc_argv_cnt++] = "-isystem";
	    gcc_argv[gcc_argv_cnt++] = __WRAPPER_CXX_INCLUDE_DIR__ "/" __WRAPPER_GCC_TARGET_MACHINE__;
	    gcc_argv[gcc_argv_cnt++] = "-isystem";
	    gcc_argv[gcc_argv_cnt++] = __WRAPPER_CXX_INCLUDE_DIR__ "/backward";
	}
	gcc_argv[gcc_argv_cnt++] = "-isystem";
	gcc_argv[gcc_argv_cnt++] = uClibc_include_dir;
	if (is_arm) {
	    gcc_argv[gcc_argv_cnt++] = "-isystem";
	} else {
	    gcc_argv[gcc_argv_cnt++] = "-idirafter";
	}
	gcc_argv[gcc_argv_cnt++] = gcc_include_dir;
	gcc_argv[gcc_argv_cnt++] = "-I" __WRAPPER_FIRMWARE_INCLUDE_DIR__;
    }

    if (linking && source_cnt) {

	gcc_argv[gcc_argv_cnt++] = nostdlib;
	if (use_static_linking) {
	    gcc_argv[gcc_argv_cnt++] = static_linking;
	} else if (dlstr) {
	    gcc_argv[gcc_argv_cnt++] = dlstr;
	}
	for (i = 0 ; i < libpath_cnt ; i++) {
	    if (libpath[i]) gcc_argv[gcc_argv_cnt++] = libpath[i];
	}
	gcc_argv[gcc_argv_cnt++] = "-L" __WRAPPER_FIRMWARE_INSTALL_ROOT__ "/lib";
	gcc_argv[gcc_argv_cnt++] = "-Wl,--rpath-link," __WRAPPER_FIRMWARE_INSTALL_ROOT__ "/lib";
	gcc_argv[gcc_argv_cnt++] = "-L" __WRAPPER_FIRMWARE_INSTALL_ROOT__ "/usr/lib";
	gcc_argv[gcc_argv_cnt++] = "-Wl,--rpath-link," __WRAPPER_FIRMWARE_INSTALL_ROOT__ "/usr/lib";
	
	if (profile) gcc_argv[gcc_argv_cnt++] = gcrt1_path;

	if (ctor_dtor) {
	    gcc_argv[gcc_argv_cnt++] = crti_path;
	    gcc_argv[gcc_argv_cnt++] = use_pic ? crtbeginS_path : crtbegin_path;
	}

	if (use_start && !profile) gcc_argv[gcc_argv_cnt++] = crt1_path;

	for (i = 0 ; i < gcc_argument_cnt ; i++) {
	    if (gcc_argument[i]) gcc_argv[gcc_argv_cnt++] = gcc_argument[i];
	}

	for (i = 0 ; i < libpath_cnt ; i++) {
	    if (libpath[i]) gcc_argv[gcc_argv_cnt++] = libpath[i];
	}

	for (i = 0 ; i < library_cnt; i++) {
	    if (libraries[i]) gcc_argv[gcc_argv_cnt++] = libraries[i];
	}

	if (use_stdlib) {
	    if (cplusplus) {
		char * cxx_link_stdcpp = getenv("CXX_LINK_STDCPP");
		if (cxx_link_stdcpp && !strcmp(cxx_link_stdcpp, "1")) {
		    gcc_argv[gcc_argv_cnt++] = libstdcpp_spec;
		} else {
		    gcc_argv[gcc_argv_cnt++] = libsupcpp_spec;
		}
		gcc_argv[gcc_argv_cnt++] = libm_spec;
	    }
	    if (use_static_linking) {
		gcc_argv[gcc_argv_cnt++] = "-Wl,--start-group";
	    } else if (cplusplus) {
		gcc_argv[gcc_argv_cnt++] = libgcc_s_spec;
	    }
	    gcc_argv[gcc_argv_cnt++] = libgcc_a_spec;

	    if (!is_arm && (use_static_linking || !cplusplus)) gcc_argv[gcc_argv_cnt++] = libgcc_eh_spec;
	    if (pthread) gcc_argv[gcc_argv_cnt++] = libpthread_spec;
	    gcc_argv[gcc_argv_cnt++] = libc_spec;
	    if (use_static_linking) {
		gcc_argv[gcc_argv_cnt++] = "-Wl,--end-group";
	    } else {
	    	gcc_argv[gcc_argv_cnt++] = ldso_spec;
		if (cplusplus) gcc_argv[gcc_argv_cnt++] = libgcc_s_spec;
		gcc_argv[gcc_argv_cnt++] = libgcc_a_spec;
		if (!is_arm && (use_static_linking || !cplusplus)) gcc_argv[gcc_argv_cnt++] = libgcc_eh_spec;
	    }
	}

	if (ctor_dtor) {
	    gcc_argv[gcc_argv_cnt++] = use_pic ? crtendS_path : crtend_path;
	    gcc_argv[gcc_argv_cnt++] = crtn_path;
	}

    } else {

	for (i = 0 ; i < gcc_argument_cnt ; i++) {
	    if (gcc_argument[i]) gcc_argv[gcc_argv_cnt++] = gcc_argument[i];
	}

    }
    gcc_argv[gcc_argv_cnt++] = NULL;

    if (verbose) {
	for (i = 0 ; gcc_argv[i] ; i++) {
	    printf("arg[%2i] = %s\n", i, gcc_argv[i]);
	}
	fflush(stdout);
    }
#ifdef __WRAPPER_USE_DISTCC__
    if (verbose) printf("distcc call\n");
    call_distcc(gcc_argv);
#else
    if (verbose) printf("normal call\n");
    execvp(compiler, gcc_argv);
#endif
    fprintf(stderr, "%s: %s\n", compiler, strerror(errno));
    exit(1);
}
