// system includes

// firmware includes
#include <pp/um.h>

// local includes
#include "cfg_conv.h"

static const char *priv_lvl_key = "user[%u].ipmi_priv_level";
static const char *use_sol_key = "user[%u].ipmi_may_use_sol";

static int convert_ipmi_uid_to_gid(int uid, int gid) {
    char *tmp;
    int ret = PP_SUC;
    
    if(PP_ERR != pp_cfg_get_nodflt(&tmp, priv_lvl_key, uid)) {
        ret |= pp_cfg_set(tmp, "group[%u].acl[ipmi_priv]", gid);
        free(tmp);
        pp_cfg_remove(priv_lvl_key, uid);
    }
    if(PP_ERR != pp_cfg_get_nodflt(&tmp, use_sol_key, uid)) {
        ret |= pp_cfg_set(tmp, "group[%u].acl[ipmi_sol]", gid);
        free(tmp);
        pp_cfg_remove(use_sol_key, uid);
    }
    return ret;
}

static int convert_all_users(void) {
    int ret = PP_SUC;
    vector_t *all_users = pp_um_get_all_users();
    unsigned int u, all_users_sz = vector_size(all_users);
    
    for(u = 0; u < all_users_sz; ++u) {
        int uid, gid;
        char *username = vector_get(all_users, u);
        char *groupname = NULL;
        pp_um_group_state_t state;
        
        /* TODO: now this is very similar to pp_um_ipmi_user_set_permission
                 decide if we could reuse parts of that */
        if(PP_ERR == (uid = pp_um_user_get_uid(username)) ||
           PP_ERR == (gid = pp_um_user_get_gid(username)) ||
           PP_ERR == pp_cfg_get_nodflt(&groupname, "group[%u].name", gid) ||
           PP_ERR == (ret = pp_um_group_is_individual_for_user(groupname,
                                                               username,
                                                               &state))) {
            free(groupname);
            ret = PP_ERR;
            break;
        }
        
        if(state == PP_UM_GROUP_GENERIC) {
            /* group is generic, check if current user is the only one in it */
            vector_t *users;
            int users_sz;
            
            if(NULL == (users = pp_um_get_all_users_by_gid(gid))) {
                ret = PP_ERR;
                free(groupname);
                break;
            }
            
            users_sz = vector_size(users);
            vector_delete(users);
            
            assert(users_sz > 0); // at least current user!
            
            if(users_sz > 1) {
                /* we know, the current user is member of its group assigned.
                   if users_sz > 1, he is not the only one */
    
                char groupbuf[65];
            
                snprintf(groupbuf, sizeof(groupbuf), "%c%s",
                         um_individual_group_prefix, username);
                 
                /* create new individual group of specified name */
                while(1) {
                    if(PP_SUC == pp_um_group_copy(groupname, groupbuf)) {
                        break;
                    } else if (errno != PP_UM_ERR_GROUP_ALREADY_EXISTS || 
                               sizeof(groupbuf) < strlen(groupbuf) + 1) {
                        /* maximum size of groupname reached, break */
                        errno = PP_UM_ERR_INTERNAL_ERROR;
                        goto bail;
                    } else {
                        /* group of that name exists, append '_' to groupname
                           and try again */
                        snprintf(groupbuf, sizeof(groupbuf), "%s_", groupbuf);
                    }
                }
                
                /* set gid to new group */
                if(PP_ERR == (gid = pp_um_group_get_gid(groupbuf)) ||
                   PP_ERR == pp_um_user_set_group(username, groupbuf)) {
                    assert(0); // may not happen, we just created that group!
                    errno = PP_UM_ERR_INTERNAL_ERROR;
                    goto bail;
                }
            }
        }

        convert_ipmi_uid_to_gid(uid, gid);

        free(groupname);

        continue;
 bail:
        ret = PP_ERR;
        free(groupname);
        break;
    }
    
    vector_delete(all_users);
    
    return ret;
}

int conf_common_um(void) {
    int ret;
    pp_cfg_tx_t *tx;

    /* start transaction, great deeds are to be done! */
    tx = pp_cfg_tx_begin(1);
    
    if(PP_ERR != (ret = pp_um_init())) {
        ret = convert_all_users();
    }

    if (ret == PP_SUC) {
        ret = pp_cfg_tx_commit(tx);
    } else {
        pp_cfg_tx_rollback(tx);
    }

    pp_um_cleanup();

    return ret;
}
