/**
 * dmtf.c
 *
 * Broadcom's Universal Management Protocol
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <netpacket/packet.h>
#include <stdio.h>

#include "dmtf.h"
#include "hex_dump.h"
#include "../../include/pp/base.h"

#define NAME "DMTF"
#define LOG_COLORED
#define LOG_DBG
#include "log.h"

// Ethernet link level packet type
#define ETH_P_BROADCOM  0x8874

// interface revision (part of dmtf header)
#define DMTF_INTF_REV    0x0001

// MII ID of DMTF device
#define DMTF_ID          0x01

// IEEE 802.3 compliant
#define DMTF_MIN_PKT_SIZE    46 + 14
#define DMTF_MAX_PKT_SIZE    1500 + 14

// response codes
#define DMTF_RESP_CMD_COMPLETED  0x00
#define DMTF_RESP_CMD_FAILED     0x01
#define DMTF_RESP_CMD_INVAL      0x02

// reason codes
#define DMTF_REAS_NO_ERR                     0x00
#define DMTF_REAS_SET_SEQ_NO_EXPECTED        0x01
#define DMTF_REAS_PARAM_INVAL                0x02
#define DMTF_REAS_VLAN_TAG_INVAL             0x03
#define DMTF_REAS_MAC_ADDR_ZERO              0x04
#define DMTF_REAS_SET_LINK_OS_CONFICT        0x05
#define DMTF_REAS_SET_LINK_MEDIA_CONFICT     0x06
#define DMTF_REAS_SET_LINK_PARAM_CONFICT     0x07
#define DMTF_REAS_SET_LINK_PWR_MODE_CONFICT  0x08
#define DMTF_REAS_UNKNOWN_CMD                0x09
#define DMTF_REAS_HW_ACCESS_ERR              0x0a
#define DMTF_REAS_NOT_SUPPORTED              0x0b
#define DMTF_REAS_SYSTEM_BUSY                0x0c

// asynchronous event notofications
#define DMTF_AEN_LINK_CHANGE                 0xff
#define DMTF_AEN_RESET_EVENT                 0xfe
#define DMTF_AEN_DRIVER_EVENT                0xfd

// asynchronous event notofication request bits
#define DMTF_AEN_REQ_LINK_CHANGE                 (1 << 0)
#define DMTF_AEN_REQ_RESET_EVENT                 (1 << 1)
#define DMTF_AEN_REQ_DRIVER_EVENT                (1 << 2)

// command types
#define DMTF_CMD_SET_SEQ_NO              0x00
#define DMTF_CMD_GET_VER_ID              0x01
#define DMTF_CMD_GET_PARAMS              0x02
#define DMTF_CMD_GET_PKT_STATS           0x03
#define DMTF_CMD_GET_LINK_STAT           0x04
#define DMTF_CMD_ENABLE_CHAN             0x05
#define DMTF_CMD_DISABLE_CHAN            0x06
#define DMTF_CMD_RESET_CHAN              0x07
#define DMTF_CMD_ENABLE_VLAN             0x08
#define DMTF_CMD_DISABLE_VLAN            0x09
#define DMTF_CMD_ENABLE_BCAST_FILTER     0x0a
#define DMTF_CMD_DISABLE_BCAST_FILTER    0x0b
#define DMTF_CMD_SET_MAC_ADDR            0x0c
#define DMTF_CMD_CLEAR_MAC_ADDR          0x0d
#define DMTF_CMD_GET_DMTF_STATS           0x0e
#define DMTF_CMD_ENABLE_DMTF_FLOW_CTRL    0x0f
#define DMTF_CMD_DISABLE_DMTF_FLOW_CTRL   0x10
#define DMTF_CMD_SET_LINK                0x11
#define DMTF_CMD_ENABLE_MCAST_FILTER     0x12
#define DMTF_CMD_DISABLE_MCAST_FILTER    0x13

// response types
#define DMTF_RESP_FLAG                   0x80


#define DMTF_RECV_RETRIES 3

typedef struct {
    unsigned char dest_addr[6];
    unsigned char src_addr[6];
    unsigned short ether_type_ns;
    unsigned short intf_rev_be16;
    unsigned short seq_no_be16;
    unsigned char cmd_type;
    unsigned char dmtf_id;
    unsigned short len_be16;
    int res1;
    int res2;
}  __attribute__(( packed )) dmtf_header_t;

typedef struct {
    dmtf_header_t hdr;
    union {
        struct {
            unsigned char data[DMTF_MAX_PKT_SIZE - sizeof(dmtf_header_t)];
        } __attribute__(( packed )) req;
        struct {
            unsigned short resp_be16;
            unsigned short reas_be16;
            unsigned char data[DMTF_MAX_PKT_SIZE - sizeof(dmtf_header_t) - 2 * sizeof(unsigned short)];
        } __attribute__(( packed )) rsp;
    };
}  __attribute__(( packed )) dmtf_packet_t;


#ifdef NDEBUG
inline const char* dmtf_resp_to_str(unsigned char resp UNUSED) { return ""; }
inline const char* dmtf_reas_to_str(unsigned char reas UNUSED) { return ""; }
inline const char* dmtf_cmd_to_str(unsigned char cmd UNUSED) { return ""; }
#else /* !NDEBUG */
static const char* dmtf_resp_to_str(unsigned char resp)
{
    switch (resp) {
    case DMTF_RESP_CMD_COMPLETED : return "Command Completed";
    case DMTF_RESP_CMD_FAILED    : return "Command Failed";
    case DMTF_RESP_CMD_INVAL     : return "Command Invalid";
    default                     : return "<unknown>";
    }
}

static const char* dmtf_reas_to_str(unsigned char reas)
{
    switch (reas) {
    case DMTF_REAS_NO_ERR                    : return "No Error";
    case DMTF_REAS_SET_SEQ_NO_EXPECTED       : return "Set Sequence Number Command Expected";
    case DMTF_REAS_PARAM_INVAL               : return "Parameter Is Invalid-Out of Range";
    case DMTF_REAS_VLAN_TAG_INVAL            : return "VLAN Tag Is Invalid";
    case DMTF_REAS_MAC_ADDR_ZERO             : return "MAC Address Is Zero";
    case DMTF_REAS_SET_LINK_OS_CONFICT       : return "Set Link Host OS/Driver Conflict";
    case DMTF_REAS_SET_LINK_MEDIA_CONFICT    : return "Set Link Media Conflict";
    case DMTF_REAS_SET_LINK_PARAM_CONFICT    : return "Set Link Parameter Conflict";
    case DMTF_REAS_SET_LINK_PWR_MODE_CONFICT : return "Set Link Power Mode Conflict";
    case DMTF_REAS_UNKNOWN_CMD               : return "Unknown Command Type";
    case DMTF_REAS_HW_ACCESS_ERR             : return "Link Command Failed-Hardware Access Error";
    case DMTF_REAS_NOT_SUPPORTED             : return "Feature/Capability Not Supported By Current Chip Architecture";
    default                                 : return "<unknown>";
    }
}

static const char* dmtf_cmd_to_str(unsigned char cmd)
{
    switch (cmd & ~DMTF_RESP_FLAG) {
    case DMTF_CMD_SET_SEQ_NO             : return "Set Sequence Number";
    case DMTF_CMD_GET_VER_ID             : return "Get Version ID";
    case DMTF_CMD_GET_PARAMS             : return "Get Parameters";
    case DMTF_CMD_GET_PKT_STATS          : return "Get Controller Packet Statistics";
    case DMTF_CMD_GET_LINK_STAT          : return "Get Link Status";
    case DMTF_CMD_ENABLE_CHAN            : return "Enable Channel";
    case DMTF_CMD_DISABLE_CHAN           : return "Disable Channel";
    case DMTF_CMD_RESET_CHAN             : return "Reset Channel";
    case DMTF_CMD_ENABLE_VLAN            : return "Enable VLAN";
    case DMTF_CMD_DISABLE_VLAN           : return "Disable VLAN";
    case DMTF_CMD_ENABLE_BCAST_FILTER    : return "Enable Broadcast Filtering";
    case DMTF_CMD_DISABLE_BCAST_FILTER   : return "Disable Broadcast Filtering";
    case DMTF_CMD_SET_MAC_ADDR           : return "Set MAC Address";
    case DMTF_CMD_CLEAR_MAC_ADDR         : return "Clear MAC Address";
    case DMTF_CMD_GET_DMTF_STATS          : return "Get DMTF Interface Statistics";
    case DMTF_CMD_ENABLE_DMTF_FLOW_CTRL   : return "Enable DMTF Flow Control";
    case DMTF_CMD_DISABLE_DMTF_FLOW_CTRL  : return "Disable DMTF Flow Control";
    case DMTF_CMD_SET_LINK               : return "Set Link";
    case DMTF_CMD_ENABLE_MCAST_FILTER    : return "Enable Multicast Filtering";
    case DMTF_CMD_DISABLE_MCAST_FILTER   : return "Disable Multicast Filtering";
    default                             : return "<unknown>";
    }
}
#endif /* !NDEBUG */

static void dmtf_dump_pkt(const dmtf_packet_t *pkt, size_t max_size)
{
    const dmtf_header_t *hdr = &pkt->hdr;
    if (hdr->cmd_type & DMTF_RESP_FLAG) {
        unsigned short resp = be16_to_cpu(pkt->rsp.resp_be16);
        unsigned short reas = be16_to_cpu(pkt->rsp.reas_be16);
        INFO("Dest=%02x:%02x:%02x:%02x:%02x:%02x Src=%02x:%02x:%02x:%02x:%02x:%02x "
             "Ether=%04x IntfRev=0x%04x SeqNo=%d Cmd=0x%02x(%s) UmpId=%d Len=%d"
             " RespCode=0x%02x(%s) ReasCode=0x%02x(%s)",
            hdr->dest_addr[0], hdr->dest_addr[1], hdr->dest_addr[2],
            hdr->dest_addr[3], hdr->dest_addr[4], hdr->dest_addr[5],
            hdr->src_addr[0], hdr->src_addr[1], hdr->src_addr[2],
            hdr->src_addr[3], hdr->src_addr[4], hdr->src_addr[5],
            ntohs(hdr->ether_type_ns),
            be16_to_cpu(hdr->intf_rev_be16),
            be16_to_cpu(hdr->seq_no_be16),
            hdr->cmd_type, dmtf_cmd_to_str(hdr->cmd_type),
            hdr->dmtf_id,
            be16_to_cpu(hdr->len_be16),
            resp, dmtf_resp_to_str(resp),
            reas, dmtf_reas_to_str(reas));
    } else {
        INFO("Dest=%02x:%02x:%02x:%02x:%02x:%02x Src=%02x:%02x:%02x:%02x:%02x:%02x "
             "Ether=%04x IntfRev=0x%04x SeqNo=%d Cmd=0x%02x(%s) UmpId=%d Len=%d",
            hdr->dest_addr[0], hdr->dest_addr[1], hdr->dest_addr[2],
            hdr->dest_addr[3], hdr->dest_addr[4], hdr->dest_addr[5],
            hdr->src_addr[0], hdr->src_addr[1], hdr->src_addr[2],
            hdr->src_addr[3], hdr->src_addr[4], hdr->src_addr[5],
            ntohs(hdr->ether_type_ns),
            be16_to_cpu(hdr->intf_rev_be16),
            be16_to_cpu(hdr->seq_no_be16),
            hdr->cmd_type, dmtf_cmd_to_str(hdr->cmd_type),
            hdr->dmtf_id,
            be16_to_cpu(hdr->len_be16));
    }

    max_size = max_size;
//    hex_dump(pkt, min(sizeof(dmtf_header_t) + be16_to_cpu(hdr->len_be16), max_size), 1, 0);
}

static unsigned int dmtf_calc_chksum(const dmtf_packet_t *pkt, int len)
{
    assert((len & 1) == 0);
    unsigned int chksum = 0;
    const unsigned short *ptr = (const unsigned short*)pkt; // checksum is build of 16bit values
    int sz = len / sizeof(unsigned short);
    // for DMTF the checksum is build only on DMTF heade + payload
    // (ETH header is left out)
    ptr += 7; sz -= 7;
    while (--sz >= 0) chksum += be16_to_cpu(*(ptr++)); // calc sum
    return -chksum; // create 2's complement
}

static int dmtf_fill_hdr(unsigned char dmtf_id,
                        unsigned char cmd,
                        unsigned short seq_no,
                        unsigned short data_len,
                        dmtf_packet_t *pkt,
                        size_t *pkt_len)
{
    // pkt size = header + data + crc
    if (sizeof(dmtf_header_t) + data_len + sizeof(unsigned long) > DMTF_MAX_PKT_SIZE) {
        return PP_ERR; // data size too long
    }

    // fill header
    dmtf_header_t *hdr = &pkt->hdr;
    memset(hdr->dest_addr, 0xff, 6);
    memset(hdr->src_addr, 0x00, 6);
    hdr->ether_type_ns  = htons(ETH_P_BROADCOM);
    hdr->intf_rev_be16  = cpu_to_be16(DMTF_INTF_REV);
    hdr->seq_no_be16    = cpu_to_be16(seq_no);
    hdr->cmd_type       = cmd;
    hdr->dmtf_id         = dmtf_id;
    hdr->len_be16       = cpu_to_be16(data_len);
    hdr->res1           = 0;
    hdr->res2           = 0;

    // cmd args
    unsigned char *payload = (unsigned char*)(hdr + 1);

    // word padding
    unsigned int pad_len = (4 - data_len) & 3;
    memset(payload + data_len, 0, pad_len);

    // calc crc
    unsigned int *chksum_be32 = (unsigned int*)(payload + data_len + pad_len);
    *chksum_be32 = cpu_to_be32(dmtf_calc_chksum(pkt, sizeof(dmtf_header_t) + data_len + pad_len));

    // fill up to min IEEE 802.3's packet size of 46 bytes
    unsigned char *pad = (unsigned char*)(chksum_be32 + 1);
    int cnt = pad - (unsigned char*)pkt;
    if (cnt < DMTF_MIN_PKT_SIZE) {
        memset(pad, 0, DMTF_MIN_PKT_SIZE - cnt); // packet padding
        cnt = DMTF_MIN_PKT_SIZE;
    }

    *pkt_len = cnt;
    return PP_SUC;
}

struct dmtf_data {
    int sock;
    struct sockaddr_ll addr;
    unsigned short seq_no;
} g_dmtf = {
    .sock = -1,
};


static int dmtf_sendrecv(dmtf_data_t* dmtf, unsigned char cmd,
                        dmtf_packet_t* req, int req_data_len,
                        dmtf_packet_t* rsp, int rsp_data_len)
{
    assert(dmtf);
    assert(dmtf->sock != -1);

    size_t res, req_len;

    if (PP_FAILED(dmtf_fill_hdr(DMTF_ID, cmd, dmtf->seq_no, req_data_len, req, &req_len))) {
        ERR("request len too large (%d)", req_len);
        return PP_ERR;
    }

    int retry = DMTF_RECV_RETRIES;
    while (retry-- > 0) {
        res = sendto(dmtf->sock, req, req_len, 0, (struct sockaddr*)&dmtf->addr, sizeof(dmtf->addr));
        if (res == (size_t)-1) {
            ERR("sendto() failed: %s (%d)", strerror(errno), errno);
            return PP_ERR;
        } else if (res != req_len) {
            ERR("only %d of %d bytes sent", res, req_len);
            return PP_ERR;
        } else {
            DBG("%d bytes sent", res);
            dmtf_dump_pkt(req, res);
        }

        struct sockaddr_ll from;
        socklen_t fromlen = sizeof(from);

        res = recvfrom(dmtf->sock, rsp, sizeof(*rsp), 0, (struct sockaddr*)&from, &fromlen);
        if (res == (size_t)-1) {
            if (retry > 0) {
                WARN("recvfrom() failed (%d times): %s (%d)",
                     DMTF_RECV_RETRIES - retry, strerror(errno), errno);
                continue;
            } else {
                ERR("recvfrom() failed (%d times): %s (%d)",
                    DMTF_RECV_RETRIES - retry, strerror(errno), errno);
                return PP_ERR;
            }
        } else if (res == 0) {
            ERR("peer shut down connection");
            return PP_ERR;
        } else {
            DBG("%d bytes received (proto=%04x addr=%02x:%02x:%02x:%02x:%02x:%02x)",
                res, ntohs(from.sll_protocol),
                from.sll_addr[0], from.sll_addr[1], from.sll_addr[2],
                from.sll_addr[3], from.sll_addr[4], from.sll_addr[5]);

            if (ntohs(from.sll_protocol) != ETH_P_BROADCOM) {
                ERR("bad Ethernet protocol %04x", ntohs(from.sll_protocol));
                return PP_ERR;
            }

            dmtf_dump_pkt(rsp, res);

            // check cmd and sequence number
            if (rsp->hdr.cmd_type != (req->hdr.cmd_type | DMTF_RESP_FLAG)) {
                ERR("resp cmd type %.2x does not match req cmd type %.2x",
                    rsp->hdr.cmd_type, req->hdr.cmd_type);
                return PP_ERR;
            }
            if (dmtf->seq_no != be16_to_cpu(rsp->hdr.seq_no_be16)) {
                dmtf->seq_no = be16_to_cpu(rsp->hdr.seq_no_be16) + 1; // adjust sequence number
                ERR("seq no %d invalid (should be %d) - locally corrected",
                    be16_to_cpu(rsp->hdr.seq_no_be16), dmtf->seq_no);
                return PP_ERR;
            }

            // response valid so far: inc seq no
            dmtf->seq_no++;

            // check error code
            unsigned short resp = be16_to_cpu(rsp->rsp.resp_be16);
            if (resp != DMTF_RESP_CMD_COMPLETED) {
                ERR("error responsed: %d %s\n", resp, dmtf_resp_to_str(resp));
                return PP_ERR;
            }

            // check data len
            size_t data_len = be16_to_cpu(rsp->hdr.len_be16);
            if (data_len < rsp_data_len + 2 * sizeof(unsigned short)) {
                ERR("response data truncated (%d < %d)\n",
                    data_len, rsp_data_len + 2 * sizeof(unsigned short));
                return PP_ERR;
            }

            // check checksum
            int pad_len = (4 - data_len) & 3;
            int chksum_len = sizeof(dmtf_header_t) + data_len + pad_len;
            unsigned int chksum = be32_to_cpu(*(unsigned int*)((unsigned char*)rsp + chksum_len));
            unsigned int chksum_exp = dmtf_calc_chksum(rsp, chksum_len);
            if (chksum != chksum_exp) {
                ERR("bad checksum (exp=0x%08x, is=0x%08x)", chksum_exp, chksum);
                return PP_ERR;
            }

            // check completion code
            if (be16_to_cpu(rsp->rsp.resp_be16) != DMTF_RESP_CMD_COMPLETED) {
                ERR("%s / %s",
                    dmtf_resp_to_str(be16_to_cpu(rsp->rsp.resp_be16)),
                    dmtf_reas_to_str(be16_to_cpu(rsp->rsp.reas_be16)));
                return PP_ERR;
            }
        }
        break; // leave retry loop
    }

    return PP_SUC;
}

dmtf_data_t* dmtf_open(const char* intf_name)
{
    dmtf_data_t *dmtf = &g_dmtf;

    if (dmtf->sock != -1) {
        ERR("socket() failed");
        goto fail;
    }

    dmtf->sock = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_BROADCOM));
    if (dmtf->sock == -1) {
        ERR("socket() failed: %s (%d)", strerror(errno), errno);
        goto fail;
    }

    // fill addr struct
    {
        int intf = if_nametoindex(intf_name);
        if (intf == 0) {
            ERR("if_nametoindex(\"%s\") failed: %s (%d)", intf_name, strerror(errno), errno);
            goto fail;
        }

        dmtf->addr.sll_family     = AF_PACKET;
        dmtf->addr.sll_protocol   = htons(ETH_P_BROADCOM);
        dmtf->addr.sll_ifindex    = intf;
        dmtf->addr.sll_hatype     = 0;
        dmtf->addr.sll_pkttype    = 0;
        dmtf->addr.sll_halen      = 6;
        memset(dmtf->addr.sll_addr, 0xff, 6);
    }

    // bind to interface eth1
    if (bind(dmtf->sock, (struct sockaddr*)&dmtf->addr, sizeof(dmtf->addr)) < 0) {
        ERR("bind() failed: %s (%d)", strerror(errno), errno);
        goto fail;
    }

    // set socket timeouts
    {
        struct timeval tv = { .tv_sec = 1, .tv_usec = 0, }; // 1s

        if (setsockopt(dmtf->sock, SOL_SOCKET, SO_SNDTIMEO, (void*)&tv, sizeof(tv)) < 0) {
            ERR("setsockopt(SO_SNDTIMEO) failed: %s (%d)", strerror(errno), errno);
            goto fail;
        }

        if (setsockopt(dmtf->sock, SOL_SOCKET, SO_RCVTIMEO, (void*)&tv, sizeof(tv)) < 0) {
            ERR("setsockopt(SO_RCVTIMEO) failed: %s (%d)", strerror(errno), errno);
            goto fail;
        }
    }

    // send Set Sequence Number cmd
    dmtf_cmd_set_seq_no(dmtf, 0, 0);

    return dmtf;
fail:
    if (dmtf->sock != -1) {
        close(dmtf->sock);
        dmtf->sock = -1;
    }
    return NULL;
}

int dmtf_close(dmtf_data_t* dmtf)
{
    assert(dmtf);
    assert(dmtf->sock != -1);

    int sock = dmtf->sock;
    dmtf->sock = -1;
    return close(sock);
};

int dmtf_cmd_set_seq_no(dmtf_data_t* dmtf, unsigned short seq_no, UNUSED unsigned long aen_req_bits)
{
    dmtf_packet_t req, rsp;
    dmtf->seq_no = seq_no;
    // TODO: not supported by DMTF v0.3, but will be added until v1.0
    //*(unsigned long*)req.req.data = aen_req_bits;
    //return dmtf_sendrecv(dmtf, DMTF_CMD_SET_SEQ_NO, &req, sizeof(unsigned long), &rsp, 0);
    return dmtf_sendrecv(dmtf, DMTF_CMD_SET_SEQ_NO, &req, 0, &rsp, 0);
}

int dmtf_cmd_get_ver_id(dmtf_data_t* dmtf, dmtf_ver_id_t* ver_id)
{
    dmtf_packet_t req, rsp;
    struct {
        char name[12];
        unsigned char ver[4];
        unsigned short did_be16;
        unsigned short vid_be16;
        unsigned short sdid_be16;
        unsigned short svid_be16;
    } __attribute__(( packed )) *data = (void*)rsp.rsp.data;

    if (PP_FAILED(dmtf_sendrecv(dmtf, DMTF_CMD_GET_VER_ID, &req, 0, &rsp, sizeof(*data)))) {
        return PP_ERR;
    }

    memcpy(ver_id->name, data->name, 14); ver_id->name[12] = '\0';
    memcpy(ver_id->ver, data->ver, 4);
    ver_id->did  = be16_to_cpu(data->did_be16);
    ver_id->vid  = be16_to_cpu(data->vid_be16);
    ver_id->sdid = be16_to_cpu(data->sdid_be16);
    ver_id->svid = be16_to_cpu(data->svid_be16);
    return PP_SUC;
}

int dmtf_cmd_get_params(dmtf_data_t* dmtf, dmtf_params_t* params)
{
    dmtf_packet_t req, rsp;
    struct {
        unsigned char mac1[6];
        unsigned char mac2[6];
        unsigned long vlan_be32;
        unsigned long link_be32;
        unsigned long filter_be32;
        unsigned long flags_be32;
        //unsigned char host_mac[6]; // not supported by DMTF but by UMP!
    } __attribute__(( packed )) *data = (void*)rsp.rsp.data;

    if (PP_FAILED(dmtf_sendrecv(dmtf, DMTF_CMD_GET_PARAMS, &req, 0, &rsp, sizeof(*data)))) {
        return PP_ERR;
    }

    memcpy(params->mac1, data->mac1, 6);
    memcpy(params->mac2, data->mac2, 6);
    params->vlan   = be32_to_cpu(data->vlan_be32);
    params->link   = be32_to_cpu(data->link_be32);
    params->filter = be32_to_cpu(data->filter_be32);
    params->flags  = be32_to_cpu(data->flags_be32);
    //memcpy(params->host_mac, data->host_mac, 6);
    return PP_SUC;
}

int dmtf_cmd_get_pkt_stats(dmtf_data_t* dmtf, dmtf_pkt_stats_t* pkt_stats)
{
    dmtf_packet_t req, rsp;
    struct {
        unsigned long cnt_be32[39];
    } __attribute__(( packed )) *data = (void*)rsp.rsp.data;

    if (PP_FAILED(dmtf_sendrecv(dmtf, DMTF_CMD_GET_PKT_STATS, &req, 0, &rsp, sizeof(*data)))) {
        return PP_ERR;
    }

    size_t i;
    for (i = 0; i < sizeof(pkt_stats->cnt) / sizeof(pkt_stats->cnt[0]); i++)
        pkt_stats->cnt[i] = be32_to_cpu(data->cnt_be32[i]);
    return PP_SUC;
}

int dmtf_cmd_get_link_stat(dmtf_data_t* dmtf, dmtf_link_stat_t* link_stat)
{
    dmtf_packet_t req, rsp;
    struct {
        unsigned long link_be32;
        unsigned long other_be32;
    } __attribute__(( packed )) *data = (void*)rsp.rsp.data;

    if (PP_FAILED(dmtf_sendrecv(dmtf, DMTF_CMD_GET_LINK_STAT, &req, 0, &rsp, sizeof(*data)))) {
        return PP_ERR;
    }

    link_stat->link  = be32_to_cpu(data->link_be32);
    link_stat->other = be32_to_cpu(data->other_be32);
    return PP_SUC;
}

int dmtf_cmd_enable_chan(dmtf_data_t* dmtf)
{
    dmtf_packet_t req, rsp;
    return dmtf_sendrecv(dmtf, DMTF_CMD_ENABLE_CHAN, &req, 0, &rsp, 0);
}

int dmtf_cmd_disable_chan(dmtf_data_t* dmtf)
{
    dmtf_packet_t req, rsp;
    return dmtf_sendrecv(dmtf, DMTF_CMD_DISABLE_CHAN, &req, 0, &rsp, 0);
}

int dmtf_cmd_reset_chan(dmtf_data_t* dmtf)
{
    dmtf_packet_t req, rsp;
    return dmtf_sendrecv(dmtf, DMTF_CMD_RESET_CHAN, &req, 0, &rsp, 0);
}

int dmtf_cmd_set_mac(dmtf_data_t* dmtf, unsigned char* mac, unsigned char addr)
{
    dmtf_packet_t req, rsp;
    memcpy(req.req.data, mac, 6);
    *(unsigned short*)(req.req.data + 6) = cpu_to_be16(addr);
    return dmtf_sendrecv(dmtf, DMTF_CMD_SET_MAC_ADDR, &req, 8, &rsp, 0);
}

int dmtf_cmd_clr_mac(dmtf_data_t* dmtf, unsigned char addr)
{
    dmtf_packet_t req, rsp;
    *(unsigned short*)(req.req.data + 2) = cpu_to_be16(addr);
    return dmtf_sendrecv(dmtf, DMTF_CMD_CLEAR_MAC_ADDR, &req, 4, &rsp, 0);
}

int dmtf_cmd_get_dmtf_stats(dmtf_data_t* dmtf, dmtf_stats_t* dmtf_stats)
{
    dmtf_packet_t req, rsp;
    struct {
        unsigned long cnt_be32[19];
    } __attribute__(( packed )) *data = (void*)rsp.rsp.data;

    if (PP_FAILED(dmtf_sendrecv(dmtf, DMTF_CMD_GET_DMTF_STATS, &req, 0, &rsp, sizeof(*data)))) {
        return PP_ERR;
    }

    size_t i;
    for (i = 0; i < sizeof(dmtf_stats->cnt) / sizeof(dmtf_stats->cnt[0]); i++)
        dmtf_stats->cnt[i] = be32_to_cpu(data->cnt_be32[i]);
    return PP_SUC;
}
