/*
 * Dropbear - a SSH2 server
 * 
 * Copyright (c) 2002,2003 Matt Johnston
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE. */

/* This file (auth.c) handles authentication requests, passing it to the
 * particular type (auth-passwd, auth-pubkey). */

#include <liberic_notify.h>
#include "includes.h"
#include "dbutil.h"
#include "session.h"
#include "buffer.h"
#include "ssh.h"
#include "packet.h"
#include "auth.h"
#include "runopts.h"

static void authclear();
static int checkusername(unsigned char *username, unsigned int userlen);
static void send_msg_userauth_banner();

/* initialise the first time for a session, resetting all parameters */
void svr_authinitialise() {

	GLOBALS(g);
	g->ses.authstate.failcount = 0;
	authclear();
	
}

/* Reset the auth state, but don't reset the failcount. This is for if the
 * user decides to try with a different username etc, and is also invoked
 * on initialisation */
static void authclear() {
	
	GLOBALS(g);
	memset(&g->ses.authstate, 0, sizeof(g->ses.authstate));
#ifdef ENABLE_SVR_PUBKEY_AUTH
	g->ses.authstate.authtypes |= AUTH_TYPE_PUBKEY;
#endif
#if defined(ENABLE_SVR_PASSWORD_AUTH) || defined(ENABLE_SVR_PAM_AUTH)
	if (!svr_opts.noauthpass) {
		g->ses.authstate.authtypes |= AUTH_TYPE_PASSWORD;
	}
#endif

}

/* Send a banner message if specified to the client. The client might
 * ignore this, but possibly serves as a legal "no trespassing" sign */
static void send_msg_userauth_banner() {

	GLOBALS(g);
	TRACE(("enter send_msg_userauth_banner"))
	if (svr_opts.banner == NULL) {
		TRACE(("leave send_msg_userauth_banner: banner is NULL"))
		return;
	}

	CHECKCLEARTOWRITE();

	buf_putbyte(g->ses.writepayload, SSH_MSG_USERAUTH_BANNER);
	buf_putstring(g->ses.writepayload, buf_getptr(svr_opts.banner,
				svr_opts.banner->len), svr_opts.banner->len);
	buf_putstring(g->ses.writepayload, "en", 2);

	encrypt_packet();
	buf_free(svr_opts.banner);
	svr_opts.banner = NULL;

	TRACE(("leave send_msg_userauth_banner"))
}

/* handle a userauth request, check validity, pass to password or pubkey
 * checking, and handle success or failure */
void recv_msg_userauth_request() {

	unsigned char *username = NULL, *servicename = NULL, *methodname = NULL;
	unsigned int userlen, servicelen, methodlen;
        log_event_t event;	
	GLOBALS(g);

	TRACE(("enter recv_msg_userauth_request"))

	/* ignore packets if auth is already done */
	if (g->ses.authstate.authdone == 1) {
		TRACE(("leave recv_msg_userauth_request: authdone already"))
		return;
	}

	/* send the banner if it exists, it will only exist once */
	if (svr_opts.banner) {
		send_msg_userauth_banner();
	}

	
	username = buf_getstring(g->ses.payload, &userlen);
	servicename = buf_getstring(g->ses.payload, &servicelen);
	methodname = buf_getstring(g->ses.payload, &methodlen);

	/* only handle 'ssh-connection' currently */
	if (servicelen != SSH_SERVICE_CONNECTION_LEN
			&& (strncmp(servicename, SSH_SERVICE_CONNECTION,
					SSH_SERVICE_CONNECTION_LEN) != 0)) {
		
		/* TODO - disconnect here */
		m_free(username);
		m_free(servicename);
		m_free(methodname);
		dropbear_exit("unknown service in auth");
	}

	/* user wants to know what methods are supported */
	if (methodlen == AUTH_METHOD_NONE_LEN &&
			strncmp(methodname, AUTH_METHOD_NONE,
				AUTH_METHOD_NONE_LEN) == 0) {
		TRACE(("recv_msg_userauth_request: 'none' request"))
		send_msg_userauth_failure(0, 0);
		goto out;
	}
	
	/* check username is good before continuing */
	if (checkusername(username, userlen) == DROPBEAR_FAILURE) {
		/* username is invalid/no shell/etc - send failure */
		TRACE(("sending checkusername failure"))
		send_msg_userauth_failure(0, 1);
		goto out;
	}


#ifdef ENABLE_SVR_PASSWORD_AUTH
	if (!svr_opts.noauthpass &&
			!(svr_opts.norootpass && g->ses.authstate.pw->pw_uid == 0) ) {
		/* user wants to try password auth */
		if (methodlen == AUTH_METHOD_PASSWORD_LEN &&
				strncmp(methodname, AUTH_METHOD_PASSWORD,
					AUTH_METHOD_PASSWORD_LEN) == 0) {

			unsigned char * password;
			unsigned int passwordlen;

			//jim@peppercon.de
			//user-management

			int server_sid = 0;

			unsigned int changepw;

			/* check if client wants to change password */
			changepw = buf_getbyte(g->ses.payload);
			if (changepw) {
				/* not implemented by this server */
				send_msg_userauth_failure(0, 1);
				return;
			}

			password = buf_getstring(g->ses.payload, &passwordlen);

			//jim@peppercon.de
			//user-management
			
			event.eventdata.logintrapdata.loginname = username;
			event.eventdata.logintrapdata.remote_host = g->svr_ses.addrstring;
			
			if (PP_ERR == pp_um_user_authenticate_with_ip_str(
                                                          username, password,
                                                          PP_UM_AUTH_NO_FLAGS,
                                                          g->svr_ses.addrstring,
                                                          &server_sid,
                                                          NULL)) {
				dropbear_log(LOG_WARNING,
                                             "'%s' failed to authenticate with %s",
                                             username, g->svr_ses.addrstring);

				event.trapoid = "Loginfailed";
				eric_notify_post_event(&event, "auth", PP_NOTIFY_EVENT_LEGACY); 

				m_burn(password, passwordlen);
				m_free(password);

				send_msg_userauth_failure(0, 1);
			} else if (PP_ERR == pp_um_user_has_permission(
                                                       username, "telnet",
                                                       pp_acl_raasip_yes_str)) {
                            /* NOTE: for convenience reasons, the permission
                                     "telnet" addresses telnet and/or ssh
                                     console dependant on product features */
				dropbear_log(LOG_WARNING,
						"'%s' can not run ssh from %s",
						 username,
						g->svr_ses.addrstring);

				event.trapoid = "Loginfailed";
				eric_notify_post_event(&event, "auth", PP_NOTIFY_EVENT_LEGACY); 

				m_burn(password, passwordlen);
				m_free(password);

				send_msg_userauth_failure(0, 1);
			} else {
				/* successful authentication */
				dropbear_log(LOG_NOTICE, 
						"password auth succeeded for '%s' from %s",
						 username,
						g->svr_ses.addrstring);

				event.trapoid = "Loginsuccess";
				eric_notify_post_event(&event, "auth", PP_NOTIFY_EVENT_LEGACY); 

				m_burn(password, passwordlen);
				m_free(password);

				send_msg_userauth_success();
			} 
		}
		goto out;
	}
#endif

	/* nothing matched, we just fail */
	send_msg_userauth_failure(0, 1);

out:
	m_free(username);
	m_free(servicename);
	m_free(methodname);
}

//jim@peppercon.de
//We check the username and password at the same time
//So, we just fill the environment and return success
//
//
/* Check that the username exists, has a non-empty password, and has a valid
 * shell.
 * returns DROPBEAR_SUCCESS on valid username, DROPBEAR_FAILURE on failure */
//bug, did not use
static int checkusername(unsigned char *username, unsigned int userlen) {

	GLOBALS(g);
	
	TRACE(("enter checkusername"))
	if (userlen > MAX_USERNAME_LEN) {
		return DROPBEAR_FAILURE;
	}

	/* new user or username has changed */
	if (g->ses.authstate.username == NULL ||
	    strcmp(username, g->ses.authstate.username) != 0) {
			struct passwd *pw;
			/* the username needs resetting */
			if (g->ses.authstate.username != NULL) {
				dropbear_log(LOG_WARNING, "client trying multiple usernames from %s",
							g->svr_ses.addrstring);
				m_free(g->ses.authstate.username);
			}
			authclear();
			//g->ses.authstate.pw = getpwnam((char*)username);
			pw = m_malloc(sizeof(struct passwd));
			pw->pw_name = (char*)username;
			pw->pw_passwd = NULL;
			g->ses.authstate.pw = pw;

			g->ses.authstate.username = m_strdup(username);
			m_free(g->ses.authstate.printableuser);
	}

	/* We can set it once we know its a real user */

	g->ses.authstate.printableuser = m_strdup(g->ses.authstate.pw->pw_name);

	return DROPBEAR_SUCCESS;
}

/* Send a failure message to the client, in responds to a userauth_request.
 * Partial indicates whether to set the "partial success" flag,
 * incrfail is whether to count this failure in the failure count (which
 * is limited. This function also handles disconnection after too many
 * failures */
void send_msg_userauth_failure(int partial, int incrfail) {

	buffer *typebuf = NULL;
	GLOBALS(g);

	TRACE(("enter send_msg_userauth_failure"))

	CHECKCLEARTOWRITE();
	
	buf_putbyte(g->ses.writepayload, SSH_MSG_USERAUTH_FAILURE);

	/* put a list of allowed types */
	typebuf = buf_new(30); /* long enough for PUBKEY and PASSWORD */

	if (g->ses.authstate.authtypes & AUTH_TYPE_PUBKEY) {
		buf_putbytes(typebuf, AUTH_METHOD_PUBKEY, AUTH_METHOD_PUBKEY_LEN);
		if (g->ses.authstate.authtypes & AUTH_TYPE_PASSWORD) {
			buf_putbyte(typebuf, ',');
		}
	}
	
	if (g->ses.authstate.authtypes & AUTH_TYPE_PASSWORD) {
		buf_putbytes(typebuf, AUTH_METHOD_PASSWORD, AUTH_METHOD_PASSWORD_LEN);
	}

	buf_setpos(typebuf, 0);
	buf_putstring(g->ses.writepayload, buf_getptr(typebuf, typebuf->len),
			typebuf->len);
	buf_free(typebuf);

	buf_putbyte(g->ses.writepayload, partial ? 1 : 0);
	encrypt_packet();

	TRACE(("auth fail: methods %d, '%s'", g->ses.authstate.authtypes,
				buf_getptr(typebuf, typebuf->len)));

	if (incrfail) {
		usleep(300000); /* XXX improve this */
		g->ses.authstate.failcount++;
	}

	if (g->ses.authstate.failcount >= MAX_AUTH_TRIES) {
		char * userstr;
		/* XXX - send disconnect ? */
		TRACE(("Max auth tries reached, exiting"))

		if (g->ses.authstate.printableuser == NULL) {
			userstr = "is invalid";
		} else {
			userstr = g->ses.authstate.printableuser;
		}
		dropbear_exit("Max auth tries reached - user '%s' from %s",
				userstr, g->svr_ses.addrstring);
	}
	
	TRACE(("leave send_msg_userauth_failure"))
}

/* Send a success message to the user, and set the "authdone" flag */
void send_msg_userauth_success() {

	GLOBALS(g);
	TRACE(("enter send_msg_userauth_success"))

	CHECKCLEARTOWRITE();

	buf_putbyte(g->ses.writepayload, SSH_MSG_USERAUTH_SUCCESS);
	encrypt_packet();

	g->ses.authstate.authdone = 1;
	g->ses.connecttimeout = 0;


	if (g->ses.authstate.pw->pw_uid == 0) {
		g->ses.allowprivport = 1;
	}

	g->authenticated = 1;

	TRACE(("leave send_msg_userauth_success"))

}
