package nn.pp.drvredir;

import java.lang.*;
import java.io.*;
import java.net.*;
import java.security.*;
import nn.pp.rc.*;

class DrvRedirNative implements DriveAccess {
    /* all native methods */
    native boolean isSupportedByOSVersion();
    private native String cdExtGetAvailableDrives();
    private native long cdExtractionInit();
    private native void cdExtractionCleanup(long cdExt);
    private native boolean cdExtMediumChangeSupported(long cdExt);
    private native String getErrorString(int error);
    private native int cdExtractionWriteISOImage(String driveLetter, String filename, int sectors);
    private native int cdExtGetDriveGeometry(long cdExt);
    private native int cdExtGetMSPDriveType(long cdExt);
    private native String cdExtGetDriveTypeString(String driveLetter);
    private native int cdExtOpenDrive(long cdExt, String drive_letter);
    private native int cdExtCloseDrive(long cdExt);
    private native int cdExtLockDriveRemoval(long cdExt, boolean unlock);
    private native int cdExtLockDriveAccess(long cdExt, boolean force, boolean askOnError, boolean unlock);
    private native int cdExtGetSectorSize(long cdExt);
    private native int cdExtGetSectorNo(long cdExt);
    private native int cdExtReadCDSectors(long cdExt, long startSector, long noSectors, byte dataBuf[]);
    private native int cdExtWriteCDSectors(long cdExt, long startSector, long noSectors, byte dataBuf[]);
    private native int cdExtGetMediumChangeState(long cdExt);
    private native int cdExtMediumRemovedForChange(long cdExt);
    private native int cdExtMediumInsertedAfterChange(long cdExt);
    private static boolean already_loaded = false;
    
    private URL codeBase;
    
    /* This is actually a pointer to some memory allocated by the native functions.
       The native functions may store some information in this structure, e.g.
       drive handles and drive geometry data. */
    long cdExt = 0;
    
    /* Java methods - wrappers around the native methods */
    public Drive[] getAvailableDrives() {
    	Drive[] ret = null;
    	
    	String drives = cdExtGetAvailableDrives();
    	if (drives != null && drives.length() > 0) {
    	    ret = new Drive[drives.length()];
    	    
    	    for (int i = 0; i < drives.length(); i++) {
    	    	ret[i] = new Drive("" + drives.charAt(i) + ":", getDriveTypeString("" + drives.charAt(i)));
    	    }
    	}
    	
    	return ret;
    }
    
    void initDriveRedirection() throws Exception {
    	if ((cdExt = cdExtractionInit()) == 0) {
    	    throw new IOException(T._("Could not initialize Drive Redirection library."));
    	}
    }
    
    void cleanupDriveRedirection() {
    	if (cdExt != 0) {
    	    cdExtractionCleanup(cdExt);
    	}
    	cdExt = 0;
    }
    
    public boolean mediumChangeSupported() {
    	if (cdExt != 0) {
    	    return cdExtMediumChangeSupported(cdExt);
    	} else {
    	    return false;
    	}
    }
    
    void writeISOImage(String driveLetter, String filename, int sectors) throws IOException {
    	int ret = cdExtractionWriteISOImage(driveLetter, filename, sectors);
    	if (ret != 0) {
    	    throw new IOException(T._("Could not write ISO image:") + " " + getErrorString(ret));
    	}
    }
    
    synchronized void getDriveGeometry() throws IOException {
    	if (cdExt != 0) {
    	    int ret = cdExtGetDriveGeometry(cdExt);
    	    if (ret != 0) {
    	    	throw new IOException(T._("Could not query drive geometry:") + " " + getErrorString(ret));
    	    }
    	}
    }
    
    public synchronized int getMSPDriveType() throws IOException {
    	if (cdExt != 0) {
    	    int ret = cdExtGetMSPDriveType(cdExt);
    	    if (ret < 0) {
    	    	throw new IOException(T._("Could not query drive type:") + " " + getErrorString(ret));
    	    }
    	    return ret;
    	} else {
    	    return 0;
    	}
    }
    
    synchronized String getDriveTypeString(String driveLetter) {
    	return cdExtGetDriveTypeString(driveLetter);
    }
    
    synchronized void openDrive(String driveLetter) throws IOException {
    	if (cdExt != 0) {
    	    int ret = cdExtOpenDrive(cdExt, driveLetter);
    	    if (ret != 0) {
    	    	throw new IOException(T._("Could not open drive:") + " " + getErrorString(ret));
    	    }
    	}
    }
    
    synchronized void closeDrive() throws IOException {
    	if (cdExt != 0) {
    	    int ret = cdExtCloseDrive(cdExt);
    	    if (ret != 0) {
    	    	throw new IOException(T._("Could not close drive:") + " " + getErrorString(ret));
    	    }
    	}
    }
    
    synchronized void lockDriveRemoval(boolean unlock) throws IOException {
    	if (cdExt != 0) {
    	    int ret = cdExtLockDriveRemoval(cdExt, unlock);
    	    /*
    	    if (ret != 0) {
    	    	throw new IOException(T._("Could not lock drive:") + " " + getErrorString(ret));
    	    }
    	    */
    	}
    }
    
    synchronized void lockDriveAccess(boolean force, boolean askOnError, boolean unlock) throws IOException {
    	if (cdExt != 0) {
    	    int ret = cdExtLockDriveAccess(cdExt, force, askOnError, unlock);
    	    if (ret != 0) {
    	    	throw new IOException(T._("Could not lock drive access:") + " " + getErrorString(ret));
    	    }
    	}
    }

    synchronized public int getSectorSize() {
    	if (cdExt != 0) {
    	    return cdExtGetSectorSize(cdExt);
    	} else {
    	    return 1;
    	}
    }
    
    synchronized public int getSectorNo() {
    	if (cdExt != 0) {
    	    return cdExtGetSectorNo(cdExt);
    	} else {
    	    return 1;
    	}
    }
    
    synchronized public void readCDSectors(long startSector, long noSectors, byte dataBuf[]) throws IOException {
    	if (cdExt != 0) {
    	    int ret = cdExtReadCDSectors(cdExt, startSector, noSectors, dataBuf);
    	    if (ret != 0) {
    	    	throw new IOException(T._("Could not read from drive:") + " " + getErrorString(ret));
    	    }
    	}
    }
    
    synchronized public void writeCDSectors(long startSector, long noSectors, byte dataBuf[]) throws IOException {
    	if (cdExt != 0) {
    	    int ret = cdExtWriteCDSectors(cdExt, startSector, noSectors, dataBuf);
    	    if (ret != 0) {
    	    	throw new IOException(T._("Could not write to drive:") + " " + getErrorString(ret));
    	    }
    	}
    }
    
    public int getMediumChangeState() {
        return cdExtGetMediumChangeState(cdExt);
    }
    
    synchronized public void mediumRemovedForChange() {
        cdExtMediumRemovedForChange(cdExt);
    }
    
    synchronized public void mediumInsertedAfterChange() {
        cdExtMediumInsertedAfterChange(cdExt);
    }

    /* load the native implementation */
    private void loadNativeCode() throws Exception {
    	LibraryDownloader downloader = new LibraryDownloader();
    	File libraryFile;
    	String libraryName;
    	
    	if (codeBase != null) {
    	    libraryFile = downloader.downloadFromWeb(codeBase.toString());
    	} else {
    	    libraryFile = downloader.downloadFromLocalFile();
    	}
    	
    	try {
    	    System.load(libraryFile.getCanonicalPath().trim());
    	} catch (Throwable e) {
    	    throw new Exception(e.getMessage());
    	}
    }
    
    /* Constructor */
    public DrvRedirNative(URL codeBase) throws Exception {
    	this.codeBase = codeBase;
    	
    	try {
	    if (already_loaded == false) {
		loadNativeCode();
		already_loaded = true;
	    }
    	} catch (Exception e) {
    	    e.printStackTrace();
    	    throw new IOException(e.getMessage());
    	}
    }
    
    protected void finalize() {
    	cleanupDriveRedirection();
    }
    
    public boolean showLockConfirm() {
    	ConfirmDialog dlg = new ConfirmDialog(
    		T._(	"Access to the selected drive could not be locked.\n" +
			"If you continue, the remote computer may get confused when\n" +
			"the local computer writes data to the disk. The drive cache\n" +
			"in the remote computer does not expect this.\n\n" +
			"If you have Write Support enabled, this will certainly lead\n" +
			"to file system errors and destroy the data on the disk.\n\n" +
			"Do you want to continue?"),
    		T._("Locking drive failed."),
    		ConfirmDialog.YES_NO, true);
    	dlg.show();
    	return dlg.yes;
    }
}
