/*
 * The Chat, how should it look like, and how should it function?
 * + window as is, see below
 * + rfb: protocol extensions
 *   - user name to Chat: really? we put that into another message,
 *                                like the version string
 *   - send chat messages: unstructured, pure string
 *   - recv chat messagge: same unstructured
 *     opens up the Chat Frame, if not already done so
 */
package nn.pp.rc;

import java.awt.*;
import java.awt.event.*;
import java.net.*;
import java.io.*;

public class ChatFrame {

    private TextArea chatarea;
    private TextField chatline;
    private Label idlabel;
    private Checkbox beep;
    private Component parent;
    private PrintStream logger;
    private Dialog myFrame;
    private RFBproto proto;
    private String me;
    private URL codebase;
    private static final String NASTRING = "unknown";

    public ChatFrame(Component parent, PrintStream logger, URL codebase) {
	this.parent = parent;
	this.logger = logger;
	this.codebase = codebase;
    }

    public Window getAsWindow() {
	if(myFrame == null) {
	    chatline = new TextField(40);
	    chatline.addActionListener(new ChatHandler());
	    chatarea = new TextArea("", 10, 60,
				    TextArea.SCROLLBARS_VERTICAL_ONLY);
	    chatarea.setEditable(false);
	    chatarea.setBackground(chatline.getBackground());
	    Panel clp = new Panel();
	    clp.setLayout(new BorderLayout());
	    idlabel = new Label(getIdLabel());
	    beep = new Checkbox("Beep", false);
	    clp.add(idlabel, BorderLayout.WEST);
	    clp.add(chatline, BorderLayout.CENTER);
	    clp.add(beep, BorderLayout.EAST);
	    while(!(parent instanceof Frame))
		if(null == (parent = parent.getParent())) break;
	    myFrame = new Dialog((Frame)parent);
	    myFrame.setLayout(new BorderLayout());
	    myFrame.add(chatarea, BorderLayout.CENTER);
	    myFrame.add(clp, BorderLayout.SOUTH);
	    myFrame.setTitle(getFrameTitle());
	    myFrame.pack();
	    myFrame.addWindowListener(new WindowAdapter() {
		    public void windowClosing(WindowEvent e) {
			setVisible(false);
		    }
		});
	}
	return myFrame;
    }

    public void setRFBProto(RFBproto proto) {
	this.proto = proto;
	if(null != myFrame) {
	    idlabel.setText(getIdLabel());
	    myFrame.setTitle(getFrameTitle());
	}
    }

    public void setVisible(boolean v) {
	Window win = getAsWindow();
	if(v && !win.isShowing()) {
	    Point p = parent.getLocationOnScreen();
	    Dimension pdim = parent.getSize();
	    Dimension cdim = win.getSize();
	    int x = pdim.width - cdim.width;
	    int y = pdim.height - cdim.height;
	    p.translate(x, y);
	    win.setLocation(p);
	}
	if(v) {
	    if(!win.isVisible()) win.setVisible(v);
	} else {
	    win.setVisible(v);
	    win.dispose();
	}
    }

    public void newMsg(String msg) {
	setVisible(true);
	chatarea.append(msg + '\n');
	chatarea.setCaretPosition(chatarea.getText().length() + 8192);
	if(beep.getState()) Toolkit.getDefaultToolkit().beep();
	chatline.requestFocus();
    }

    public void close() {
	if(myFrame != null) {
	    myFrame.setVisible(false);
	    myFrame.dispose();
	}
    }

    private String getIdLabel() {
	if(null == me || me.equals(NASTRING)) {
	    try { // first the easy version for our hostname
		me = InetAddress.getLocalHost().getHostName();
		if(me.equals("localhost")) me = toughGetHostName();
	    } catch(Exception e) { // now the tough trial
		me = toughGetHostName();
	    }
	}
	String n = proto == null ? NASTRING : proto.name;
	String u = null;
	try { u = System.getProperty("user.name") + '@'; }
	catch (Exception e) { u = ""; }
	return u + me + "(" + n + ")";
    }

    private String toughGetHostName() {
	String n = null;
	Socket s = null;
	try {
	    String host = codebase.getHost();
	    int port = codebase.getPort();
	    if(port < 0) {
		String protocol = codebase.getProtocol();
		if(protocol.equals("https")) port = 443;
		else port = 80;
	    }
	    s = new Socket(host, port);
	    n = s.getLocalAddress().getHostName();
	    s.close();
	} catch(Exception e) { // last thing is to look into the exception
	    String msg = e.getMessage();
	    int i1 = msg.indexOf('"');
	    if(i1 >= 0) {
		n = msg.substring(i1+1, msg.lastIndexOf('"'));
	    } else {
		System.out.println("ERROR: no local hostname: " + e);
		n = NASTRING;
	    }
	}
	if(s != null) try {s.close();} catch (Exception ignore) {}
	return n;
    }

    private String getFrameTitle() {
	String a = proto == null ? T._("not connected") : proto.profile.remoteHost;
	return a + " Chat";
    }

    class ChatHandler implements ActionListener {
	public void actionPerformed(ActionEvent e) {
	    if(null != proto) {
		try {
		    proto.writeString(idlabel.getText() + "=> "
				      + chatline.getText());
		    chatline.setText("");
		} catch(IOException ex) {
		    logger.println(T._("WARNING: Chat:") + " " + ex);
		}
	    } else {
		logger.println(T._("WARNING: Chat not connected!"));
	    }
	}
    }
}
