package nn.pp.rc;

import java.io.*;

/*
 * This is our new MonitoringDataInputStream.
 * It's a merge of the default Sun DataInputStream and our 
 * first  MonitoringInputStream. This was necessary because 
 * of the call-overhead of the virtual function calls. 
 * 
 * A buffer was addes to the stream. This allows higher
 * level code to use a buffered reading without using the
 * virtual function calls at all. But the code could read
 * too much data ahead, and it has to stall the data which
 * it didn't use. The next read operations will use the
 * buffer with the stalled data for reading.
 *
 * @author Lars Taubert
 */
public class MonitoringDataInputStream extends FilterInputStream
				       implements MonitoringStream
{

    private int countC;

    // Creates a MonitoringDataInputStream that uses the specified
    // underlying InputStream.
    public MonitoringDataInputStream(InputStream in) {
	super(in);
	
	buflen = 64*1024;
	buf = new byte[buflen];
    }
    
    // internal helper functions and variables for the buffering
    private int buflen = 0;
    private int count = 0;
    private byte[] buf = null;
    private int pos = 0;
    
    private final int readAByte() throws IOException {
    	if((count - pos) <= 0) {
    	    return in.read();
    	}
    	return buf[pos++] & 0xff;
    }

    private final int readBytes(byte b[], int off, int len) throws IOException {
	int avail = count - pos;
    	if(avail <= 0) {
    	    return in.read(b, off, len);
    	}
	
	int cnt = Math.min(len, avail);
	System.arraycopy(buf, pos, b, off, cnt);
	pos += cnt;
	
	return cnt;
    }

    // stall a pre-read buffer
    public void stall(byte b[], int off, int bytes) {
    	if(bytes == 0) {
    	    return;
    	}
    	
	synchronized (this) {
	    if(countC - bytes > 0) {
	    	countC -= bytes;
	    } else {
	    	countC = 0;
	    }
	}
    	int old_bytes = count - pos;
    	if(old_bytes > 0) {
    	    if((old_bytes + bytes) > buflen) {
    	    	// allocate new array
    	    	buflen = old_bytes + bytes;
    	    	byte[] new_buf = new byte[buflen];
    	    	// copy old data
    	    	System.arraycopy(buf, pos, new_buf, bytes, old_bytes);
    	    	buf = new_buf;
    	    } else {
    	    	System.arraycopy(buf, pos, buf, bytes, old_bytes);
    	    }
    	}
    	// set pointers
    	pos = 0;
    	count = old_bytes + bytes;
    	
    	// add the new data in front of the old data
    	System.arraycopy(b, off, buf, pos, bytes);
    }

    // public functions
    public final int read(byte b[]) throws IOException {
	return read(b, 0, b.length);
    }

    public final int read(byte b[], int off, int len) throws IOException {
	int ret = readBytes(b, off, len);
	synchronized (this) {
	    countC+=ret;
	}
	return ret;
    }

    public final int read() throws IOException {
	int b = readAByte();
	synchronized (this) {
	    countC++;
	}
	return b;
    }

    public final void readFully(byte b[]) throws IOException {
	readFully(b, 0, b.length);
    }

    public final void readFully(byte b[], int off, int len) throws IOException {
	if (len < 0)
	    throw new IndexOutOfBoundsException();
	InputStream in = this.in;
	int n = 0;
	while (n < len) {
	    int count = readBytes(b, off + n, len - n);
	    if (count < 0)
		throw new EOFException();
	    n += count;
	}
	synchronized (this) {
	    countC+=len;
	}
    }

    public final boolean readBoolean() throws IOException {
	int ch = readAByte();
	if (ch < 0)
	    throw new EOFException();
	synchronized (this) {
	    countC++;
	}
	return (ch != 0);
    }

    public final byte readByte() throws IOException {
	int ch = readAByte();
	if (ch < 0)
	    throw new EOFException();
	synchronized (this) {
	    countC++;
	}
	return (byte)(ch);
    }

    public final int readUnsignedByte() throws IOException {
	int ch = readAByte();
	if (ch < 0)
	    throw new EOFException();
	synchronized (this) {
	    countC++;
	}
	return ch;
    }

    public final short readShort() throws IOException {
	InputStream in = this.in;
	int ch1 = readAByte();
	int ch2 = readAByte();
	if ((ch1 | ch2) < 0)
	     throw new EOFException();
	synchronized (this) {
	    countC+=2;
	}
	return (short)((ch1 << 8) + (ch2 << 0));
    }

    public final int readUnsignedShort() throws IOException {
	InputStream in = this.in;
	int ch1 = readAByte();
	int ch2 = readAByte();
	if ((ch1 | ch2) < 0)
	     throw new EOFException();
	synchronized (this) {
	    countC+=2;
	}
	return (ch1 << 8) + (ch2 << 0);
    }

    public final char readChar() throws IOException {
	InputStream in = this.in;
	int ch1 = readAByte();
	int ch2 = readAByte();
	if ((ch1 | ch2) < 0)
	     throw new EOFException();
	synchronized (this) {
	    countC+=2;
	}
	return (char)((ch1 << 8) + (ch2 << 0));
    }

    public final int readInt() throws IOException {
	InputStream in = this.in;
	int ch1 = readAByte();
	int ch2 = readAByte();
	int ch3 = readAByte();
	int ch4 = readAByte();
	if ((ch1 | ch2 | ch3 | ch4) < 0)
	     throw new EOFException();
	synchronized (this) {
	    countC+=4;
	}
	return ((ch1 << 24) + (ch2 << 16) + (ch3 << 8) + (ch4 << 0));
    }

    public final long readUnsignedInt() throws IOException {
	InputStream in = this.in;
	long ch1 = readAByte();
	long ch2 = readAByte();
	long ch3 = readAByte();
	long ch4 = readAByte();
	if ((ch1 | ch2 | ch3 | ch4) < 0)
	     throw new EOFException();
	synchronized (this) {
	    countC+=4;
	}
	return ((ch1 << 24) + (ch2 << 16) + (ch3 << 8) + (ch4 << 0));
    }

    public final String readUTF() throws IOException {
        int utflen = readUnsignedShort();
        StringBuffer str = new StringBuffer(utflen);
        byte bytearr [] = new byte[utflen];
        int c, char2, char3;
	int count = 0;

 	readFully(bytearr, 0, utflen);

	while (count < utflen) {
     	    c = (int) bytearr[count] & 0xff;
	    switch (c >> 4) {
	        case 0: case 1: case 2: case 3: case 4: case 5: case 6: case 7:
		    /* 0xxxxxxx*/
		    count++;
                    str.append((char)c);
		    break;
	        case 12: case 13:
		    /* 110x xxxx   10xx xxxx*/
		    count += 2;
		    if (count > utflen)
			throw new UTFDataFormatException();
		    char2 = (int) bytearr[count-1];
		    if ((char2 & 0xC0) != 0x80)
			throw new UTFDataFormatException(); 
                    str.append((char)(((c & 0x1F) << 6) | (char2 & 0x3F)));
		    break;
	        case 14:
		    /* 1110 xxxx  10xx xxxx  10xx xxxx */
		    count += 3;
		    if (count > utflen)
			throw new UTFDataFormatException();
		    char2 = (int) bytearr[count-2];
		    char3 = (int) bytearr[count-1];
		    if (((char2 & 0xC0) != 0x80) || ((char3 & 0xC0) != 0x80))
			throw new UTFDataFormatException();	  
                    str.append((char)(((c     & 0x0F) << 12) |
                    	              ((char2 & 0x3F) << 6)  |
                    	              ((char3 & 0x3F) << 0)));
		    break;
	        default:
		    /* 10xx xxxx,  1111 xxxx */
		    throw new UTFDataFormatException();		  
		}
	}
        // The number of chars produced may be less than utflen
        return new String(str);
    }

    synchronized public final MonitorData getAndClearIn() {
	MonitorData ret = new MonitorData(countC);
	countC = 0;
	return ret;
    }
    
    synchronized public final int getAndClearOut() {
    	return 0;
    }
    
    synchronized public final void increaseCounters(int value) {
	countC += value;
    }
}
