package nn.pp.rc;

import java.io.*;
import java.net.*;
import java.awt.*;
import java.awt.event.*;
import nn.pp.rckbd.*;

/**
 * The RFBHandler is the actual protocoll handler running
 * the RFB protocoll.
 *
 * It makes use of two Co-classes:
 * RFBproto:    is the low level, message oriented network stuff
 * RFBRenderer: is a polymorphic class depending on the actual
 *              client capabilities and higher level protocoll
 *              parameters responsible to bring the bits on the screen
 *
 * RFBHandler is a thread and should be started as such
 */
public abstract class RFBHandler
    extends RCHandler
    implements ActionListener {

    RFBproto           rfbProto;
    RFBRenderer        rdr = null;
    private RFBRenderer        rdr16 = null;
    private RFBRenderer        rdr8 = null;
    ForensicPanel      forensicPanel;
    
    boolean pendingFormatChange = false;
    boolean pendingScreenUpdate = false;

    protected boolean      repaint_scheduled = false;
    protected GenericTimer repaint_timer;
    
    protected Socket sock = null;
    
    int versionMajor = -1, versionMinor = -1;
    
    ConnectionParameters params = new ConnectionParameters();

    public RFBHandler (RCHandler parent, RFBProfile profile) {
	super(parent, profile);
    }
    
    public RFBHandler (ServerConsolePanelBase scp, RCCanvasPanel canvas,
		       ChatFrame chat, TrafficMonitor tmoni,
		       VideoSettingsFrame vsframe,
		       MouseHandler mhandler1, MouseHandler mhandler2,
		       PrintStream logger, RFBProfile profile) {
	super(scp, canvas, chat, tmoni, vsframe, mhandler1, mhandler2, logger, profile);
    }
    
    /* an RFBHandler might be instantiated without parameters in the constructor;
       then, init() must be called afterwards */
    public RFBHandler() {
    }
    
    public void setSocket(Socket sock) {
        this.sock = sock;
    }
    
    void setProtocolVersion(int major, int minor) {
        versionMajor = major;
        versionMinor = minor;
    }
    
    public abstract void init(ServerConsolePanelBase scp, RCCanvasPanel canvas,
		              ChatFrame chat, TrafficMonitor tmoni,
		              VideoSettingsFrame vsframe,
		              MouseHandler mhandler1, MouseHandler mhandler2,
		              PrintStream logger, RFBProfile profile);
    
    public RCRenderer getRenderer() {
    	return rdr;
    }
    
    public RFBproto getRFBProto() {
    	return rfbProto;
    }

    abstract protected void establishConnection() throws IOException;
    
    protected void disposeConnect() {
    	//rdr.dispose();
    	scp.disconnected(this);
    }

    protected void dispose() {
    	rdr.dispose();
    	if (parent != null) {
    	    parent.distributeProto();
    	}
    	scp.disconnected(this);
    }

    public void distributeProto() {
	rdr.setRFBProto(rfbProto);
	if (chat != null) chat.setRFBProto(rfbProto);
	mhandler1.setRCProto(rfbProto);
	if (mhandler2 != null)
	    mhandler2.setRCProto(rfbProto);
	tmoni.setStreams(rfbProto.min, rfbProto.mout);
	if (vsframe != null) vsframe.setRFBproto(rfbProto);
	canvas.setRCHandler(this);
	logger.println(canvas);
    }

    // repaint_timer callback
    public void actionPerformed(ActionEvent e) {
	repaint_scheduled = true;
    }

    public abstract void sendFormatAndEncodings(boolean framebufferUpdateRequest) throws IOException;

    abstract void sendFramebufferUpdateRequest() throws IOException;
    
    abstract void sendFullFramebufferUpdateRequest() throws IOException;
    
    public void setBlank(boolean blank) {
    	if (rdr != null) {
    	    rdr.setBlank(blank);
    	}
    }
    
    public String toString() {
	if(proto == null)
	    return T._("not connected");
	return rfbProto.getSecDesc();
    }

    RFBRenderer loadRFBRenderer(int bpp, boolean doSwitch) {
	RFBRenderer newRdr;
	Dimension d = null;
	if (rdr != null) {
	    d = rdr.mysize;
	}
	if (bpp == 16) {
	    if (rdr16 == null) {
		rdr16 = loadRFBRenderer(canvas, true);
		rdr16.setRFBProto(rfbProto);
	    }
	    newRdr = rdr16;
	} else {
	    if (rdr8 == null) {
		rdr8 = loadRFBRenderer(canvas, false);
		rdr8.setRFBProto(rfbProto);
	    }
	    newRdr = rdr8;
	}
	
	if (doSwitch && rdr != newRdr) {
	    System.out.println("Switching to new RFB Renderer: " + newRdr);
	    rdr = newRdr;
	    canvas.setRCRenderer(rdr);
	    if (d != null) {
	    	rdr.setRenderSize(d);
	    }
	}
	
	return newRdr;

    }

    private RFBRenderer loadRFBRenderer(Component scc, boolean truecolor) {
	RFBRenderer r;
	if(truecolor && JVMVersionInfo.getJVMVersionInfo().isJava14()) {
	    rdr16 = r = new J14TrueColorRFBRenderer(scc);
	} else if(truecolor) {
	    rdr16 = r = new J11TrueColorRFBRenderer(scc);
	} else if(JVMVersionInfo.getJVMVersionInfo().isJava14()) {
	    rdr8 = r = new J14ByteColorRFBRenderer(scc);
	} else {
	    rdr8 = r = new J11ByteColorRFBRenderer(scc);
	}
	r.setOSDColorAndPosition(params.osdBgColor, params.osdFgColor, params.osdAlpha, params.osdPosition);
	r.setRFBHandler(this);
	return r;
    }
}
