package nn.pp.rc;

import java.io.*;
import java.net.*;
import java.text.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.util.zip.*;
import java.lang.reflect.*;
import java.security.*;
import javax.net.ssl.*;

/**
 * The protocoll implementation for the RFB.
 * The protocoll connects to a server using the 
 * given profile in the constructor.
 * 
 * @author Thomas Breitfeld, added RFBProfile and connection magic
 */
public class RFBproto_01_16 extends RFBproto implements RCProto {

    private static final boolean debug = false;

    VideoSettings vs = new VideoSettings();
    
    RFBHandler handler;
    
    int bitsPerPixel, depth, ack_bitsPerPixel, ack_depth;
    boolean bigEndian, trueColour, isUnsupported;
    int redMax, greenMax, blueMax, redShift, greenShift, blueShift;
    int updateNRects;
    int copyRectSrcX, copyRectSrcY;
    int updateRectX, updateRectY, updateRectW, updateRectH, updateRectEncoding;
    int oldMod;
    boolean shiftset = false;
    boolean inNormalProtocol = false;
    String message;
    String osdStateMessage;
    boolean osdStateBlanking;
    int osdStateTimeout;

    MonitoringDataInputStream        is_buf = null;
    ByteArrayInputStream	     baStream = null;

    // handle ZLIBbed rects    
    final static int updateBufSize = 65536;
    private int updateRectSize;
    private int rectBufSize;
    private byte rectBuf[];
    private byte updateBuf[] = new byte[updateBufSize];    
    private Inflater zlibInflater = null;
    
    String serverCommand;
    String serverCommandParams;
    
    Date start, end;
    
    int connectionFlags = 0;
    
    int spCommand;
    int spCommandParam16;
    int spCommandParam8_1, spCommandParam8_2;

    RFBproto_01_16(RFBProfile profile, PrintStream logger, RFBHandler handler) {
	super(profile, logger);
	this.handler = handler;
	connected = false;
    }
    
    private void authenticatePlain() throws IOException {
	System.out.println("Authenticating plain");

	writeLoginMessage(profile.username, AuthPlain, connectionFlags);

	if(readServerMessageType() != SessionChallenge) 
		throw new IOException(T._("Protocol error: no Session Challenge"));
	readChallengeMessage(0, 0);
	writeChallengeResponseMessage(profile.password);
    }
    
    private void authenticateMD5() throws IOException {
	System.out.println("Authenticating via MD5");
       
    	MessageDigest digest;
    	try {
    	    digest = MessageDigest.getInstance("MD5");
    	} catch (NoSuchAlgorithmException e) {
    	    System.out.println("MD5 not available, using plain authentication.");
    	    authenticatePlain();
    	    return;
    	}
    	
	writeLoginMessage(profile.username, AuthMD5, connectionFlags);

	if(readServerMessageType() != SessionChallenge) 
		throw new IOException(T._("Protocol error: no Session Challenge"));
	byte[] salt = readChallengeMessage(0, 64);
	
    	if (salt != null) {
    	    digest.update(salt);
    	}
    	digest.update(profile.password.getBytes("ISO-8859-1"));
    	
    	byte[] hash = digest.digest();
    	String hashString = binToHex(hash);
	writeChallengeResponseMessage(hashString);
    }
    
    private void authenticateSessionID() throws IOException {
	System.out.println("Authenticating via HTTP session ID");
	
	writeLoginMessage("", AuthHttpSessId, connectionFlags);

	if(readServerMessageType() != SessionChallenge) 
		throw new IOException(T._("Protocol error: no Session Challenge"));
	byte[] _challenge = readChallengeMessage(65, 65);
	byte[] challenge = new byte[64];
	System.arraycopy(_challenge, 0, challenge, 0, 64);

	byte[] response = new byte[32];
	getChallengeResponse(challenge, response, 0);
	writeChallengeResponseMessage(response);
    }
    
    private void authenticateRDMsession() throws IOException {
    	System.out.println("Authenticating via RDM session");
    	
    	writeLoginMessage("super", AuthRDMsession, connectionFlags);
    	
	if (readServerMessageType() != SessionChallenge)
		throw new IOException(T._("Protocol error: no Session Challenge"));
	readChallengeMessage(0, 0);
	writeChallengeResponseMessage(profile.rdmSession+'\0');
    }
    
    private void authenticate() throws IOException {
	// negotiate auth method and login
	if(readServerMessageType() != AuthCaps) {
	    throw new IOException(T._("Protocol error: no Authentication Capabilities"));
	}
	
	int caps = readAuthCapsMessage();
	if ((caps & AuthMD5) != 0 && profile.username != null && profile.password != null) {
	    authenticateMD5();
	} else if ((caps & AuthPlain) != 0 && profile.username != null && profile.password != null) {
	    authenticatePlain();
	} else if ((caps & AuthRDMsession) != 0 && profile.rdmSession != null) {
            authenticateRDMsession();
	} else if ((caps & AuthHttpSessId) != 0) {
	    authenticateSessionID();
	} else {
	    throw new IOException(T._("Server does not support authentication schemes"));
	}

	if(readServerMessageType() != AuthAck) {
	    throw new IOException(T._("Protocol error: no Auth Ack"));
	}
	is.readByte();
	is.readUnsignedShort();
	connectionFlags = (int)(is.readUnsignedInt() & (long)0xffffffff);
    }
    
    private void sendSessionID() throws IOException {
    	byte b[] = new byte[4];
    	b[0] = (byte)SpSetSession;
    	b[1] = 1;   // stop at end
    	os.write(b);
    	os.writeUnsignedInt(profile.replaySessionID);
    }
    
    private void sendSpSessionTimes(Date start, Date end) throws IOException {
    	byte b[] = new byte[4];
    	b[0] = (byte)SpSetSessionTimes;
    	os.write(b);
    	writeTimestamp(start);
    	writeTimestamp(end);
    }
    
    private void sendSpSessionCim(int cimID) throws IOException {
    	byte b[] = new byte[4];
    	b[0] = (byte)SpSetCimId;
    	os.write(b);
    	os.writeUnsignedInt(cimID);
    }
    
    private void sendSpSessionCimAndTimes() throws IOException {
    	sendSpSessionCim(profile.replayCimId);
    	sendSpSessionTimes(profile.replayStartTime, profile.replayEndTime);
    }
    
    private void sendSessionParameters() throws IOException {
    	if ((profile.replayStartTime != null) && (profile.replayEndTime != null) && (profile.replayCimId != -1)) {
    	    sendSpSessionCimAndTimes();
    	} else if (profile.replaySessionID != -1) {
    	    sendSessionID();
    	} else {
    	    throw new IOException("No replay parameters found!");
    	}
    }
    
    protected void runInitialHandshake() throws IOException {
	try {
	    // say hello and negotiate protocol version
	    negotiateProtocolVersion();
	    
	    logger.println(MessageFormat.format(T._("RFB: server supports protocol version {0}.{1}"),
	                   new Object[] { new Integer(serverMajor), new Integer(serverMinor) }));

	    // authenticate
	    connectionFlags = 0;
	    if (profile.forensicConsole) {
	    	connectionFlags |= ConnectionFlagSasSw;
	    }
	    if (profile.forensicNoKeyboard) {
	    	connectionFlags |= (ConnectionFlagSasKbdNoKbd << ConnectionFlagSasKbdShift);
	    } else if (profile.forensicFilterKeyboard) {
	    	connectionFlags |= (ConnectionFlagSasKbdFilter << ConnectionFlagSasKbdShift);
	    }
	    
	    if ((profile.replaySessionID != -1) || ((profile.replayStartTime != null) && (profile.replayEndTime != null))) {
	    	connectionFlags |= ConnectionFlagSp;
	    }

	    authenticate();
	    
	    if ((connectionFlags & ConnectionFlagSp) != 0) {
	    	sendSessionParameters();
	    }

	    int t = readServerMessageType();
	    
	    while (t == OsdState) {
	        readOsdStateMessage();
	        t = readServerMessageType();
	    }
	    
	    if (t == SpSetSessionTimes) {
	    	readSpSessionTimes();
	    	t = readServerMessageType();
	    }
	    
	    if (t != UTFString)
		throw new IOException(T._("Protocol error: no Chat Welcome"));
	    name = readString();

	    if (readServerMessageType() != OsdState) 
		throw new IOException(T._("Protocol error: no OSD State"));
	    readOsdStateMessage();


	    if (readServerMessageType() != KbdLayout) 
		throw new IOException(T._("Protocol error: no Keyboard layout"));
	    readKbdLayout();

	    if (readServerMessageType() != CapabilityTable)
		throw new IOException(T._("Protocol error: no Capability Table"));
	    readCapabilityTableMessage();

	    writeClientInit();
	    readServerMessageType();

	    readServerInit();

	    connected = true;
	    
	} catch(IOException e) {
	    close();
	    throw e;
	}
    }
    
    int readAuthCapsMessage() throws IOException {
	return is.readUnsignedByte();
    }

    void writeLoginMessage(String user_name, int auth_method, int flags) throws IOException {
	os.write(Login);
	os.write(auth_method);
	os.write(user_name.length() + 1);
	os.write(0);	// pad
	os.writeUnsignedInt(flags);
	os.write(user_name.getBytes("ISO-8859-1"));
	os.write(0); // terminating zero
    }

    byte[] readChallengeMessage(int expLen, int maxLen) throws IOException {
	int len = is.readUnsignedByte();
	
	if ((expLen != 0 && len != expLen) || len > maxLen)
	    throw new IOException(T._("Protocol error: bad challenge size"));

	if (len > 0) {
	    byte[] challenge = new byte[len];
	    is.readFully(challenge);
	    return challenge;
	}
	return null;
    }

    void writeChallengeResponseMessage(byte[] response, boolean terminate) throws IOException {
	byte[] b = new byte[2];
	b[0] = ChallengeResponse;
	b[1] = (byte)(response.length + (terminate ? 1 : 0));
	os.write(b);
	os.write(response);
	if (terminate) os.write(0); // terminating zero
    }
    
    void writeChallengeResponseMessage(byte[] response) throws IOException {
    	writeChallengeResponseMessage(response, true);
    }
    
    void writeChallengeResponseMessage(String response) throws IOException {
	writeChallengeResponseMessage(response.getBytes("ISO-8859-1"), false);
    }
    
    String readString() throws IOException {
	is.readUnsignedByte(); // padding
	return is.readUTF();
    }

    void writeString(String msg) throws IOException {
	byte[] b = new byte[2];
	b[0] = UTFString;
	os.write(b);
	os.writeUTF(msg);
    }

    //
    // Write the client initialisation message
    //
    void writeClientInit() throws IOException {
	os.write(1);
	os.write((byte)profile.channelId);
	os.flush();
    }

    //
    // Read the server initialisation message
    //
    void readServerInit() throws IOException {
	isUnsupported = (is.readUnsignedByte() != 0);
	
	framebufferWidth = framebufferWidthPadded = is.readUnsignedShort();
	framebufferHeight = framebufferHeightPadded = is.readUnsignedShort();

	if ((framebufferWidthPadded % tileWidth) != 0)
	    framebufferWidthPadded = (framebufferWidthPadded / tileWidth + 1) * tileWidth;
	if ((framebufferHeightPadded % tileHeight) != 0)
	    framebufferHeightPadded = (framebufferHeightPadded / tileHeight + 1) * tileHeight;
	
	is.readUnsignedShort(); // pad
	Date d = readTimestamp();
	
	bitsPerPixel = is.readUnsignedByte();
	depth = is.readUnsignedByte();
	bigEndian = (is.readUnsignedByte() != 0);
	trueColour = (is.readUnsignedByte() != 0);

	redMax = is.readUnsignedShort();
	greenMax = is.readUnsignedShort();
	blueMax = is.readUnsignedShort();

	redShift = is.readUnsignedByte();
	greenShift = is.readUnsignedByte();
	blueShift = is.readUnsignedByte();

	byte[] pad = new byte[3];
	is.readFully(pad);

	inNormalProtocol = true;
    }

    int readServerMessageType() throws IOException {
	// increase the monitoring counters manually
	// (see also MonitoringDataInputStream)
	min.increaseCounters(1);
	int type = is.readUnsignedByte();
	if (type == Quit) throw new IOException(getQuitReasonMsg(readQuitReason()));
	return type;
    }
    
    Date readTimestamp() throws IOException {
    	long sec = is.readUnsignedInt();
    	long usec = is.readUnsignedInt();
    	return new Date(sec * 1000 + usec / 1000);
    }
    
    void writeTimestamp(Date timestamp) throws IOException {
    	long msec = timestamp.getTime();
    	long sec = msec / 1000;
    	long usec = (msec % 1000) * 1000;
    	os.writeUnsignedInt(sec);
    	os.writeUnsignedInt(usec);
    }

    //
    // Read a FramebufferUpdate message
    //
    void readFramebufferUpdate() throws IOException 
    {
       int flags = is.readByte();
       updateNRects = is.readUnsignedShort();
       //System.out.println("nr rects=" + updateNRects);
       if ((flags & FbUpdateFlagTimestamp) != 0) {
           Date d = readTimestamp();
           //System.out.println("Timestamp: " + d);
       }
    }

    int readQuitReason() throws IOException
    {
	return is.readByte();
    }

    // Read a FramebufferUpdate rectangle header
    void readFramebufferUpdateRectHdr() throws IOException {
	updateRectX = is.readUnsignedShort();
	updateRectY = is.readUnsignedShort();

	updateRectW = is.readUnsignedShort();
	updateRectH = is.readUnsignedShort();

	updateRectEncoding = is.readInt();

	/*
	System.out.println("Rect ("+updateRectX+","+updateRectY+") WH("+updateRectW+","+updateRectH+"), Encoding="+
			   Integer.toHexString(updateRectEncoding));
	*/
			   
	if ((updateRectX + updateRectW > framebufferWidthPadded) ||
	    (updateRectY + updateRectH > framebufferHeightPadded)) {
	    throw new IOException("Framebuffer update rectangle " +
				  "too large: " +
				  updateRectW + "x" +
				  updateRectH + " at (" +
				  updateRectX + "," +
				  updateRectY + ")");
	}
    }

    
    // Read CopyRect source X and Y.
    void readCopyRect() throws IOException 
    {
        copyRectSrcX = is.readUnsignedShort();
        copyRectSrcY = is.readUnsignedShort();
    }

    //
    // Read size of hardware encoded rect
    //
    // if we are not ZLIBbed we need to read it from network,
    // otherwise we already have it
    int readHWencSize() throws IOException
    {
	if ((updateRectEncoding & EncodingParamZLIBMask) != 0 || (updateRectEncoding & EncodingParamCompressMask) != 0) {
	    return updateRectSize;
	} else {
	    return is.readInt();
	}
    }

    //
    // Read hwenc padding bytes if not ZLIBbed
    //
    void readHWencPadding(int size) throws IOException
    {
	if ((updateRectEncoding & EncodingParamZLIBMask) != 0 ||
	    (updateRectEncoding & EncodingParamCompressMask) != 0 ||
	    size % 4 == 0) {
	    
	    return;
	}

	int pad_size = 4 - (size % 4);

	for (int i = 0; i < pad_size; i++) {
	    is.readByte();
	}
    }

    
    void readConsoleMessage() throws IOException
    {
        int size;

        byte[] pad = new byte[3];
        is.readFully(pad);

        size = is.readInt();
        Date d = readTimestamp();

        byte[] buf = new byte[size];
        is.readFully(buf, 0, size);

        message = new String(buf, T.getCharset());
    }
    
    void readServerCommand() throws IOException
    {
    	int sizeCommand, sizeParams;
    	byte bufCommand[], bufParams[];
    	
    	is.readUnsignedByte();	// padding

	sizeCommand = is.readUnsignedShort();
	sizeParams = is.readUnsignedShort();
	
	bufCommand = new byte[sizeCommand];
	bufParams = new byte[sizeParams];

        is.readFully(bufCommand, 0, sizeCommand);
        is.readFully(bufParams, 0, sizeParams);
        
        serverCommand = new String(bufCommand);
        serverCommandParams = new String(bufParams);
    }
    
    void readCapabilityTableMessage() throws IOException
    {
    	int no_caps = is.readUnsignedByte();
    	
    	for (int i = 0; i < no_caps; i++) {
    	    int name_len = is.readUnsignedByte();
    	    int value_len = is.readUnsignedByte();
    	    
    	    byte[] name = new byte[name_len];
    	    byte[] value = new byte[value_len];
    	    
    	    is.readFully(name, 0, name_len);
    	    is.readFully(value, 0, value_len);

    	    handler.params.add(new String(name), new String(value));
    	}

    	try {
    	    handler.params.setParameters();
    	} catch (Exception e) {
    	    throw new IOException(e.getMessage());
    	}
    }

    void readOsdStateMessage() throws IOException
    {
	int size;
	byte blanking;
	
	blanking = (byte) is.readUnsignedByte();
	osdStateBlanking = (blanking == 1) ? true : false;
	osdStateTimeout = is.readUnsignedShort();
	
	size = is.readUnsignedShort();
	is.readUnsignedShort();
	Date d = readTimestamp();

	byte[] buf = new byte[size];
        is.readFully(buf, 0, size);

        osdStateMessage = new String(buf, T.getCharset());
    }
    
    void readKbdLayout() throws IOException
    {
	int size;

	is.readUnsignedByte(); // padding

	size = is.readUnsignedShort();
	byte[] buf = new byte[size];
	is.readFully(buf, 0, size);

	kbdlayout = new String(buf);
    }
				
    void readVideoSettingsUpdate() throws IOException
    {
	is.readUnsignedByte();  // padding
	vs.brightness		= is.readUnsignedByte();
	vs.contrast_red		= is.readUnsignedByte();
	vs.contrast_green	= is.readUnsignedByte();
	vs.contrast_blue	= is.readUnsignedByte();
	vs.clock		= is.readUnsignedShort();
	vs.phase		= is.readUnsignedShort();
	vs.x_offset		= is.readUnsignedShort();
	vs.y_offset		= is.readUnsignedShort();
	vs.x_res		= is.readUnsignedShort();
	vs.y_res		= is.readUnsignedShort();
	vs.refresh		= is.readUnsignedShort();
	vs.y_max_offset		= is.readUnsignedShort();
    }
    
    void readVideoQualityUpdate() throws IOException
    {
    	is.readUnsignedByte();	// noise filter setting
    	is.readUnsignedByte();	// quality/speed setting
    }

    byte readRCModeReply() throws IOException {
    	return is.readByte();
    }

    void readAckPixelFormat() throws IOException {
	byte[] pad = new byte[3];
	is.readFully(pad);

	/* we only care about the bpp for know */
	ack_bitsPerPixel = is.readUnsignedByte();
	ack_depth = is.readUnsignedByte();
	
	bigEndian = (is.readUnsignedByte() != 0); // bigendian
	is.readUnsignedByte(); // trueColour

	is.readUnsignedShort(); // redMax
	is.readUnsignedShort(); // greenMax
	is.readUnsignedShort(); // blueMax

	is.readUnsignedByte(); // redShift
	is.readUnsignedByte(); // greenShift
	is.readUnsignedByte(); // blueShift

	is.readFully(pad);
    }
    
    //
    // Write a FramebufferUpdateRequest message
    //
    synchronized void writeFramebufferUpdateRequest(int x, int y, int w, int h,
						    boolean incremental)
	throws IOException {
        os.write(FramebufferUpdateRequest);
	os.write(incremental ? 1 : 0);
        os.writeUnsignedShort(x);
        os.writeUnsignedShort(y);
        os.writeUnsignedShort(w);
        os.writeUnsignedShort(h);
	os.flush();
    }

    void writeFullFramebufferUpdateRequest() throws IOException {
    	writeFramebufferUpdateRequest(0, 0, framebufferWidthPadded, framebufferHeightPadded, false);
    }

    //
    // Write a SetPixelFormat message
    //
    synchronized void writeSetPixelFormat(int bitsPerPixel, int depth,
                                          boolean bigEndian,
                                          boolean trueColour,
                                          int redMax, int greenMax,
                                          int blueMax,
                                          int redShift, int greenShift,
                                          int blueShift)
	throws IOException {

        byte[] b = {
	    (byte) SetPixelFormat,
	    0,
	    0,
	    0,
	    (byte) bitsPerPixel,
	    (byte) depth,
            (byte)(bigEndian ? 1 : 0),
	    (byte)(trueColour ? 1 : 0),
	    (byte)((redMax >> 8) & 0xff),
	    (byte)(redMax & 0xff),
	    (byte)((greenMax >> 8) & 0xff),
	    (byte)(greenMax & 0xff),
	    (byte)((blueMax >> 8) & 0xff),
	    (byte)(blueMax & 0xff),
	    (byte) redShift,
	    (byte) greenShift,
	    (byte) blueShift,
	    0,
	    0,
	    0	    
	};

        os.write(b);
	os.flush();
    }

    //
    // Write a SetEncodings message
    //
    synchronized void writeSetEncodings(int[] encs, int len)
	throws IOException {

        os.write(SetEncodings);
        os.write(0);
        os.writeUnsignedShort(len);

        for (int i = 0; i < len; i++) {
            os.writeUnsignedInt(encs[i]);
	}

	os.flush();
    }

    public synchronized void writePointerEvent(boolean relative,
					int x, int y, int z, int pointerMask)
	throws IOException
    {
	if (monitorMode) return;

	if (!relative) {
	    if (x < 0)  x = 0;
	    if (y < 0)  y = 0;
	}

	os.write((relative==true) ? PointerEventRelative : PointerEvent);
	os.write(pointerMask);
	os.writeUnsignedShort(x);
	os.writeUnsignedShort(y);
	os.writeUnsignedShort(z);
	
	os.flush();
    }

    int readPing() throws IOException
    {
	/* read three dummy bytes, than the int */
	is.readByte();
	is.readByte();
	is.readByte();
	return is.readInt();
    }
    
    synchronized void writePingReply(int serial) throws IOException {
	os.write(PingReply);
	os.write(0);
	os.write(0);
	os.write(0);
	os.writeUnsignedInt(serial);
	os.flush();
    }

    void readBandwidthRequest() throws IOException {

	is.readByte(); /* pad */
	short len = is.readShort();

	byte b[] = new byte [len];

	is.readFully(b);
    }
    
    synchronized void writeBandwidthTick(byte stage) throws IOException {
	byte b[] = {
	    (byte)BandwidthTick,
	    stage
	};
        os.write(b);
	os.flush();
    }
    
    public synchronized void writeKeyboardEvent(byte keycode) throws IOException
    {
	if (monitorMode) return;

        if(debug) {
	    System.out.print("writeKeyCodeEvent: ");
	    System.out.print(Integer.toHexString(0xff & (int)keycode));
	    System.out.println(" done...");
	}
	byte b[] = {
	    (byte) KeyboardEvent,
	    keycode
	};
        os.write(b);
	os.flush();
    }

    synchronized void writeUserPropChangeEvent(String key, String value)
	throws IOException {
	int keyLength, valueLength;
	byte[] keyBytes;
	byte[] valueBytes;

	if(debug) {
	    System.out.print("writeUserPropChangeEvent: "
			     + key + " - Value: " + value + "\n");
    	}
	keyLength = key.length();
	valueLength = value.length();
	keyBytes  = key.getBytes();
	valueBytes = value.getBytes();
	byte[] b = {
	    (byte) UserPropChangeEvent,
	    (byte) keyLength,
	    (byte) valueLength
	};
	os.write(b);
	os.write(keyBytes, 0, keyLength);
	os.write(valueBytes, 0, valueLength);
	os.flush();
    }    
    
    void writeGlobalPropChangeEvent(String key, String value) throws IOException {
        throw IOException("not supported");
    }
    
    public synchronized void writeMouseSyncEvent(byte subevent)
	throws IOException
    {
	if (monitorMode) return;

	if(debug) {
            System.out.print("writeMouseSyncEvent: \n"+subevent);
	}

	byte b[] = {
	    (byte) MouseSyncEvent,
	    (byte) subevent
	};
	
        os.write(b);
	os.flush();
    }

    synchronized void writeCursorChangeEvent(String sName) throws IOException {
	int length;
	byte bytes[];

	if(debug) {
            System.out.print("writeCursorChangeEvent: " + sName + "\n");
        }
	length = sName.length();
	bytes  = sName.getBytes();

	byte b[] = {
	    (byte) CursorChangeEvent,
	    (byte) length,
	};
	os.write(b);
	os.write(bytes, 0, length);
	os.flush();
    }

    synchronized void writeKvmSwitchEvent(short newport) throws IOException
    {
	if (monitorMode) return;

	os.write(KvmSwitchEvent);
	os.write(0);
	os.writeUnsignedShort(newport);

	os.flush();	
    }

    synchronized void writeVideoSettingsEvent(byte event, short value)
	throws IOException {
	if(debug) {
            System.out.print("writeVideoSettingsEvent: "
			     + event + ", value=" + value + "\n");
        }
        
        os.write(VideoSettingsEventC2S);
        os.write(event);
        os.writeUnsignedShort(value);
        
	os.flush();
    }

    synchronized void writeVideoSettingsRequest(byte request)
	throws IOException {
	if(debug) {
            System.out.print("writeVideoRequest: " + request + "\n");
        }
	byte b[] = {
	    (byte) VideoSettingsRequest,
	    (byte) request
	};
        os.write(b);
	os.flush();
    }

    synchronized void writeVideoRefresh() throws IOException {
	if(debug) {
            System.out.print("writeVideoRefresh\n");
        }
	byte b[] = {
	    (byte) VideoRefresh
	};
        os.write(b);
	os.flush();
    }

    public synchronized void writeRCModeRequest(byte request) throws IOException {
	if(debug) {
            System.out.print("writeRCModeRequest\n");
        }
	byte b[] = {
	    (byte) RCModeRequest,
	    (byte) request
	};
        os.write(b);
	os.flush();
    }

    /* prepare the input stream for our renderers so they
       either access the raw one or the already inflated one
       in case ZLIB was used */
    public void prepareUpdateRect() throws java.io.IOException {
	if ((updateRectEncoding & EncodingParamZLIBMask) != 0) {
	    updateRectSize = readCompactLen();
	    if (updateRectSize > rectBufSize) {
	    	rectBuf = new byte[updateRectSize];
	    	rectBufSize = updateRectSize;
	    }

	    int bytesTotal = updateRectSize;
	    int dest = 0;
	    if (zlibInflater == null) zlibInflater = new Inflater();
	    
	    while (bytesTotal > 0) {	    
	    	int portionSize = readCompactLen();
	    	is.readFully(updateBuf, 0, portionSize);

	    	zlibInflater.setInput(updateBuf, 0, portionSize);
	    	try {
		    int bytesRead = zlibInflater.inflate(rectBuf, dest, bytesTotal);
		    dest += bytesRead;
		    bytesTotal -= bytesRead;
	    	} catch(DataFormatException dfe) {
                    throw new IOException(dfe.toString());
            	}
	    }

	    baStream = new ByteArrayInputStream(rectBuf);	    
	    is_buf = new MonitoringDataInputStream(baStream);
	
	    is_rect = is_buf;
	} else if ((updateRectEncoding & EncodingParamCompressMask) != 0) {
	    
	    int comprSize = (int)is.readUnsignedInt();
	    int uncomprSize = (int)is.readUnsignedInt();
	    int read = 0;
	    int dest = 0;
	    int bytesTotal = uncomprSize;
	    
	    if (uncomprSize > rectBufSize) {
	    	rectBuf = new byte[uncomprSize];
	    	rectBufSize = uncomprSize;
	    }
	    
	    Inflater decompressor = new Inflater();
	    
	    while (read < comprSize) {
	    	int portionSize = ((comprSize - read) > updateBufSize) ? updateBufSize : ((comprSize - read) % updateBufSize);
	    	is.readFully(updateBuf, 0, portionSize);
	    	
	    	decompressor.setInput(updateBuf, 0, portionSize);

	    	try {
		    int bytesRead = decompressor.inflate(rectBuf, dest, bytesTotal);
		    dest += bytesRead;
		    read += portionSize;
		    bytesTotal -= bytesRead;
	    	} catch(DataFormatException dfe) {
                    throw new IOException(dfe.toString());
            	}
	    }

	    decompressor.end();

	    baStream = new ByteArrayInputStream(rectBuf);	    
	    is_buf = new MonitoringDataInputStream(baStream);
	
	    is_rect = is_buf;
	} else {
	    is_rect = is;
	}
    }
    
    // FIXME: set this to false
    static boolean sasDebug = false;
    
    void readSasError() throws IOException {
    	int err = is.readByte();
    	logger.println("SAS Error: " + getSasErrorMsg(err));
    }
    
    void readCausingKvmSession(SasEvent evt) throws IOException {
    	long id = is.readUnsignedInt();
    	evt.setCausingKvmSession(id);
    }
    
    private void readSasInputEvent(SasEvent evt) throws IOException {
    	int input_type = is.readUnsignedByte();
    	
    	switch (input_type) {
    	    case KeyboardEvent:
    	    	int keycode = is.readByte();
    	    	if (sasDebug) {
    	    	    System.out.println("Keyboard event: keycode 0x" + Integer.toHexString(keycode));
    	    	}
    	    	evt.setKeyEvent(input_type, keycode);
    	    	break;
    	    case PointerEvent:
    	    case PointerEventRelative:
    	    	int buttonMask = is.readByte();
    	    	int x = is.readShort();
    	    	int y = is.readShort();
    	    	int z = is.readShort();
    	    	if (sasDebug) {
    	    	    System.out.println("Pointer event (" + (input_type == PointerEvent ? "Absolute" : "Relative") +
    	    	    	"): x = " + x + ", y = " + y + ", z = " + z + ", bm = " + buttonMask);
    	    	}
    	    	evt.setMouseEvent(input_type, buttonMask, x, y, z);
    	    	break;
    	    case MouseSyncEvent:
    	    	int type = is.readByte();
    	    	if (sasDebug) {
    	    	    System.out.println("MouseSync event: type 0x" + Integer.toHexString(type));
    	    	}
    	    	evt.setMouseSyncEvent(input_type, type);
    	    	break;
    	    default:
    	    	byte dummy[] = new byte[evt.len - 1];
    	    	is.readFully(dummy);
    	    	logger.println(T._("Unknown forensic input event received."));
    	    	System.out.println("Unknown forensic event: 0x" + Integer.toHexString(input_type));
    	}
    }
    
    private String readString(int len) throws IOException {
    	if (len > 0) {
    	    byte _s[] = new byte[len];
    	    is.readFully(_s);
    	    return new String(_s);
    	} else {
    	    return "";
    	}
    }
    
    private void readSasExistingSession(SasEvent evt) throws IOException {
    	int self = is.readByte();
    	is.readByte(); // pad
    	is.readUnsignedShort();	// pad
    	Date time = readTimestamp();

    	int user_len = is.readUnsignedByte();
    	int ip_len = is.readUnsignedByte();

    	String user = readString(user_len);
    	String ip = readString(ip_len);
    	
    	evt.setExistingSession(self != 0, user, ip, time);
    }
    
    private void readSasExistingKvmSession(SasEvent evt) throws IOException {
    	int excl = is.readByte();
    	is.readByte(); // pad
    	is.readUnsignedShort();	// pad
    	long id = is.readUnsignedInt();
    	Date time = readTimestamp();
    	
    	evt.setExistingKvmSession(id, excl != 0, time);
    }
    
    private void readSasUserInfo(SasEvent evt) throws IOException {
    	int user_len = is.readUnsignedByte();
    	int ip_len = is.readUnsignedByte();

    	String user = readString(user_len);
    	String ip = readString(ip_len);
    	
    	evt.setUserInfo(user, ip);
    }
    
    private void readSasNewSession(SasEvent evt) throws IOException {
    	Date login = readTimestamp();
    	int user_len = is.readUnsignedByte();
    	int ip_len = is.readUnsignedByte();

    	String user = readString(user_len);
    	String ip = readString(ip_len);
    	
    	evt.setNewSession(user, ip, login);
    }
    
    private void readSasNewKvmSession(SasEvent evt) throws IOException {
        Date login = readTimestamp();
        evt.setNewKvmSession(login);
    }
    
    private void readSasKvmSwitchEvent(SasEvent evt) throws IOException {
    	int channel = is.readUnsignedByte();
    	int unit = is.readUnsignedByte();
    	int port = is.readUnsignedShort();
    	
    	evt.setKvmSwitchEvent(channel, unit, port);
    }
    
    SasEvent readSasEvent() throws IOException {
    	int evt_type = is.readUnsignedByte();
    	int len = is.readUnsignedShort();
    	long session = is.readUnsignedInt();
    	Date time = readTimestamp();
    	
    	if (sasDebug) {
    	    System.out.println("Got SAS event 0x" + Integer.toHexString(evt_type) + ":");
    	    System.out.println("Timestamp: " + time + " Session: " + session);
    	}
    	
    	SasEvent evt = new SasEvent(evt_type, time, session, len);
    	
    	switch (evt.type) {
    	    // nothing to do with these events
    	    case SasEvent.UserSessionClosed:
    	    case SasEvent.ResetSessions:
    	    	break;

    	    case SasEvent.KvmSessionOpened:
    	    	readCausingKvmSession(evt);
    	    	readSasNewKvmSession(evt);
    	    	break;
    	    
    	    case SasEvent.KvmSessionClosed:
    	    case SasEvent.KvmExclusiveOn:
    	    case SasEvent.KvmExclusiveOff:
    	    	readCausingKvmSession(evt);
    	    	break;
    	    
    	    case SasEvent.UserSessionOpened:
    	    	readSasNewSession(evt);
    	    	break;

    	    case SasEvent.UserLoginFailure:
    	    	readSasUserInfo(evt);
    	    	break;

    	    case SasEvent.ExistingSession:
    	    	readSasExistingSession(evt);
    	    	break;

    	    case SasEvent.ExistingKvmSession:
    	    	readSasExistingKvmSession(evt);
    	    	break;

    	    case SasEvent.Input:
    	    	readCausingKvmSession(evt);
    	    	readSasInputEvent(evt);
    	    	break;
    	    
    	    case SasEventKvmSwitch:
    	    	readSasKvmSwitchEvent(evt);
    	    	break;

    	    /* unknown events; we know the length, so don't worry */
    	    default:
    	    	if (evt.len > 0) {
    	    	    byte dummy[] = new byte[evt.len];
    	    	    is.readFully(dummy);
    	    	}
    	    	logger.println(T._("Unknown forensic event received."));
    	    	System.out.println("Unknown forensic event: 0x" + Integer.toHexString(evt.type));
    	}
    	
    	return evt;
    }
    
    void readSpSessionTimes() throws IOException {
    	is.readUnsignedByte();
    	is.readUnsignedShort();
    	
    	start = readTimestamp();
    	end = readTimestamp();
    }
    
    Date readSpTimestamp() throws IOException {
    	is.readUnsignedByte();
    	is.readUnsignedShort();
    	return readTimestamp();
    }
    
    void sendSpTimestamp(Date pos) throws IOException {
    	sendSpCommand(SpCommandSeek);
    	writeTimestamp(pos);
    }
    
    void readSpCommand() throws IOException {
    	spCommand = is.readByte();
    	spCommandParam8_1 = is.readByte();
    	spCommandParam8_2 = is.readByte();
    	spCommandParam16 = (spCommandParam8_1 << 8) + (spCommandParam8_2 << 0);
    }
    
    void sendSpCommand(int type, int param8_1, int param8_2) throws IOException {
	os.write(SpCommand);
	os.write(type);
	os.write(param8_1);
	os.write(param8_2);
    }

    void sendSpCommand(int type, int param16) throws IOException {
	os.write(SpCommand);
	os.write(type);
	os.writeUnsignedShort(param16);
    }
    
    void sendSpCommand(int type) throws IOException {
    	sendSpCommand(type, 0);
    }
    
    RecordedSessionInfo readSessionInfo() throws IOException {
        int r_colorDepth = is.readUnsignedByte();
        int r_keyboardMode = is.readUnsignedByte();
        boolean useJpeg = is.readUnsignedByte() != 0;
        int jpegLevel = is.readUnsignedByte();
        boolean useZlib = is.readUnsignedByte() != 0;
        int zlibLevel = is.readUnsignedByte();
        int hostLen = is.readUnsignedByte();
        int ipLen = is.readUnsignedByte();
        is.readUnsignedByte();  // pad
        int keyframeInterval = is.readUnsignedShort();
        
        String hostName = readString(hostLen);
        String hostAddress = readString(ipLen);
        
        int colorDepth = RecordedSessionInfo.colorDepthUnknown;
        switch (r_colorDepth) {
            case SpSessionInfoColorDepth1BitBW:
                colorDepth = RecordedSessionInfo.colorDepth1BitBW;
                break;
            case SpSessionInfoColorDepth2BitGrey:
                colorDepth = RecordedSessionInfo.colorDepth2BitGrey;
                break;
            case SpSessionInfoColorDepth3BitGrey:
                colorDepth = RecordedSessionInfo.colorDepth3BitGrey;
                break;
            case SpSessionInfoColorDepth4BitGrey:
                colorDepth = RecordedSessionInfo.colorDepth4BitGrey;
                break;
            case SpSessionInfoColorDepth4BitColor:
                colorDepth = RecordedSessionInfo.colorDepth4BitColor;
                break;
            case SpSessionInfoColorDepth7BitColor:
                colorDepth = RecordedSessionInfo.colorDepth7BitColor;
                break;
            case SpSessionInfoColorDepth15BitColor:
                colorDepth = RecordedSessionInfo.colorDepth15BitColor;
                break;
        }
        
        int keyboardMode = RecordedSessionInfo.keyboardNormal;
        switch (r_keyboardMode) {
            case ConnectionFlagSasKbdNormal:
                keyboardMode = RecordedSessionInfo.keyboardNormal;
                break;
            case ConnectionFlagSasKbdNoKbd:
                keyboardMode = RecordedSessionInfo.keyboardNoKbd;
                break;
            case ConnectionFlagSasKbdFilter:
                keyboardMode = RecordedSessionInfo.keyboardFilter;
                break;
            case ConnectionFlagSasKbdAllFilterPb:
                keyboardMode = RecordedSessionInfo.keyboardAllFilterPb;
                break;
        }
        
        return new RecordedSessionInfo(colorDepth, keyboardMode, useJpeg, jpegLevel, useZlib, zlibLevel, keyframeInterval, hostName, hostAddress);
    }
}
