package nn.pp.rc;

import java.util.*;

public class SasEvent {
    public final static int ExistingSession		= 0x01;
    public final static int ExistingKvmSession		= 0x02;
    public final static int UserLoginFailure		= 0x03;
    public final static int UserSessionOpened		= 0x04;
    public final static int UserSessionClosed		= 0x05;
    public final static int KvmSessionOpened		= 0x06;
    public final static int KvmSessionClosed		= 0x07;
    public final static int KvmExclusiveOn		= 0x08;
    public final static int KvmExclusiveOff		= 0x09;
    public final static int Input			= 0x0A;
    public final static int KvmSwitch			= 0x0B;
    public final static int ResetSessions		= 0x0C;
    
    public final static int InputPointer		= 0x01;
    public final static int InputPointerRelative	= 0x02;
    public final static int InputMouseSync		= 0x03;
    public final static int InputKeyboard		= 0x04;
    
    private int getEventType(int rfbEvent) {
    	switch (rfbEvent) {
    	    case RFBproto.SasEventExistingSession:
    	    	return ExistingSession;
    	    case RFBproto.SasEventExistingKvmSession:
    	    	return ExistingKvmSession;
    	    case RFBproto.SasEventUserLoginFailure:
    	    	return UserLoginFailure;
    	    case RFBproto.SasEventUserSessionOpened:
    	    	return UserSessionOpened;
    	    case RFBproto.SasEventUserSessionClosed:
    	    	return UserSessionClosed;
    	    case RFBproto.SasEventKvmSessionOpened:
    	    	return KvmSessionOpened;
    	    case RFBproto.SasEventKvmSessionClosed:
    	    	return KvmSessionClosed;
    	    case RFBproto.SasEventKvmExclusiveOn:
    	    	return KvmExclusiveOn;
    	    case RFBproto.SasEventKvmExclusiveOff:
    	    	return KvmExclusiveOff;
    	    case RFBproto.SasEventInput:
    	    	return Input;
    	    case RFBproto.SasEventKvmSwitch:
    	    	return KvmSwitch;
    	    case RFBproto.SasEventResetSessions:
    	    	return ResetSessions;
    	}
    	
    	return -1;
    }
    
    public class SasInputEvent {
    	public int type;	/* relative/absolute */
    	public int buttonMask;
    	public int x, y, z;
    	public int keycode;
    	public String syncType;
    	
    	int getInputType(int rfbType) {
    	    switch (rfbType) {
    	    	case RFBproto.PointerEvent:
    	    	    return InputPointer;
    	    	case RFBproto.PointerEventRelative:
    	    	    return InputPointerRelative;
    	    	case RFBproto.MouseSyncEvent:
    	    	    return InputMouseSync;
    	    	case RFBproto.KeyboardEvent:
    	    	    return InputKeyboard;
    	    }
    	    
    	    return 0;
    	}
    	
    	SasInputEvent(int type, int buttonMask, int x, int y, int z) {
    	    this.type = getInputType(type);
    	    this.buttonMask = buttonMask;
    	    this.x = x;
    	    this.y = y;
    	    this.z = z;
    	}

    	SasInputEvent(int type, int keycode) {
    	    this.type = getInputType(type);
    	    this.keycode = keycode;
    	}

    	SasInputEvent(int type, String syncType) {
    	    this.type = getInputType(type);
    	    this.syncType = syncType;
    	}
    	
    	public boolean equals(Object obj) {
    	    if (!(obj instanceof SasInputEvent)) {
    	        return false;
    	    }
    	    SasInputEvent e = (SasInputEvent)obj;
    	    
    	    if (type != e.type) {
    	        return false;
    	    }
    	    
    	    switch (type) {
    	        case InputPointer:
    	        case InputPointerRelative:
    	            return (buttonMask == e.buttonMask) && (x == e.x) && (y == e.y) && (z == e.z);
    	        case InputMouseSync:
    	            // equal if both strings are null
    	            if (syncType == null && e.syncType == null) {
    	                return true;
    	            }
    	            // not equal if one of the strings is null
    	            if (syncType == null || e.syncType == null) {
    	                return false;
    	            }
    	            return syncType.equals(e.syncType);
    	        case InputKeyboard:
    	            return keycode == e.keycode;
    	        default:
    	            // ???
    	            return true;
    	    }
    	}
    }

    public class SasExistingSession {
    	public boolean self;
    	public String user, ip;
    	public Date loginTime;
    	
    	SasExistingSession(boolean self, String user, String ip, Date loginTime) {
    	    this.self = self;
    	    this.user = user;
    	    this.ip = ip;
    	    this.loginTime = loginTime;
    	}

    	public boolean equals(Object obj) {
    	    if (!(obj instanceof SasExistingSession)) {
    	        return false;
    	    }
    	    SasExistingSession e = (SasExistingSession)obj;
    	    return (self == e.self) && user.equals(e.user) && ip.equals(e.ip) && loginTime.equals(e.loginTime);
    	}
    }
    
    public class SasExistingKvmSession {
    	public boolean exclusive;
    	public long kvmSessionId;
    	public Date loginTime;
    	
    	SasExistingKvmSession(long kvmSessionId, boolean exclusive, Date loginTime) {
    	    this.exclusive = exclusive;
    	    this.kvmSessionId = kvmSessionId;
    	    this.loginTime = loginTime;
    	}

    	public boolean equals(Object obj) {
    	    if (!(obj instanceof SasExistingKvmSession)) {
    	        return false;
    	    }
    	    SasExistingKvmSession e = (SasExistingKvmSession)obj;
    	    return (exclusive == e.exclusive) && (kvmSessionId == e.kvmSessionId) && loginTime.equals(e.loginTime);
    	}
    }

    public class SasUserInfo {
    	public String user, ip;
    	
    	SasUserInfo(String user, String ip) {
    	    this.user = user;
    	    this.ip = ip;
    	}

    	public boolean equals(Object obj) {
    	    if (!(obj instanceof SasUserInfo)) {
    	        return false;
    	    }
    	    SasUserInfo e = (SasUserInfo)obj;
    	    return user.equals(e.user) && ip.equals(e.ip);
    	}
    }
    
    public class SasNewSession {
    	public String user, ip;
    	public Date loginTime;
    	
    	SasNewSession(String user, String ip, Date loginTime) {
    	    this.user = user;
    	    this.ip = ip;
    	    this.loginTime = loginTime;
    	}

    	public boolean equals(Object obj) {
    	    if (!(obj instanceof SasNewSession)) {
    	        return false;
    	    }
    	    SasNewSession e = (SasNewSession)obj;
    	    return user.equals(e.user) && ip.equals(e.ip) && loginTime.equals(e.loginTime);
    	}
    }
    
    public class SasNewKvmSession {
        public Date loginTime;

    	SasNewKvmSession(Date loginTime) {
    	    this.loginTime = loginTime;
    	}

    	public boolean equals(Object obj) {
    	    if (!(obj instanceof SasNewKvmSession)) {
    	        return false;
    	    }
    	    SasNewKvmSession e = (SasNewKvmSession)obj;
    	    return loginTime.equals(e.loginTime);
    	}
    }
    
    public class SasKvmSwitchEvent {
    	public int channel, unit, port;
    	
    	SasKvmSwitchEvent(int channel, int unit, int port) {
    	    this.channel = channel;
    	    this.unit = unit;
    	    this.port = port;
    	}

    	public boolean equals(Object obj) {
    	    if (!(obj instanceof SasKvmSwitchEvent)) {
    	        return false;
    	    }
    	    SasKvmSwitchEvent e = (SasKvmSwitchEvent)obj;
    	    return (channel == e.channel) && (unit == e.unit) && (port == e.port);
    	}
    }

    public int type;
    public Date time;
    public long session;
    public int len;
    public SasInputEvent inputEvent;
    public SasExistingSession exSession;
    public SasExistingKvmSession exKvmSession;
    public SasNewSession newSession;
    public SasNewKvmSession newKvmSession;
    public SasUserInfo userInfo;
    public SasKvmSwitchEvent kvmSwitchEvent;
    public long causingKvmSession;
    
    public SasEvent(int type, Date time, long session, int len) {
    	this.type = getEventType(type);
    	this.time = time;
    	this.session = session;
    	this.len = len;
    	causingKvmSession = 0;
    }
    
    void setKeyEvent(int type, int keycode) {
    	inputEvent = new SasInputEvent(type, keycode);
    }
    
    void setMouseEvent(int type, int buttonMask, int x, int y, int z) {
    	inputEvent = new SasInputEvent(type, buttonMask, x, y, z);
    }
    
    void setMouseSyncEvent(int type, int syncType) {
    	String syncTypeString;
    	switch (syncType) {
    	    case RFBproto.MouseSyncNorm:	syncTypeString = "Normal";	break;
    	    case RFBproto.MouseSyncHard:	syncTypeString = "Hard";	break;
    	    case RFBproto.MouseSyncFast:	syncTypeString = "Fast";	break;
    	    default:				syncTypeString = "Unknown";
    	}
    	inputEvent = new SasInputEvent(type, syncTypeString);
    }
    
    void setExistingSession(boolean self, String user, String ip, Date loginTime) {
    	exSession = new SasExistingSession(self, user, ip, loginTime);
    }
    
    void setExistingKvmSession(long kvmSessionId, boolean exclusive, Date loginTime) {
    	exKvmSession = new SasExistingKvmSession(kvmSessionId, exclusive, loginTime);
    }
    
    void setNewSession(String user, String ip, Date loginTime) {
    	newSession = new SasNewSession(user, ip, loginTime);
    }
    
    void setNewKvmSession(Date loginTime) {
        newKvmSession = new SasNewKvmSession(loginTime);
    }
    
    void setUserInfo(String user, String ip) {
    	userInfo = new SasUserInfo(user, ip);
    }
    
    void setCausingKvmSession(long causingKvmSession) {
    	this.causingKvmSession = causingKvmSession;
    }
    
    void setKvmSwitchEvent(int channel, int unit, int port) {
    	this.kvmSwitchEvent = new SasKvmSwitchEvent(channel, unit, port);
    }
    
    public boolean equals(Object obj) {
        if (!(obj instanceof SasEvent)) {
            return false;
        }
        SasEvent e = (SasEvent)obj;
        
        if ((type != e.type) || !time.equals(e.time) || (session != e.session)) {
            return false;
        }
        
        switch (type) {
    	    case ExistingSession:
                // equal if both objects are null
                if (exSession == null && e.exSession == null) {
                    return true;
                }
                // not equal if one of the objects is null
                if (exSession == null || e.exSession == null) {
                    return false;
                }
                return exSession.equals(e.exSession);

    	    case ExistingKvmSession:
                // equal if both objects are null
                if (exKvmSession == null && e.exKvmSession == null) {
                    return true;
                }
                // not equal if one of the objects is null
                if (exKvmSession == null || e.exKvmSession == null) {
                    return false;
                }
                return exKvmSession.equals(e.exKvmSession);

    	    case KvmSessionOpened:
                if (causingKvmSession != e.causingKvmSession) {
                    return false;
                }
                // equal if both objects are null
                if (newKvmSession == null && e.newKvmSession == null) {
                    return true;
                }
                // not equal if one of the objects is null
                if (newKvmSession == null || e.newKvmSession == null) {
                    return false;
                }
                return newKvmSession.equals(e.newKvmSession);

    	    case KvmSessionClosed:
    	    case KvmExclusiveOn:
    	    case KvmExclusiveOff:
    	        return causingKvmSession == e.causingKvmSession;

    	    case UserSessionOpened:
                // equal if both objects are null
                if (newSession == null && e.newSession == null) {
                    return true;
                }
                // not equal if one of the objects is null
                if (newSession == null || e.newSession == null) {
                    return false;
                }
                return newSession.equals(e.newSession);

    	    case UserLoginFailure:
                // equal if both objects are null
                if (userInfo == null && e.userInfo == null) {
                    return true;
                }
                // not equal if one of the objects is null
                if (userInfo == null || e.userInfo == null) {
                    return false;
                }
                return userInfo.equals(e.userInfo);

    	    case Input:
                if (causingKvmSession != e.causingKvmSession) {
                    return false;
                }
                // equal if both objects are null
                if (inputEvent == null && e.inputEvent == null) {
                    return true;
                }
                // not equal if one of the objects is null
                if (inputEvent == null || e.inputEvent == null) {
                    return false;
                }
                return inputEvent.equals(e.inputEvent);

    	    case KvmSwitch:
                // equal if both objects are null
                if (kvmSwitchEvent == null && e.kvmSwitchEvent == null) {
                    return true;
                }
                // not equal if one of the objects is null
                if (kvmSwitchEvent == null || e.kvmSwitchEvent == null) {
                    return false;
                }
                return kvmSwitchEvent.equals(e.kvmSwitchEvent);
                
    	    case UserSessionClosed:
    	    case ResetSessions:
    	        return true;
            default:
                return (len == e.len);
        }
    }
    
    public int hashCode() {
        return time == null ? 0 : (int)time.getTime();
    }
    
    public Date getTime() {
        if ((type == ExistingSession) && (exSession != null)) {
            return exSession.loginTime;
        }
        if ((type == ExistingKvmSession) && (exKvmSession != null)) {
            return exKvmSession.loginTime;
        }
        if ((type == UserSessionOpened) && (newSession != null)) {
            return newSession.loginTime;
        }
        if ((type == KvmSessionOpened) && (newKvmSession != null)) {
            return newKvmSession.loginTime;
        }
        
        return time;
    }
}
