/* GNU gettext for Java
 * Copyright (C) 2001 Free Software Foundation, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

package nn.pp.rc;

import java.lang.reflect.*;
import java.util.*;

/**
 * This class implements the main GNU libintl functions in Java.
 * <P>
 * Using the GNU gettext approach, compiled message catalogs are normal
 * Java ResourceBundle classes and are thus interoperable with standard
 * ResourceBundle based code.
 * <P>
 * The main differences between the Sun ResourceBundle approach and the
 * GNU gettext approach are:
 * <UL>
 *   <LI>In the Sun approach, the keys are abstract textual shortcuts.
 *       In the GNU gettext approach, the keys are the English/ASCII version
 *       of the messages.
 *   <LI>In the Sun approach, the translation files are called
 *       "<VAR>Resource</VAR>_<VAR>locale</VAR>.properties" and have non-ASCII
 *       characters encoded in the Java
 *       <CODE>\</CODE><CODE>u<VAR>nnnn</VAR></CODE> syntax. Very few editors
 *       can natively display international characters in this format. In the
 *       GNU gettext approach, the translation files are called
 *       "<VAR>Resource</VAR>.<VAR>locale</VAR>.po"
 *       and are in the encoding the translator has chosen. Many editors
 *       can be used. There are at least three GUI translating tools
 *       (Emacs PO mode, KDE KBabel, GNOME gtranslator).
 *   <LI>In the Sun approach, the function
 *       <CODE>ResourceBundle.getString</CODE> throws a
 *       <CODE>MissingResourceException</CODE> when no translation is found.
 *       In the GNU gettext approach, the <CODE>gettext</CODE> function
 *       returns the (English) message key in that case.
 *   <LI>In the Sun approach, there is no support for plural handling.
 *       Even the most elaborate MessageFormat strings cannot provide decent
 *       plural handling. In the GNU gettext approach, we have the
 *       <CODE>ngettext</CODE> function.
 * </UL>
 * <P>
 * To compile GNU gettext message catalogs into Java ResourceBundle classes,
 * the <CODE>msgfmt</CODE> program can be used.
 *
 * @author Bruno Haible
 */

public class T {
    static ResourceBundle langCatalog = null;
    static String charset = "UTF-8";
    
    public void T() {
    }
    
    /**
     * Returns the translation of <VAR>msgid</VAR>.
     * @param catalog a ResourceBundle
     * @param msgid the key string to be translated, an ASCII string
     * @return the translation of <VAR>msgid</VAR>, or <VAR>msgid</VAR> if
     *         none is found
     */
    public static String gettext (ResourceBundle catalog, String msgid) {
        if(catalog == null) {
            return msgid;
        }
        try {
            String result = (String)catalog.getObject(msgid);
            if (result != null) {
                return result;
            }
        }
        catch (MissingResourceException e) {
        }
        
        return msgid;
    }
    
    public static String _(String s) {
    	return gettext(T.langCatalog, s);
    }
    
    public static String N_(String s) {
    	return s;
    }
    
    public static void loadCatalog(String lang) {
    	ResourceBundle catalog = null;
    	String className = "nn.pp.rclang.Lang_" + lang;
    	try {
    	    catalog = (ResourceBundle)Class.forName(className).newInstance();
    	    System.out.println("Translator: " + T._("loaded language class:") +
    	                       " " + className);
    	}
    	catch (Exception e) {
    	    System.out.println("Translator: " + T._("could not load language class:") +
    	                       " " + className);
    	    e.printStackTrace();
    	}
    	
    	T.langCatalog = catalog;
    }
    
    synchronized public static String getCharset() {
        return T.charset;
    }
    
    synchronized public static void setCharset(String charset) {
        System.out.println("new charset: " + charset);
        T.charset = charset.toUpperCase();
    }
}
