package nn.pp.rcrdp;

import java.io.*;
import java.net.*;
import java.text.*;
import nn.pp.rc.T;

/**
 * class ISOLayer
 *
 * the ISO-Layer of the MS RDP protocol
 */
 
public class ISOLayer {
    private static final int sizeOfISOHdr = 7;
    
    private TCPLayer tcpLayer;
    
    private boolean debug = false;
    
    private int recvMsgCode;
    
    private void debug(String s) {
    	if(debug) System.out.println(s);
    }
    
    // Constructor
    public ISOLayer() {
    	tcpLayer = new TCPLayer();
    }
    
    // get the TCPLayer
    public TCPLayer getTCPLayer() {
    	return tcpLayer;
    }
    
    // private methods
    
    // Send a self-contained ISO PDU
    private void sendMsg(int code) throws IOException {
	RDPStream s;

	s = tcpLayer.init(11);

	s.outUInt8(3);		/* version */
	s.outUInt8(0);		/* reserved */
	s.outUInt16BE(11);	/* length */

	s.outUInt8(6);		/* hdrlen */
	s.outUInt8(code);
	s.outUInt16(0);		/* dst_ref */
	s.outUInt16(0);		/* src_ref */
	s.outUInt8(0);		/* class */

	s.markEnd();
	tcpLayer.send(s);
    }
    
    // Send a connection request
    private void sendConnectionRequest(String username) throws IOException {
    	/*
    	sendMsg(RDPConstants.isoPDUCR);
    	*/
    	
    	int len = 30 + username.length();
    	RDPStream s = tcpLayer.init(len);
    	
	s.outUInt8(3);		// version
	s.outUInt8(0);		// reserved
	s.outUInt16BE(len);	// length

    	s.outUInt8(len - 5);	// hdrlen
    	s.outUInt8(RDPConstants.isoPDUCR);
    	
	s.outUInt16(0);		// dst_ref
	s.outUInt16(0);		// src_ref
	s.outUInt8(0);		// class
	
	String str = "Cookie: mstshash=";
	s.outByteStringNoTerm(str, str.length());
	s.outByteStringNoTerm(username, username.length());
	
	s.outUInt8(0x0d);
	s.outUInt8(0x0a);
	
	s.markEnd();
	tcpLayer.send(s);
    }
    
    // Receive a message on the ISO layer, return code
    // we need 2 return parameters here, that's why we use the Object[] as return value
    private RDPStream recvMsg() throws IOException {
    	RDPStream s = null;
    	
    	s = tcpLayer.receive(4);
    	if(s == null) return null;
    	
    	int version = s.inUInt8();	//version
    	if(version != 3) {
    	    throw new IOException(MessageFormat.format(T._("Wrong RDP version number, got {0}"),
    	                          new Object[] { new Integer(version) }));
    	}
    	s.incrementPosition(1);		//pad
    	
    	int length = s.inUInt16BE();	//length
    	
    	s = tcpLayer.receive(length - 4);
    	if(s == null) return null;
    	
    	s.incrementPosition(1);		//hdrlen
    	recvMsgCode = s.inUInt8();	//code
    	
    	if(recvMsgCode == RDPConstants.isoPDUDT) {
    	    s.incrementPosition(1);	//eot
	}
	else {
    	    s.incrementPosition(5);	//dst_ref, src_ref, class
	}
    	
    	return s;
    }
    
    // public methods
    
    // Initialise ISO transport data packet
    public RDPStream init(int len) {
    	RDPStream s = tcpLayer.init(len + sizeOfISOHdr);
    	s.pushLayer(RDPStream.isoHdrID, sizeOfISOHdr);
    	
    	return s;
    }
    
    public RDPStream initNew(int len) {
    	RDPStream s = tcpLayer.initNew(len + sizeOfISOHdr);
    	s.pushLayer(RDPStream.isoHdrID, sizeOfISOHdr);
    	
    	return s;
    }
    
    // Establish a connection up to the ISO layer
    public void connect(String server, int port, String username) throws IOException {
    	tcpLayer.connect(server, port);
    	
    	sendConnectionRequest(username);
    	
    	recvMsg();
    	
    	if(recvMsgCode != RDPConstants.isoPDUCC) {
    	    throw new IOException(T._("Error: Connection not confirmed"));
    	}
    }
    
    // Disconnect from the ISO layer
    public void disconnect() {
    	try {
    	    sendMsg(RDPConstants.isoPDUDR);
    	}
    	catch (IOException e) {
    	    //nothing; we want to disconnect here, so we ignore the error
    	}
    	tcpLayer.disconnect();
    }
    
    // Send an ISO data PDU
    public void send(RDPStream s) throws IOException {
    	int len;
    	
    	s.popLayer(RDPStream.isoHdrID);
    	len = s.getEnd() - s.getPosition();
    	
    	s.outUInt8(3);				//version
    	s.outUInt8(0);				//reserved
    	s.outUInt16BE(len);			//lenght
    	
    	s.outUInt8(2);				//hdrlen
    	s.outUInt8(RDPConstants.isoPDUDT);	//code
    	s.outUInt8(0x80);			//eot
    	
    	tcpLayer.send(s);
    }
    
    // Receive ISO transport data packet
    public RDPStream receive() throws IOException {
    	RDPStream s = recvMsg();
    	
    	if(recvMsgCode != RDPConstants.isoPDUDT) {
    	    throw new IOException(T._("Error: expected isoPDUDT"));
    	}
    	    	
    	return s;
    }
    
}
