package nn.pp.rcrdp;

import java.io.*;
import java.net.*;
import java.security.*;
import java.math.*;

import nn.pp.rcrdp.crypto.*;
import nn.pp.rc.T;

class Licence {
    private boolean licenceIssued;
    private String hostname;
    private String username;
    private SecureLayer secureLayer;
    
    private byte licenceKey[];
    private byte licenceSignKey[];
    
    private byte inToken[];
    private byte inSig[];
    
    public boolean licenceIssued() {
    	return licenceIssued;
    }
    
    // constructor
    public Licence(SecureLayer sec, String host, String user) {
    	licenceIssued = false;
    	hostname = host;
    	username = user;
    	secureLayer = sec;
    	
    	licenceSignKey = new byte[16];
    	licenceKey = new byte[16];
    }
    
    // Generate a session key and RC4 keys, given client and server randoms
    private void generateKeys(byte clientKey[], byte serverKey[], byte clientRSA[]) throws CryptoException {
    	// Generate session key - two rounds of sec_hash_48
    	byte tempHash[] = secureLayer.hash48(clientRSA, clientKey, serverKey, 65);
    	byte sessionKey[] = secureLayer.hash48(tempHash, serverKey, clientKey, 65);
    	
    	// Store first 16 bytes of session key, for generating signatures
    	System.arraycopy(sessionKey, 0, licenceSignKey, 0, 16);
    	
    	licenceKey = secureLayer.hash16(sessionKey, clientKey, serverKey, 16);
    }
    
    // generate a HWID
    private byte[] generateHwid() {
    	byte hwid[] = new byte[RDPConstants.licenceHwidSize];
    	
    	secureLayer.bufOutUInt32(hwid, 2);
    	byte host[] = null;
    	try { 
    	    host = hostname.getBytes("US-ASCII");
    	} catch (UnsupportedEncodingException e) {
    	    return hwid;
    	}

    	System.arraycopy(host, 0, hwid, 4, Math.min(host.length, RDPConstants.licenceHwidSize - 4));
    	
    	return hwid;
    }
    
    // Present an existing licence to the server
    // private void present()
    //   not implemented because we can't save a licence from within the applet
    
    // Send a licence request packet
    private void sendRequest(byte clientRandom[], byte rsaData[], String user, String host) throws IOException {
    	int flags = RDPConstants.secLicenceNeg;
    	int hostlen = host.length() + 1;
    	int userlen = user.length() + 1;
    	int len = 128 + userlen + hostlen;
    	
    	RDPStream s = secureLayer.init(flags, len + 2);
    	
    	s.outUInt8(RDPConstants.licenceTagRequest);
    	s.outUInt8(2);		// version
    	s.outUInt16LE(len);
    	
    	s.outUInt32LE(1);
    	s.outUInt16(0);
    	s.outUInt16LE(0xff01);
    	
    	s.outUInt8Array(clientRandom, 0, RDPConstants.secRandomSize);
    	s.outUInt16(0);
    	s.outUInt16LE(RDPConstants.secModulusSize + RDPConstants.secPaddingSize);
    	s.outUInt8Array(rsaData, 0, RDPConstants.secModulusSize);
    	s.incrementPosition(RDPConstants.secPaddingSize);
    	
    	s.outUInt16LE(RDPConstants.licenceTagUser);
    	s.outUInt16LE(userlen);
    	s.outByteStringNoTerm(user, userlen);
    	
    	s.outUInt16LE(RDPConstants.licenceTagHost);
    	s.outUInt16LE(hostlen);
    	s.outByteStringNoTerm(host, hostlen);
    	
    	s.markEnd();
    	
    	secureLayer.send(s, flags);
    }
    
    // Process a licence demand packet
    private void processDemand(RDPStream s) throws IOException, CryptoException {
    	byte nullData[] = new byte[RDPConstants.secModulusSize];
    	byte serverRandom[] = new byte[RDPConstants.secRandomSize];
    	
    	s.inUInt8Array(serverRandom, 0, RDPConstants.secRandomSize);
    	generateKeys(nullData, serverRandom, nullData);
    	
    	// the original C code of rdesktop tries to load a stored
    	// licence from disk; this is not possible from within the
    	// Java applet, so we have to request a new one in any case
    	sendRequest(nullData, nullData, username, hostname);
    }
    
    // Send an authentication response packet
    private void sendAuthresp(byte token[], byte cryptHwid[], byte signature[]) throws IOException {
    	int secFlags = RDPConstants.secLicenceNeg;
    	int len = RDPConstants.licenceTokenSize + RDPConstants.licenceHwidSize + RDPConstants.licenceSignatureSize + 12;
    	
    	RDPStream s = secureLayer.init(secFlags, len + 2);
    	
    	s.outUInt8(RDPConstants.licenceTagAuthresp);
    	s.outUInt8(2);	// version
    	s.outUInt16LE(len);
    	
    	s.outUInt16LE(1);
    	s.outUInt16LE(RDPConstants.licenceTokenSize);
    	s.outUInt8Array(token, 0, RDPConstants.licenceTokenSize);
    	
    	s.outUInt16LE(1);
    	s.outUInt16LE(RDPConstants.licenceHwidSize);
    	s.outUInt8Array(cryptHwid, 0, RDPConstants.licenceHwidSize);
    	
    	s.outUInt8Array(signature, 0, RDPConstants.licenceSignatureSize);
    	
    	s.markEnd();
    	secureLayer.send(s, secFlags);
    }
    
    // Parse an authentication request packet
    private boolean parseAuthreq(RDPStream s) throws IOException {
    	int tokenLen;
    	
    	 s.incrementPosition(6);	// unknown: f8 3d 15 00 04 f6
    	 
    	 tokenLen = s.inUInt16LE();
    	 if(tokenLen != RDPConstants.licenceTokenSize) {
    	     throw new IOException(T._("RDP: Licence: wrong token length received"));
    	 }
    	 
    	 inToken = new byte[tokenLen];
    	 s.inUInt8Array(inToken, 0, tokenLen);
    	 inSig = new byte[RDPConstants.licenceSignatureSize];
    	 s.inUInt8Array(inSig, 0, RDPConstants.licenceSignatureSize);
    	 
    	 return s.checkSize();
    }
    
    // Process an authentication request packet
    private void processAuthreq(RDPStream s) throws IOException, CryptoException {
    	byte outToken[] = new byte[RDPConstants.licenceTokenSize];
    	byte decryptToken[] = new byte[RDPConstants.licenceTokenSize];
    	byte hwid[];
    	byte cryptHwid[] = new byte[RDPConstants.licenceHwidSize];
    	byte sealedBuffer[] = new byte[RDPConstants.licenceTokenSize + RDPConstants.licenceHwidSize];
    	byte outSig[] = new byte[RDPConstants.licenceSignatureSize];
    	RC4 cryptKey = new RC4();
    	
    	// Parse incoming packet and save the encrypted token
    	if(!parseAuthreq(s)) {
    	    throw new IOException(T._("RDP: Licence: Authentication request failed"));
    	}
    	System.arraycopy(inToken, 0, outToken, 0, RDPConstants.licenceTokenSize);
    	
    	// Decrypt the token. It should read TEST in Unicode
    	byte tempKey[] = new byte[licenceKey.length];
    	System.arraycopy(licenceKey, 0, tempKey, 0, licenceKey.length);
    	cryptKey.engineInitDecrypt(tempKey);
    	cryptKey.crypt(inToken, 0, RDPConstants.licenceTokenSize, decryptToken, 0);
    	
    	// Generate a signature for a buffer of token and HWID
    	hwid = generateHwid();
    	System.arraycopy(decryptToken, 0, sealedBuffer, 0, RDPConstants.licenceTokenSize);
    	System.arraycopy(hwid, 0, sealedBuffer, RDPConstants.licenceTokenSize, RDPConstants.licenceHwidSize);
    	outSig = secureLayer.sign(16, licenceSignKey, 16, sealedBuffer, sealedBuffer.length);
    	
    	// Now encrypt the HWID
    	System.arraycopy(licenceKey, 0, tempKey, 0, licenceKey.length);
    	cryptKey.engineInitDecrypt(tempKey);
    	cryptKey.crypt(hwid, 0, RDPConstants.licenceHwidSize, cryptHwid, 0);
    	
    	sendAuthresp(outToken, cryptHwid, outSig);
    }
    
    // Process an licence issue packet
    private void processIssue(RDPStream s) throws CryptoException {
    	int len;
    	int check;
    	RC4 cryptKey = new RC4();
    	
    	s.incrementPosition(2);		// 3d 45 - unknown
    	len = s.inUInt16LE();
    	if(!s.checkRemaining(len)) {
    	    return;
    	}
    	
    	byte tempKey[] = new byte[licenceKey.length];
    	System.arraycopy(licenceKey, 0, tempKey, 0, licenceKey.length);
    	cryptKey.engineInitDecrypt(tempKey);
    	byte buffer[] = new byte[len];
    	s.inUInt8ArrayNoCount(buffer, 0, s.getPosition(), len);
    	cryptKey.crypt(buffer, 0, len, buffer, 0);
    	s.outUInt8ArrayNoCount(buffer, 0, s.getPosition(), len);
    	
    	check = s.inUInt16LE();
    	if(check != 0) {
    	    return;
    	}
    	
    	licenceIssued = true;
    	
    	// original rdesktop C-Code saves the licence here;
    	// this is not possible in the applet, but we read it from the stream
    	s.incrementPosition(2);		// pad
    	// advance to fourth string
    	len = 0;
    	for(int i = 0; i < 4; i++) {
    	    s.incrementPosition(len);
    	    len = s.inUInt32LE();
    	    if(!s.checkRemaining(len)) {
    	    	return;
    	    }
    	}
    	s.incrementPosition(len);
    }
    
    // Process a licence packet
    public void process(RDPStream s) throws IOException {
    	int tag = s.inUInt8();
    	s.incrementPosition(3);		// version, length
    	try {
    	    switch(tag) {
    	    	case RDPConstants.licenceTagDemand:
    	    	    processDemand(s);
    	    	    break;
    	    	case RDPConstants.licenceTagAuthreq:
    	    	    processAuthreq(s);
    	    	    break;
    	    	case RDPConstants.licenceTagIssue:
    	    	    processIssue(s);
    	    	    break;
    	    	case RDPConstants.licenceTagReissue:
    	    	case RDPConstants.licenceTagResult:
    	    	    break;
    	    	default:
    	    	    System.out.println("RDP: Licence: unknown licence tag received");
    	    }
    	}
    	catch (CryptoException e) {
    	    throw new IOException(T._("RDP Licence: crypto error"));
    	}
    }
}
