package nn.pp.rcrdp;

import java.awt.*;

// the needed structures must be declared as classes in java
// we set all the members public here to make the access easier
class DestBltOrder {
    public int x;
    public int y;
    public int cx;
    public int cy;
    public int opcode;
    
    public void reset() {
    	x = 0;
    	y = 0;
    	cx = 0;
    	cy = 0;
    	opcode = 0;
    }
}

class PatBltOrder {
    public int x;
    public int y;
    public int cx;
    public int cy;
    public int opcode;
    public int bgColor;
    public int fgColor;
    public RDPBrush brush = new RDPBrush();
    
    public void reset() {
    	x = 0;
    	y = 0;
    	cx = 0;
    	cy = 0;
    	opcode = 0;
    	fgColor = 0;
    	bgColor = 0;
    	brush = new RDPBrush();
    }
}

class ScreenBltOrder {
    public int x;
    public int y;
    public int cx;
    public int cy;
    public int opcode;
    public int srcx;
    public int srcy;
    
    public void reset() {
    	x = 0;     
    	y = 0;     
    	cx = 0;    
    	cy = 0;    
    	opcode = 0;
    	srcx = 0;  
    	srcy = 0; 
    } 
}

class LineOrder {
    public int mixMode;
    public int startX;
    public int startY;
    public int endX;
    public int endY;
    public int bgColor;
    public int opcode;
    public RDPPen pen = new RDPPen();
    
    public void reset() {
    	mixMode = 0; 
    	startX = 0;  
    	startY = 0;  
    	endX = 0;    
    	endY = 0;    
    	bgColor = 0;
    	opcode = 0;  
    	pen = new RDPPen();
    }
}

class RectOrder {
    public int x;
    public int y;
    public int cx;
    public int cy;
    public int color;
    
    public void reset() {
    	x = 0;     
    	y = 0;     
    	cx = 0;    
    	cy = 0;    
    	color = 0;
    }
}

class DeskSaveOrder {
    public int offset;
    public int left;
    public int top;
    public int right;
    public int bottom;
    public int action;
    
    public void reset() {
    	offset = 0;
    	left = 0;  
    	top = 0;   
    	right = 0; 
    	bottom = 0;
    	action = 0;
    }
}

class TriBltOrder {
    public int colorTable;
    public int cacheID;
    public int x;
    public int y;
    public int cx;
    public int cy;
    public int opcode;
    public int srcx;
    public int srcy;
    public int bgColor;
    public int fgColor;
    public RDPBrush brush = new RDPBrush();
    public int cacheIndex;
    public int unknown;
    
    public void reset() {
    	colorTable = 0;
    	cacheID = 0;   
    	x = 0;         
    	y = 0;         
    	cx = 0;        
    	cy = 0;        
    	opcode = 0;    
    	srcx = 0;      
    	srcy = 0;      
    	bgColor = 0;   
    	fgColor = 0;   
    	brush = new RDPBrush();
    	cacheIndex = 0;
    	unknown = 0;  
    } 
}

class MemBltOrder {
    public int colorTable;
    public int cacheID;
    public int x;
    public int y;
    public int cx;
    public int cy;
    public int opcode;
    public int srcx;
    public int srcy;
    public int cacheIndex;
    
    public void reset() {
    	colorTable = 0; 
    	cacheID = 0;    
    	x = 0;          
    	y = 0;          
    	cx = 0;         
    	cy = 0;         
    	opcode = 0;     
    	srcx = 0;       
    	srcy = 0;       
    	cacheIndex = 0; 
    }
}

class PolyLineOrder {
    static final int maxData = 256;
    
    public int x;
    public int y;
    public int opcode;
    public int fgColor;
    public int lines;
    public int dataSize;
    public byte data[] = new byte[maxData];
    
    public void reset() {
    	x = 0;                          
    	y = 0;                          
    	opcode = 0;                     
    	fgColor = 0;                    
    	lines = 0;                      
    	dataSize = 0;                   
    	data = new byte[maxData];
    }
}

class Text2Order {
    static final int maxText = 256;
    
    public int font;
    public int flags;
    public int mixMode;
    public int unknown;
    public int fgColor;
    public int bgColor;
    public int clipLeft;
    public int clipTop;
    public int clipRight;
    public int clipBottom;
    public int boxLeft;
    public int boxTop;
    public int boxRight;
    public int boxBottom;
    public int x;
    public int y;
    public int length;
    public byte text[] = new byte[maxText];
    
    public void reset() {
    	font = 0;                       
    	flags = 0;                      
    	mixMode = 0;                    
    	unknown = 0;                    
    	fgColor = 0;                    
    	bgColor = 0;                    
    	clipLeft = 0;                   
    	clipTop = 0;                    
    	clipRight = 0;                  
    	clipBottom = 0;                 
    	boxLeft = 0;                    
    	boxTop = 0;                     
    	boxRight = 0;                   
    	boxBottom = 0;                  
    	x = 0;                          
    	y = 0;                          
    	length = 0;                     
    	text = new byte[maxText];
    }
}

class RDPOrderState {
    public int orderType;
    public RDPBounds bounds = new RDPBounds();
    
    public DestBltOrder destBlt = new DestBltOrder();
    public PatBltOrder patBlt = new PatBltOrder();
    public ScreenBltOrder screenBlt = new ScreenBltOrder();
    public LineOrder line = new LineOrder();
    public RectOrder rect = new RectOrder();
    public DeskSaveOrder deskSave = new DeskSaveOrder();
    public MemBltOrder memBlt = new MemBltOrder();
    public TriBltOrder triBlt = new TriBltOrder();
    public PolyLineOrder polyLine = new PolyLineOrder();
    public Text2Order text2 = new Text2Order();
    
    public void reset() {
    	destBlt.reset();
    	patBlt.reset();
    	screenBlt.reset();
    	line.reset();
    	rect.reset();
    	deskSave.reset();
    	memBlt.reset();
    	triBlt.reset();
    	polyLine.reset();
    	text2.reset();
    	bounds.reset();
    	
    	orderType = RDPConstants.rdpOrderPatBlt;
    }
}

// the Order class
// it processes orders and is used by RDPProto
class Orders {
    private boolean debug = false;
    
    private RDPRenderer rdr;
    private Component comp;
    
    private Cache cache;
    private BitmapDecompressor bitmapDecompressor;
    
    private RDPOrderState orderState;
    
    private byte[] byteBuffer = null;
    private int[] intBuffer = null;
    
    // Constructor
    public Orders(Cache cache, BitmapDecompressor bitmapDecompressor, RDPRenderer rdr, Component panel) {
    	orderState = new RDPOrderState();
    	this.cache = cache;
    	this.bitmapDecompressor = bitmapDecompressor;
    	this.rdr = rdr;
    	this.comp = panel;
    }
    
    // private functions
    
    // debugging
    private void debug(String s) {
    	if(debug) System.out.println(s);
    }
    
    // get and allocate (if necessary) the internal buffer
    private byte[] getByteBuffer(int size) {
    	if(byteBuffer == null || byteBuffer.length < size) {
    	    byteBuffer = new byte[size];
    	}
    	
    	return byteBuffer;
    }
    
    private int[] getIntBuffer(int size) {
    	if(intBuffer == null || intBuffer.length < size) {
    	    intBuffer = new int[size];
    	}
    	
    	return intBuffer;
    }
    
    // raster operations
    private int rop2S(int rop3) {
    	return (rop3 & 0xf);
    }
    
    private int rop2P(int rop3) {
    	return ((rop3 & 0x3) | ((rop3 & 0x30) >> 2));
    }
    
    // Read field indicating which parameters are present
    private int inPresent(RDPStream s, int flags, int size) {
    	int bits;
    	
    	if((flags & RDPConstants.rdpOrderSmall) != 0) {
    	    size--;
    	}
    	
    	if((flags & RDPConstants.rdpOrderTiny) != 0) {
    	    if(size < 2) {
    	    	size = 0;
    	    }
    	    else {
    	    	size -= 2;
    	    }
    	}
    	
    	int present = 0;
    	for(int i = 0; i < size; i++) {
    	    bits = s.inUInt8();
    	    present |= bits << (i * 8);
    	}
    	
    	return present;
    }
    
    // Read a co-ordinate (16-bit, or 8-bit delta)
    private int inCoord(RDPStream s, int coord, boolean delta) {
    	if(delta) {
    	    int change = s.inSInt8();
    	    coord += change;
    	}
    	else {
    	    coord = s.inSInt16LE();
    	}
    	
    	return coord;
    }
    
    // Read a colour entry
    private int inColor(RDPStream s) {
    	int i, color;
    	
    	i = s.inUInt8();
    	color = i;
    	i = s.inUInt8();
    	color |= (i << 8);
    	i = s.inUInt8();
    	color |= (i << 16);
    	
    	return color;
    }
    
    // Parse bounds information
    private void parseBounds(RDPStream s, RDPBounds bounds) {
    	int present = s.inUInt8();
    	
    	if((present & 0x01) != 0) {
    	    bounds.setLeft(inCoord(s, bounds.getLeft(), false));
    	}
    	else if((present & 0x10) != 0) {
    	    bounds.setLeft(inCoord(s, bounds.getLeft(), true));
    	}
    	
    	if((present & 0x02) != 0) {
    	    bounds.setTop(inCoord(s, bounds.getTop(), false));
    	}
    	else if((present & 0x20) != 0) {
    	    bounds.setTop(inCoord(s, bounds.getTop(), true));
    	}
    	
    	if((present & 0x04) != 0) {
    	    bounds.setRight(inCoord(s, bounds.getRight(), false));
    	}
    	else if((present & 0x40) != 0) {
    	    bounds.setRight(inCoord(s, bounds.getRight(), true));
    	}
    	
    	if((present & 0x08) != 0) {
    	    bounds.setBottom(inCoord(s, bounds.getBottom(), false));
    	}
    	else if((present & 0x80) != 0) {
    	    bounds.setBottom(inCoord(s, bounds.getBottom(), true));
    	}
    	
    }
    
    // Parse a pen
    private void parsePen(RDPStream s, RDPPen pen, int present) {
    	if((present & 1) != 0) {
    	    pen.setStyle(s.inUInt8());
    	}
    	if((present & 2) != 0) {
    	    pen.setWidth(s.inUInt8());
    	}
    	if((present & 4) != 0) {
    	    pen.setColor(inColor(s));
    	}
    }
    
    // Parse a brush
    private void parseBrush(RDPStream s, RDPBrush brush, int present) {
    	if((present & 0x01) != 0) {
    	    brush.setXOrigin(s.inUInt8());
    	}
    	if((present & 0x02) != 0) {
    	    brush.setYOrigin(s.inUInt8());
    	}
    	if((present & 0x04) != 0) {
    	    brush.setStyle(s.inUInt8());
    	}
    	if((present & 0x08) != 0) {
    	    s.inUInt8Array(brush.getPattern(), 0, 1);
    	}
    	if((present & 0x10) != 0) {
    	    s.inUInt8Array(brush.getPattern(), 1, 7);
    	}
    }
    
    // Process a destination blt order
    private void processDestBlt(RDPStream s, DestBltOrder os, int present, boolean delta) {
    	if((present & 0x01) != 0) {
    	    os.x = inCoord(s, os.x, delta);
    	}
    	if((present & 0x02) != 0) {
    	    os.y = inCoord(s, os.y, delta);
    	}
    	if((present & 0x04) != 0) {
    	    os.cx = inCoord(s, os.cx, delta);
    	}
    	if((present & 0x08) != 0) {
    	    os.cy = inCoord(s, os.cy, delta);
    	}
    	if((present & 0x10) != 0) {
    	    os.opcode = s.inUInt8();
    	}
    	
    	//debug("RDP Orders DestBlt: opcode = 0x" + Integer.toHexString(os.opcode & 0xff) + ", x = " + os.x + ", y = " + os.y + ", cx = " + os.cx + ", cy = " + os.cy);
    	
    	rdr.destBlt(rop2S(os.opcode), os.x, os.y, os.cx, os.cy);
    }
    
    // Process a pattern blt order
    private void processPatBlt(RDPStream s, PatBltOrder os, int present, boolean delta) {
    	if((present & 0x01) != 0) {
    	    os.x = inCoord(s, os.x, delta);
    	}
    	if((present & 0x02) != 0) {
    	    os.y = inCoord(s, os.y, delta);
    	}
    	if((present & 0x04) != 0) {
    	    os.cx = inCoord(s, os.cx, delta);
    	}
    	if((present & 0x08) != 0) {
    	    os.cy = inCoord(s, os.cy, delta);
    	}
    	if((present & 0x10) != 0) {
    	    os.opcode = s.inUInt8();
    	}
    	if((present & 0x20) != 0) {
    	    os.bgColor = inColor(s);
    	}
    	if((present & 0x40) != 0) {
    	    os.fgColor = inColor(s);
    	}
    	
    	parseBrush(s, os.brush, present >> 7);
    	
    	//debug("RDP Orders PatBlt: opcode = 0x" + Integer.toHexString(os.opcode & 0xff) + ", x = " + os.x + ", y = " + os.y + ", cx = " + os.cx + ", cy = " + os.cy + ", brushStyle = " + os.brush.getStyle() + ", bgColor = 0x" + Integer.toHexString(os.bgColor) + ", fgColor = 0x" + Integer.toHexString(os.fgColor));
    	
    	rdr.patBlt(rop2P(os.opcode), os.x, os.y, os.cx, os.cy, os.brush, os.bgColor, os.fgColor);
    }
    
    // Process a screen blt order
    private void processScreenBlt(RDPStream s, ScreenBltOrder os, int present, boolean delta) {
    	if((present & 0x01) != 0) {
    	    os.x = inCoord(s, os.x, delta);
    	}
    	if((present & 0x02) != 0) {
    	    os.y = inCoord(s, os.y, delta);
    	}
    	if((present & 0x04) != 0) {
    	    os.cx = inCoord(s, os.cx, delta);
    	}
    	if((present & 0x08) != 0) {
    	    os.cy = inCoord(s, os.cy, delta);
    	}
    	if((present & 0x10) != 0) {
    	    os.opcode = s.inUInt8();
    	}
    	if((present & 0x20) != 0) {
    	    os.srcx = inCoord(s, os.srcx, delta);
    	}
    	if((present & 0x40) != 0) {
    	    os.srcy = inCoord(s, os.srcy, delta);
    	}
    	
    	//debug("RDP Orders ScreenBlt: opcode = 0x" + Integer.toHexString(os.opcode & 0xff) + ", x = " + os.x + ", y = " + os.y + ", cx = " + os.cx + ", cy = " + os.cy + ", srcx = " + os.srcx + ", srcy = " + os.srcy);

    	rdr.screenBlt(rop2S(os.opcode), os.x, os.y, os.cx, os.cy, os.srcx, os.srcy);
    }
    
    // Process a line order
    private void processLine(RDPStream s, LineOrder os, int present, boolean delta) {
    	if((present & 0x01) != 0) {
    	    os.mixMode = s.inUInt16LE();
    	}
    	if((present & 0x02) != 0) {
    	    os.startX = inCoord(s, os.startX, delta);
    	}
    	if((present & 0x04) != 0) {
    	    os.startY = inCoord(s, os.startY, delta);
    	}
    	if((present & 0x08) != 0) {
    	    os.endX = inCoord(s, os.endX, delta);
    	}
    	if((present & 0x10) != 0) {
    	    os.endY = inCoord(s, os.endY, delta);
    	}
    	if((present & 0x20) != 0) {
    	    os.bgColor = inColor(s);
    	}
    	if((present & 0x40) != 0) {
    	    os.opcode = s.inUInt8();
    	}
    	
    	parsePen(s, os.pen, present >> 7);
    	
    	//debug("RDP Orders Line: opcode = 0x" + Integer.toHexString(os.opcode & 0xff) + ", startX = " + os.startX + ", startY = " + os.startY + ", endX = " + os.endX + ", endY = " + os.endY);
    	
    	if(os.opcode < 0x01 || os.opcode > 0x10) {
    	    System.out.println("RDP Orders processLine: bad rop2: 0x" + Integer.toHexString(os.opcode & 0xff));
    	    return;
    	}
    	
    	rdr.drawLine(os.opcode - 1, os.startX, os.startY, os.endX, os.endY, os.pen);
    }
    
    // Process an opaque rectangle order
    private void processRect(RDPStream s, RectOrder os, int present, boolean delta) {
    	int i = 0;
    	
    	if((present & 0x01) != 0) {
    	    os.x = inCoord(s, os.x, delta);
    	}
    	if((present & 0x02) != 0) {
    	    os.y = inCoord(s, os.y, delta);
    	}
    	if((present & 0x04) != 0) {
    	    os.cx = inCoord(s, os.cx, delta);
    	}
    	if((present & 0x08) != 0) {
    	    os.cy = inCoord(s, os.cy, delta);
    	}
    	
    	if((present & 0x10) != 0) {
    	    i = s.inUInt8();
    	    os.color = (os.color & 0xffffff00) | i;
    	}
    	
    	if((present & 0x20) != 0) {
    	    i = s.inUInt8();
    	    os.color = (os.color & 0xffff00ff) | (i << 8);
	}
	
	if((present & 0x40) != 0) {
    	    i = s.inUInt8();
    	    os.color = (os.color & 0xff00ffff) | (i << 16);
	}
	
	//debug("RDP Orders Rect: x = " + os.x + ", y = " + os.y + ", cx = " + os.cx + ", cy = " + os.cy + "color = 0x" + Integer.toHexString(os.color));
	
	rdr.drawRect(os.x, os.y, os.cx, os.cy, os.color);
    }
    
    // Process a desktop save order
    private void processDeskSave(RDPStream s, DeskSaveOrder os, int present, boolean delta) {
    	int width, height;
    	
    	if((present & 0x01) != 0) {
    	    os.offset = s.inUInt32LE();
    	}
    	if((present & 0x02) != 0) {
    	    os.left = inCoord(s, os.left, delta);
    	}
    	if((present & 0x04) != 0) {
    	    os.top = inCoord(s, os.top, delta);
    	}
    	if((present & 0x08) != 0) {
    	    os.right = inCoord(s, os.right, delta);
    	}
    	if((present & 0x10) != 0) {
    	    os.bottom = inCoord(s, os.bottom, delta);
    	}
    	
    	if((present & 0x20) != 0) {
    	    os.action = s.inUInt8();
    	}
    	
    	//debug("RDP Orders DeskSave: left = " + os.left + ", top = " + os.top + ", right = " + os.right + ", bottom = " + os.bottom + ", offset = " + os.offset + ", action = " + os.action);
    	
    	width = os.right - os.left + 1;
    	height = os.bottom - os.top + 1;
    	
    	if(os.action == 0) {
    	    rdr.desktopSave(os.offset, os.left, os.top, width, height);
    	}
    	else {
    	    rdr.desktopRestore(os.offset, os.left, os.top, width, height);
    	}
    }
    
    // Process a memory blt order
    private void processMemBlt(RDPStream s, MemBltOrder os, int present, boolean delta) {
    	if((present & 0x01) != 0) {
    	    os.cacheID = s.inUInt8();
    	    os.colorTable = s.inUInt8();
    	}
    	
    	if((present & 0x02) != 0) {
    	    os.x = inCoord(s, os.x, delta);
    	}
    	if((present & 0x04) != 0) {
    	    os.y = inCoord(s, os.y, delta);
    	}
    	if((present & 0x08) != 0) {
    	    os.cx = inCoord(s, os.cx, delta);
    	}
    	if((present & 0x10) != 0) {
    	    os.cy = inCoord(s, os.cy, delta);
    	}
    	
    	if((present & 0x20) != 0) {
    	    os.opcode = s.inUInt8();
    	}
    	
    	if((present & 0x40) != 0) {
    	    os.srcx = inCoord(s, os.srcx, delta);
    	}
    	if((present & 0x80) != 0) {
    	    os.srcy = inCoord(s, os.srcy, delta);
    	}
    	
    	if((present & 0x100) != 0) {
    	    os.cacheIndex = s.inUInt16LE();
    	}
    	
    	//debug("RDP Orders MemBlt: opcode = 0x" + Integer.toHexString(os.opcode & 0xff) + ", x = " + os.x + ", y = " + os.y + ", cx = " + os.cx + ", cy = " + os.cy + ", cacheID = " + os.cacheID + ", cacheIndex = " + os.cacheIndex + ", srcx = " + os.srcx + ", srcy = " + os.srcy);
    	    	
    	RDPBitmap bitmap = cache.getBitmap(os.cacheID, os.cacheIndex);
    	
    	if(bitmap == null) {
    	    return;
    	}
    	
    	rdr.memBlt(rop2S(os.opcode), os.x, os.y, os.cx, os.cy, bitmap, os.srcx, os.srcy);
    }
    
    private void processTriBlt(RDPStream s, TriBltOrder os, int present, boolean delta) {
    	if((present & 0x01) != 0) {
    	    os.cacheID = s.inUInt8();
    	    os.colorTable = s.inUInt8();
    	}
    	
    	if((present & 0x02) != 0) {
    	    os.x = inCoord(s, os.x, delta);
    	}
    	if((present & 0x04) != 0) {
    	    os.y = inCoord(s, os.y, delta);
    	}
    	if((present & 0x08) != 0) {
    	    os.cx = inCoord(s, os.cx, delta);
    	}
    	if((present & 0x10) != 0) {
    	    os.cy = inCoord(s, os.cy, delta);
    	}
    	
    	if((present & 0x20) != 0) {
    	    os.opcode = s.inUInt8();
    	}
    	
    	if((present & 0x40) != 0) {
    	    os.srcx = inCoord(s, os.srcx, delta);
    	}
    	if((present & 0x80) != 0) {
    	    os.srcy = inCoord(s, os.srcy, delta);
    	}
    	
    	if((present & 0x100) != 0) {
    	    os.bgColor = inColor(s);
    	}
    	if((present & 0x200) != 0) {
    	    os.fgColor = inColor(s);
    	}
    	
    	parseBrush(s, os.brush, present >> 10);
    	
    	if((present & 0x8000) != 0) {
    	    os.cacheIndex = s.inUInt16LE();
    	}
    	if((present & 0x10000) != 0) {
    	    os.unknown = s.inUInt16LE();
    	}
    	
    	//debug("RDP Orders TriBlt: Opcode = 0x" + Integer.toHexString(os.opcode & 0xff) + ", x = " + os.x + ", y = " + os.y + ", cx = " + os.cx + ", cy = " + os.cy + ", cacheID = " + os.cacheID + ", cacheIndex = " + os.cacheIndex + ", srcx = " + os.srcx + ", srcy = " + os.srcy + ", brush.style = " + os.brush.getStyle() + ", bgColor = " + os.bgColor + ", fgColor = " + os.fgColor);
    	
    	RDPBitmap bitmap = cache.getBitmap(os.cacheID, os.cacheIndex);
    	if(bitmap == null) {
    	    return;
    	}
    	
    	rdr.triBlt(os.opcode, os.x, os.y, os.cx, os.cy, bitmap, os.srcx, os.srcy, os.brush, os.bgColor, os.fgColor);
    }
    
    // Parse a delta co-ordinate in polyline order form
    // we need two return parameters here, so we have to declare the return type as int[]
    private int[] parseDelta(byte buffer[], int offset) {
    	int value = buffer[offset++] & 0xff;
    	boolean twoByte = (value & 0x80) != 0;
    	
    	if((value & 0x40) != 0) {	// sign bit
    	    value |= ~0x3f;
    	}
    	else {
    	    value &= 0x3f;
    	}
    	
    	if(twoByte) {
    	    value = (value << 8) | (buffer[offset++] & 0xff);
    	}
    	
    	int ret[] = new int[2];
    	ret[0] = value;
    	ret[1] = offset;
    	
    	return ret;
    }
    
    // Process a polyline order
    private void processPolyLine(RDPStream s, PolyLineOrder os, int present, boolean delta) {    
    	int index;
    		
    	if((present & 0x01) != 0) {
    	    os.x = inCoord(s, os.x, delta);
    	}
    	if((present & 0x02) != 0) {
    	    os.y = inCoord(s, os.y, delta);
    	}
    	if((present & 0x04) != 0) {
    	    os.opcode = s.inUInt8();
    	}
    	if((present & 0x10) != 0) {
    	    os.fgColor = inColor(s);
    	}
    	if((present & 0x20) != 0) {
    	    os.lines = s.inUInt8();
    	}
    	if((present & 0x40) != 0) {
    	    os.dataSize = s.inUInt8();
    	    s.inUInt8Array(os.data, 0, os.dataSize);
    	}
    	
    	//debug("RDP Orders PolyLine: opcode = 0x" + Integer.toHexString(os.opcode & 0xff) + ", x = " + os.x + ", y = " + os.y + ", fgColor = " + os.fgColor + ", lines = " + os.lines + ", dataSize = " + os.dataSize);
    	if(debug) {
    	    System.out.print("Data: ");
    	    for(index = 0; index < os.dataSize; index++) {
    	    	System.out.print("0x" + Integer.toHexString(os.data[index] & 0xff) + " ");
    	    }
    	    System.out.println("");
    	}
    	
    	if(os.opcode < 0x01 || os.opcode > 0x10) {
    	    System.out.println("RDP Orders processPolyLine: bad rop2: 0x" + Integer.toHexString(os.opcode & 0xff));
    	    return;
    	}
    	
    	int opcode = os.opcode - 1;
    	int x = os.x;
    	int y = os.y;

    	RDPPen pen = new RDPPen();
    	pen.setStyle(0);
    	pen.setWidth(0);
    	pen.setColor(os.fgColor);
    	
    	int line, data;
    	int xfrom, yfrom;
    	int flags = 0;
    	
    	index = 0;
    	data = ((os.lines - 1) / 4) + 1;
    	
    	for(line = 0; (line < os.lines) && (data < os.dataSize); line++) {
    	    xfrom = x;
    	    yfrom = y;
    	    
    	    if(line % 4 == 0) {
    	    	flags = os.data[index++];
    	    }
    	    
    	    if ((flags & 0xc0) == 0) {
		flags |= 0xc0;	/* none = both */
	    }
	    
	    if((flags & 0x40) != 0) {
	    	int p[] = parseDelta(os.data, data);
	    	x += p[0];
	    	data = p[1];
	    }
	    
	    if((flags & 0x80) != 0) {
	    	int p[] = parseDelta(os.data, data);
	    	y += p[0];
	    	data = p[1];
	    }
	    
	    rdr.drawLine(opcode, xfrom, yfrom, x, y, pen);
	    
	    flags <<= 2;
    	}
    }
    
    // Process a text order
    private void processText2(RDPStream s, Text2Order os, int present, boolean delta) {
    	if((present & 0x01) != 0) {
    	    os.font = s.inUInt8();
    	}
    	if((present & 0x02) != 0) {
    	    os.flags = s.inUInt8();
    	}
    	if((present & 0x04) != 0) {
    	    os.unknown = s.inUInt8();
    	}
    	if((present & 0x08) != 0) {
    	    os.mixMode = s.inUInt8();
    	}
    	if((present & 0x10) != 0) {
    	    os.fgColor = inColor(s);
    	}
    	if((present & 0x20) != 0) {
    	    os.bgColor = inColor(s);
    	}
    	if((present & 0x40) != 0) {
    	    os.clipLeft = s.inSInt16LE();
    	}
    	if((present & 0x80) != 0) {
    	    os.clipTop = s.inSInt16LE();
    	}
    	if((present & 0x100) != 0) {
    	    os.clipRight = s.inSInt16LE();
    	}
    	if((present & 0x200) != 0) {
    	    os.clipBottom = s.inSInt16LE();
    	}
    	if((present & 0x400) != 0) {
    	    os.boxLeft = s.inSInt16LE();
    	}
    	if((present & 0x800) != 0) {
    	    os.boxTop = s.inSInt16LE();
    	}
    	if((present & 0x1000) != 0) {
    	    os.boxRight = s.inSInt16LE();
    	}
    	if((present & 0x2000) != 0) {
    	    os.boxBottom = s.inSInt16LE();
    	}
    	if((present & 0x4000) != 0) {	// fix for connecting to a server that
    	    s.incrementPosition(10);	//     was disconnected with mstsc.exe
    	}
    	// 0x008000, 0x020000, and 0x040000 are present too ???
    	
    	if((present & 0x80000) != 0) {
    	    os.x = s.inSInt16LE();
    	}
    	if((present & 0x100000) != 0) {
    	    os.y = s.inSInt16LE();
    	}
    	if((present & 0x200000) != 0) {
    	    os.length = s.inUInt8();
    	    s.inUInt8Array(os.text, 0, os.length);
    	}
    	
    	//debug("RDP Orders Text2: x = " + os.x + ", y = " + os.y + ", clipLeft = " + os.clipLeft + ", clipTop = " + os.clipTop + ", clipRight = " + os.clipRight + ", clipBottom = " + os.clipBottom + ", boxLeft = " + os.boxLeft + ", boxTop = " + os.boxTop + ", boxRight = " + os.boxRight + ", boxBottom = " + os.boxBottom + ", font = " + os.font + ", flags = 0x" + Integer.toHexString(os.flags & 0xff) + ", fgColor = 0x" + Integer.toHexString(os.fgColor) + ", bgColor = 0x" + Integer.toHexString(os.bgColor) + ", mixMode = " + os.mixMode + ", unknown = " + os.unknown + ", length = " + os.length);
    	if(debug) {
    	    System.out.print("RDP Orders Text2: text = ");
    	    for(int i = 0; i < os.length; i++) {
    	    	System.out.print("0x" + Integer.toHexString(os.text[i] & 0xff) + " ");
    	    }
    	    System.out.println("");
    	}
    	
    	rdr.drawText(os.font, os.flags, os.mixMode, os.x, os.y,
		     os.clipLeft, os.clipTop,
		     os.clipRight - os.clipLeft,
		     os.clipBottom - os.clipTop,
		     os.boxLeft, os.boxTop,
		     os.boxRight - os.boxLeft,
		     os.boxBottom - os.boxTop,
		     os.bgColor, os.fgColor, os.text, os.length);
    }
    
    // Process a raw bitmap cache order
    private void processRawBmpCache(RDPStream s) {
    	int cacheID = s.inUInt8();
    	s.incrementPosition(1);		// pad
    	int width = s.inUInt8();
    	int height = s.inUInt8();
    	int bpp = s.inUInt8();
    	int Bpp = (bpp + 7) / 8;
    	int bufSize = s.inUInt16LE();
    	int cacheIndex = s.inUInt16LE();
    	byte data[] = getByteBuffer(bufSize * 2);
    	s.inUInt8Array(data, bufSize, bufSize);
    	
    	//debug("RDP Orders RawBmpCache: cx = " + width + ", cy = " + height + ", cacheID = " + cacheID + ", cacheIndex = " + cacheIndex);
    	for(int y = 0; y < height; y++) {
    	    System.arraycopy(data, bufSize + y * (width * Bpp), data, (height - y - 1) * (width * Bpp), width * Bpp);
    	}
    	int rgbData[] = cache.getBitmapInt(cacheID, cacheIndex, height * width);
    	rdr.translateColor(rgbData, 0, data, 0, height * width * Bpp);
    	
    	RDPBitmap bitmap = new RDPBitmap(rgbData, width, height, comp);
    	cache.putBitmap(cacheID, cacheIndex, bitmap);
    }
    
    // Process a bitmap cache order
    private void processBmpCache(RDPStream s) {
    	int cacheID = s.inUInt8();
    	int pad1 = s.inUInt8();
    	int width = s.inUInt8();
    	int height = s.inUInt8();
    	int bpp = s.inUInt8();
    	int Bpp = (bpp + 7) / 8;
    	int bufSize = s.inUInt16LE();
    	int cacheIndex = s.inUInt16LE();
    	
    	int pad2 = s.inUInt16LE();
    	int size = s.inUInt16LE();
    	int rowSize = s.inUInt16LE();
    	int finalSize = s.inUInt16LE();
    	
    	byte data[] = getByteBuffer(size);
    	s.inUInt8Array(data, 0, size);
    	
    	//debug("RDP Orders BmpCache: cx = " + width + ", cy = " + height + ", cacheID = " + cacheID + ", cacheIndex = " + cacheIndex + ", bpp = " + bpp + ", size = " + size + ", pad1 = " + pad1 + ", bufSize = " + bufSize + ", pad2 = " + pad2 + ", rowSize = " + rowSize + ", finalSize = " + finalSize);
    	
    	
    	int rgbData[] = bitmapDecompressor.decompress(data, width, height, size, Bpp);
    	
    	if(rgbData != null) {
    	    int rgbDataCache[] = cache.getBitmapInt(cacheID, cacheIndex, height * width);
    	    System.arraycopy(rgbData, 0, rgbDataCache, 0, height * width);
    	    RDPBitmap bmp = new RDPBitmap(rgbDataCache, width, height, comp);
    	    cache.putBitmap(cacheID, cacheIndex, bmp);
    	}
    	else {
    	    System.out.println("RDP Orders: Failed to decompress bitmap data");
    	}
    }
    
    // Process a colourmap cache order
    private void processColCache(RDPStream s) {
    	int cacheID = s.inUInt8();
    	int nColors = s.inUInt16LE();
    	int colors[] = new int[nColors];
    	
    	int r, g, b;
    	
    	for(int i = 0; i < nColors; i++) {
    	    b = s.inUInt8();
    	    g = s.inUInt8();
    	    r = s.inUInt8();
    	    s.incrementPosition(1);	// pad
    	    colors[i] = 0xff000000 | (r << 16) | (g << 8) | b;
    	}
    	
    	//debug("RDP Orders ColCache: CacheID = " + cacheID + ", nColors = " + nColors);
    	
    	rdr.setColorMap(colors, nColors);
    }
    
    // Process a font cache order
    private void processFontCache(RDPStream s) {
    	int font = s.inUInt8();
    	int nGlyphs = s.inUInt8();
    	
    	//debug("RDP Orders FontCache: font = " + font + ", nGlyphs = " + nGlyphs);
    	
    	for(int i = 0; i < nGlyphs; i++) {
    	    int character = s.inUInt16LE();
    	    int offset = s.inSInt16LE();
    	    int baseLine = s.inSInt16LE();
    	    int width = s.inUInt16LE();
    	    int height = s.inUInt16LE();
    	    
    	    int dataSize = (height * ((width + 7) / 8) + 3) & ~3;
    	    
    	    byte data[] = new byte[dataSize];
    	    s.inUInt8Array(data, 0, dataSize);
    	    
    	    RDPFontGlyph glyph = new RDPFontGlyph(offset, baseLine, width, height, data);
    	    cache.putFont(font, character, glyph);
    	}
    }
    
    // Process a secondary order
    private void processSecondaryOrder(RDPStream s) {
    	int len = s.inUInt16LE();
    	s.incrementPosition(2);		// pad
    	int type = s.inUInt8();
    	
    	int nextOrder = s.getPosition() + len + 7;
    	
    	switch(type) {
    	    case RDPConstants.rdpOrderRawBmpCache:	// 0
    	    	processRawBmpCache(s);
    	    	break;
    	    case RDPConstants.rdpOrderColCache:		// 1
    	    	processColCache(s);
    	    	break;
    	    case RDPConstants.rdpOrderBmpCache:		// 2
    	    	processBmpCache(s);
    	    	break;
    	    case RDPConstants.rdpOrderFontCache:	// 3
    	    	processFontCache(s);
    	    	break;
    	    default:
    	    	System.out.println("RDP Orders processSecondaryOrder: unknown order received: 0x" + Integer.toHexString(type & 0xff));
    	}
    	
    	s.setPosition(nextOrder);
    }
    
    // Process an order PDU
    public void processOrders(RDPStream s, int numOrders, int nextPacket) {
    	//debug("RDP Orders processOrders: numOrders = " + numOrders);
    	
    	int processed = 0;
    	int present, orderFlags, size;
    	boolean delta;
    	
    	//System.out.println("num orders: " + numOrders);
    	
    	while(processed < numOrders) {
    	    orderFlags = s.inUInt8();
    	    
    	    if((orderFlags & RDPConstants.rdpOrderStandard) == 0) {
    	    	System.out.println("RDP Orders processOrders: parsing orders failed");
    	    	break;
    	    }
    	    
    	    if((orderFlags & RDPConstants.rdpOrderSecondary) != 0) {
    	    	processSecondaryOrder(s);
    	    }
    	    else {
    	    	if((orderFlags & RDPConstants.rdpOrderChange) != 0) {
    	    	    orderState.orderType = s.inUInt8();
    	    	}
    	    	
    	    	switch(orderState.orderType) {
    	    	    case RDPConstants.rdpOrderTriBlt:	// 14
    	    	    case RDPConstants.rdpOrderText2:	// 27
    	    	    	size = 3;
    	    	    	break;
    	    	    case RDPConstants.rdpOrderPatBlt:	// 1
    	    	    case RDPConstants.rdpOrderMemBlt:	// 13
    	    	    case RDPConstants.rdpOrderLine:	// 9
    	    	    	size = 2;
    	    	    	break;
    	    	    default:
    	    	    	size = 1;
    		}
    		
    		present = inPresent(s, orderFlags, size);
    		
    		if((orderFlags & RDPConstants.rdpOrderBounds) != 0) {
    		    if((orderFlags & RDPConstants.rdpOrderLastBounds) == 0) {
    		    	parseBounds(s, orderState.bounds);
    		    }
    		    //if(orderState.orderType != RDPConstants.rdpOrderScreenBlt)
    		    rdr.setClip(orderState.bounds.getLeft(),
    		                orderState.bounds.getTop(),
    		                orderState.bounds.getRight() - orderState.bounds.getLeft() + 1,
    		                orderState.bounds.getBottom() - orderState.bounds.getTop() + 1);
    		}
    		
    		delta = (orderFlags & RDPConstants.rdpOrderDelta) != 0;
    		
    		switch(orderState.orderType) {
    		    case RDPConstants.rdpOrderDestBlt:		// 0
    		    	processDestBlt(s, orderState.destBlt, present, delta);
    		    	break;
    		    case RDPConstants.rdpOrderPatBlt:		// 1
    		    	processPatBlt(s, orderState.patBlt, present, delta);
    		    	break;
    		    case RDPConstants.rdpOrderScreenBlt:	// 2
    		    	processScreenBlt(s, orderState.screenBlt, present, delta);
    		    	break;
    		    case RDPConstants.rdpOrderLine:		// 9
    		    	processLine(s, orderState.line, present, delta);
    		    	break;
    		    case RDPConstants.rdpOrderRect:		// 10
    		    	processRect(s, orderState.rect, present, delta);
    		    	break;
    		    case RDPConstants.rdpOrderDeskSave:		// 11
    		    	processDeskSave(s, orderState.deskSave, present, delta);
    		    	break;
    		    case RDPConstants.rdpOrderMemBlt:		// 13
    		    	processMemBlt(s, orderState.memBlt, present, delta);
    		    	break;
    		    case RDPConstants.rdpOrderTriBlt:		// 14
    		    	processTriBlt(s, orderState.triBlt, present, delta);
    		    	break;
    		    case RDPConstants.rdpOrderPolyLine:		// 22
    		    	processPolyLine(s, orderState.polyLine, present, delta);
    		    	break;
    		    case RDPConstants.rdpOrderText2:		// 27
    		    	processText2(s, orderState.text2, present, delta);
    		    	break;
    		    default:
    		    	System.out.println("RDP Orders processOrders: unknown PDU type received: 0x" + Integer.toHexString(orderState.orderType & 0xff));
    		}
    		
    		if((orderFlags & RDPConstants.rdpOrderBounds) != 0) {
    		    rdr.resetClip();
    		}
    	    }
    	    
    	    processed++;
    	}
    }
    
    // public functions
    public void resetOrderState() {
    	orderState.reset();
    }
    
    
}
