package nn.pp.rcrdp;

import java.awt.*;
import java.awt.image.*;
import java.io.*;
import nn.pp.rc.RCRenderer;

public abstract class RDPRenderer implements RCRenderer {
    protected boolean debug = false;
    
    protected boolean clipSet = false;
    protected int clipLeft = 0;
    protected int clipTop = 0;
    protected int clipWidth = 0;
    protected int clipHeight = 0;
    
    private int repaintLeft;
    private int repaintTop;
    private int repaintWidth;
    private int repaintHeight;
    private boolean repaintNeeded;
    private boolean repaintAreaSet = false;

    protected Component comp;
    protected Dimension mysize;
    protected Cache cache;
    protected Image vimg;
    
    protected RDPProto rdp;
    protected RDPProfile profile;
    
    protected ColorTranslator colorTranslator;
    protected Blitter blitter;
    
    // graphics context
    protected Graphics gvimg;
    
    // image source for bitmap updates
    protected int[] sImgMem;
    MemoryImageSource sImgSrc;
    Image sImg;
    
    // internal buffer
    private int buffer[] = null;
    
    // abstract function
    abstract public void setInterpol(boolean ip);
    
    abstract public void paint(Graphics g, boolean scale, Dimension scalesize);
    
    abstract public void repaint();
    
    abstract public void repaint(int x, int y, int w, int h);

    // debugging
    protected void debug(String s) {
    	if(debug) System.out.println(s);
    }
    
    // Constructor
    public RDPRenderer(Component comp) {
    	this.comp = comp;
    	colorTranslator = new ColorTranslator();
    	blitter = new Blitter();
    }
    
    public ColorTranslator getColorTranslator() {
    	return colorTranslator;
    }
    
    public void setProfile(RDPProfile profile) {
    	this.profile = profile;
    	colorTranslator.setProfile(profile);
    }
    
    public void setCache(Cache cache) {
    	this.cache = cache;
    }
    
    // get and allocate (if necessary) the internal buffer
    private int[] getBuffer(int size) {
    	if(buffer == null) {
    	    buffer = new int[size];
    	}
    	else if(buffer.length < size) {
    	    buffer = new int[size];
    	}
    	
    	return buffer;
    }
    
    // create the image for bitmap updates
    protected void createImages(Dimension d) {
	if(null == sImgMem) {
	    sImgMem = new int[mysize.width * mysize.height];
	}
    }
    
    // set the size
    public void setRenderSize(Dimension d) {
    	mysize = d;
    	createImages(d);
    	blitter.setImageArray(sImgMem);
    	blitter.setRenderSize(d);
    	resetRepaintArea();
    	sImgSrc = new MemoryImageSource(mysize.width, mysize.height, sImgMem, 0, mysize.width);
    	sImgSrc.setAnimated(true);
    	sImgSrc.setFullBufferUpdates(false);
    	sImg = comp.createImage(sImgSrc);
    }
    
    // set the RDPProto
    public void setRDPProto(RDPProto proto) {
    	rdp = proto;
    }
    
    // set a colormap
    public void setColorMap(int colors[], int nColors) {
    	colorTranslator.setColorMap(colors, nColors);
    	
    	//debug("color palette update received");
    }
    
    // color translation functions
    public int translateColor(int col) {
    	return colorTranslator.translateColor(col);
    }
    
    public int[] translateColor(byte colIn[], int size) {
    	return colorTranslator.translateColor(colIn, size);
    }
    
    public void translateColor(int colOut[], int indexOut, byte colIn[], int indexIn, int len) {
    	colorTranslator.translateColor(colOut, indexOut, colIn, indexIn, len);
    }
    
    // set clipping coordinates
    public void setClip(int left, int top, int width, int height) {
    	gvimg.setClip(left, top, width, height);
    	clipLeft = left;
    	clipTop = top;
    	clipWidth = width;
    	clipHeight = height;
    	clipSet = true;
    }
    
    public void resetClip() {
    	gvimg.setClip(0, 0, mysize.width, mysize.height);
    	clipLeft = 0;
    	clipTop = 0;
    	clipWidth = mysize.width;
    	clipHeight = mysize.height;
    	clipSet = false;
    }
    
    // painting help methods
    
    // sets a new region for repainting
    private void setRepaintArea(int left, int top, int width, int height) {
    	int x0, x1, y0, y1;
    	//System.out.println("before: l = " + repaintLeft + ", w = " + repaintWidth + ", r = " + (repaintLeft + repaintWidth) + ", t = " + repaintTop + ", h = " + repaintHeight + ", b = " + (repaintTop + repaintHeight));
    	//System.out.println("in:     l = " + left + ", w = " + width + ", r = " + (left + width) + ", t = " + top + ", h = " + height + ", b = " + (top + height));
    	if(!repaintAreaSet) {
    	    x0 = left;
    	    x1 = left + width;
    	    y0 = top;
    	    y1 = top + height;
    	}
    	else {
    	    x0 = Math.min(repaintLeft, left);
    	    x1 = Math.max(repaintLeft + repaintWidth, left + width);
    	    y0 = Math.min(repaintTop, top);
    	    y1 = Math.max(repaintTop + repaintHeight, top + height);
    	}
    	repaintAreaSet = true;
    	repaintNeeded = true;
    	repaintLeft = x0;
    	repaintWidth = x1 - x0;
    	repaintTop = y0;
    	repaintHeight = y1 - y0;
    	//System.out.println("after:  l = " + repaintLeft + ", w = " + repaintWidth + ", r = " + (repaintLeft + repaintWidth) + ", t = " + repaintTop + ", h = " + repaintHeight + ", b = " + (repaintTop + repaintHeight));
    	//System.out.println();
    }

    // resets regions for repainting
    private void resetRepaintArea() {
    	repaintNeeded = false;
    	repaintAreaSet = false;
    	repaintLeft = mysize.width;
    	repaintTop = mysize.height;
    	repaintWidth = -repaintLeft;
    	repaintHeight = -repaintTop;
    }
    
    // do actually repaint
    public void doRepaint() {
    	if(repaintNeeded) {
   	    repaint(repaintLeft, repaintTop, repaintWidth, repaintHeight);
   	    //repaint(0, 0, mysize.width, mysize.height);
    	}
    	resetRepaintArea();
    }
    
    // update the screen using the internal MemoryImage
    private void updateImage(int left, int top, int width, int height, boolean drawImage) {
    	//System.out.println("in: l = " + left + ", w = " + width + ", r = " + (left + width) + ", t = " + top + ", h = " + height + ", b = " + (top + height));
    	if(clipSet) {
    	    left = clipLeft;
    	    top = clipTop;
    	    width = clipWidth;
    	    height = clipHeight;
    	}
    	else {
    	    gvimg.setClip(left, top, width, height);
    	}
    	//System.out.println("clipped: l = " + left + ", w = " + width + ", r = " + (left + width) + ", t = " + top + ", h = " + height + ", b = " + (top + height));
    	//System.out.println();
    	
    	// shall we draw our internal int array onto the screen?
    	// otherwise we only do a repaint (e.g. when a rect has been drawn)
    	if(drawImage) {
    	    sImgSrc.newPixels(left, top, width, height);
    	    gvimg.drawImage(sImg, left, top, left + width, top + height, left, top, left + width, top + height, null);
    	}
    	
    	setRepaintArea(left, top, width, height);
    	
    	if(!clipSet) {
    	    gvimg.setClip(0, 0, mysize.width, mysize.height);
    	}
    }
    
    // this method is called by the J11/J14 renderers
    // to set a correct clipping before painting the internal image onto the screen
    // when scaling is set (because otherwise some regions would not be updated)
    protected void setClippingForScaling(Graphics g, Dimension scalesize) {
	int xd1, xd2, yd1, yd2, xs1, xs2, ys1, ys2;
	Rectangle r = g.getClipBounds();
	if(r == null) {
	    xd1 = xs1 = yd1 = ys1 = 0;
	    xd2 = xs2 = mysize.width;
	    yd2 = ys2 = mysize.height;
	}
	else {
	    xd1 = r.x;
	    xd2 = xd1 + r.width;
	    yd1 = r.y;
	    yd2 = yd1 + r.height;
	    xs1 = xd1; 
	    xs2 = xd2; 
	    ys1 = yd1; 
	    ys2 = yd2;
	}
	
	// update always one more pixel to avoid rests of former updates
	if(xs1 > 0) xs1--;
	if(xs2 < mysize.width) xs2++;
	if(ys1 > 0) ys1--;
	if(ys2 < mysize.width) ys2++;
	
	xd1 = xs1 * scalesize.width / mysize.width;
	xd2 = xs2 * scalesize.width / mysize.width;
	yd1 = ys1 * scalesize.height / mysize.height;
	yd2 = ys2 * scalesize.height / mysize.height;
	
	g.setClip(new Rectangle(xd1, yd1, xd2 - xd1, yd2 - yd1));
    }
    
    // blitting methods
    
    // a generic blit method
    public void blit(int src[], int opcode, int x, int y, int cx, int cy, int srcX, int srcY, int srcCx, int srcCy, int srcScan, int bgColor) {
    	if(clipSet) {
    	    int x0 = Math.max(x, clipLeft);
    	    int y0 = Math.max(y, clipTop);
    	    int x1U = x + cx;
    	    int x1C = clipLeft + clipWidth;
    	    int y1U = y + cy;
    	    int y1C = clipTop + clipHeight;
    	    int x1 = Math.min(Math.min(x1U, x1C), mysize.width);
    	    int y1 = Math.min(Math.min(y1U, y1C), mysize.height);
    	    srcX += (x0 - x);
    	    srcY += (y0 - y);
    	    x = x0;
    	    y = y0;
    	    cx = x1 - x0;
    	    cy = y1 - y0;
    	    if(cx < 0 || cy < 0) {
    	    	return;
    	    }
    	}
    	
    	blitter.blit(src, opcode, x, y, cx, cy, srcX, srcY, srcCx, srcCy, srcScan, bgColor);
    }
    
    // a generic blit method
    // this one translates the colors before the blit
    public void blitTranslate(byte src[], int opcode, int x, int y, int cx, int cy, int srcX, int srcY, int srcCx, int srcCy, int srcScan, int bgColor) {
    	int srcInt[] = colorTranslator.translateColor(src, srcCx * srcCy);
    	
    	blit(srcInt, opcode, x, y, cx, cy, srcX, srcY, srcCx, srcCy, srcScan, bgColor);
    	
    	srcInt = null;	// garbage collector should collect it
    }
    
    // painting / order processing methods
    
    // draw a rectangle
    public void drawRect(int x, int y, int cx, int cy, int col) {
    	
    	int c = translateColor(col);
    	
    	if(clipSet) {
    	    int x0 = Math.max(x, clipLeft);
    	    int y0 = Math.max(y, clipTop);
    	    int x1U = x + cx;
    	    int x1C = clipLeft + clipWidth;
    	    int y1U = y + cy;
    	    int y1C = clipTop + clipHeight;
    	    int x1 = Math.min(Math.min(x1U, x1C), mysize.width);
    	    int y1 = Math.min(Math.min(y1U, y1C), mysize.height);
    	    x = x0;
    	    y = y0;
    	    cx = x1 - x0;
    	    cy = y1 - y0;
    	}
    	//System.out.println("drawRect: x = " + x + ", y = " + y + ", cx = " + cx + ", cy = " + cy + ", clipLeft = " + clipLeft + ", clipTop = " + clipTop + ", clipWitdh = " + clipWidth + ", clipHeight = " + clipHeight);

    	// we have to set the pixels in our MemoryImage also
    	// because we need to know a background for transparent text
    	// or some blittings

    	// we don't use our blitting functions here because it is quite
    	// faster to "draw" directly into the image source
    	// (less array copies necessary)
    	int imgMem[] = sImgMem;
    	int w = mysize.width;
    	
    	int m[] = new int[cx];
    	int i;
    	for(i = 0; i < cx; i++) {
    	    m[i] = c;
    	}
    	int index = y * w + x;
    	for(i = 0; i < cy; i++) {
    	    System.arraycopy(m, 0, imgMem, index, cx);
    	    index += w;
    	}

    	// but we draw directly into our Screen graphics because it is faster in this case
    	
    	if(!clipSet) {
    	    gvimg.setClip(x, y, cx, cy);
    	}
    	gvimg.setColor(new Color(c));
    	gvimg.fillRect(x, y, cx, cy);
    	if(!clipSet) {
    	    resetClip();
    	}
    	
    	//System.out.println("drawRect: l = " + x + ", w = " + cx + ", r = " + (x + cx) + ", t = " + y + ", h = " + cy + ", b = " + (y + cy));
    	updateImage(x, y, cx, cy, false);
    }
    
    // draw a line
    public void drawLine(int opcode, int x1, int y1, int x2, int y2, RDPPen pen) {
    	// FIXME: what about opcode?
    	int col = translateColor(pen.getColor());
    	int i, x, y;
    	int xMin = Math.min(x1, x2);
    	int yMin = Math.min(y1, y2);
    	
    	int imgMem[] = sImgMem;
    	int w = mysize.width;
    	
    	// we can easily draw a line onto the surface drawline
    	
	// but we have the problem that we also have to set the
	// appropriate pixels in our memory image array;
    	// that's why we choose the hard way here
    	int dx = Math.abs(x1 - x2) + 1;
    	int dy = Math.abs(y1 - y2) + 1;
    	
    	if(dx < dy) {
    	    if(y1 > y2) {
    	    	i = x1; x1 = x2; x2 = i; i = y1; y1 = y2; y2 = i;
    	    }
    	    for(y = y1; y < y2; y++) {
    	    	x = x1 + (y - y1) * (x2 - x1) / (y2 - y1);
    	    	if(clipSet) {
    	    	    if(x < clipLeft || x >= clipLeft + clipWidth || y < clipTop) {
    	    	    	continue;
    	    	    }
    	    	    if(y >= clipTop + clipHeight) {
    	    	    	break;
    	    	    }
    	    	}
    	    	imgMem[x + y * w] = col;
    	    }
    	}
    	else {
    	    if(x1 > x2) {
    	    	i = x1; x1 = x2; x2 = i; i = y1; y1 = y2; y2 = i;
    	    }
    	    else if(x1 == x2) {
    	    	imgMem[x1 + y1 * w] = col;
    	    }
    	    for(x = x1; x < x2; x++) {
    	    	y = y1 + (x - x1) * (y2 - y1) / (x2 - x1);
    	    	if(clipSet) {
    	    	    if(x < clipLeft || y < clipTop || y >= clipTop + clipHeight) {
    	    	    	continue;
    	    	    }
    	    	    if(x >= clipLeft + clipWidth) {
    	    	    	break;
    	    	    }
    	    	}
    	    	imgMem[x + y * w] = col;
    	    }
    	}
    	
    	gvimg.setColor(new Color(col));
    	
    	if(!clipSet) {
    	    gvimg.setClip(xMin, yMin, dx, dy);
    	}
    	gvimg.drawLine(x1, y1, x2, y2);
    	if(!clipSet) {
    	    resetClip();
    	}
    	
    	updateImage(xMin, yMin, dx, dy, false);
    }
    
    // draw a bitmap
    public void drawBitmap(int left, int top, int cx, int cy, int width, int height, int bmpData[]) {
    	blit(bmpData, RDPConstants.rdpBlitCopy, left, top, cx, cy, 0, 0, width, height, width, 0);
	updateImage(left, top, cx, cy, true);
    }
    
    // draw one glyph
    private void drawGlyph(int mixMode, int x, int y, int clipX, int clipY, int clipCx, int clipCy, RDPFontGlyph glyph, int bgColor, int fgColor) {
    	//debug("RDP RDPRenderer drawGlyph: mixMode = " + mixMode + ", x = " + x + ", y = " + y + ", bgColor = 0x" + Integer.toHexString(bgColor) + ", fgColor = 0x" + Integer.toHexString(fgColor));
    	x += glyph.getOffset();
    	y += glyph.getBaseLine();
    	int cx = glyph.getWidth();
    	int cy = glyph.getHeight();
    	byte data[] = glyph.getPixMap();
    	
    	int xMax = clipX + clipCx;
    	int yMax = clipY + clipCy;
    	
    	int imgMem[] = sImgMem;
    	int w = mysize.width;
    	
    	// yes, sometimes same fgcolour and bgcolour are sent, but because
	// of transparency, we have to change that!
	if (mixMode == RDPConstants.mixTransparent && fgColor == bgColor) {
	    bgColor = fgColor ^ 0xff;
	}
	
	int index1, index2;
	
	switch(mixMode) {
	    case RDPConstants.mixTransparent:	// 0
	    	// transparent painting
	    	// we have to re-use the old values here,
	    	// copy only the new color values into our buffer
    		for(int yy = 0; yy < cy; yy++) {
    		    if(y + yy < clipY) {
    		    	continue;
    		    }
    		    if(y + yy >= yMax) {
    		    	break;
    		    }
    		    index1 = yy * cx;
    		    index2 = (y + yy) * w + x;
    		    for(int xx = 0; xx < cx; xx++) {
    		    	if(data[index1++] != 0) {
    		    	    if(x + xx < clipX) {
    		    	    	index2++;
    		    	    	continue;
    		    	    }
    		    	    if(x + xx >= xMax) {
    		    	    	break;
    		    	    }
    		    	    imgMem[index2] = fgColor;
    		    	}
    		    	index2++;
    		    }
    		}
	    	break;
	    case RDPConstants.mixOpaque:	// 1
	    	// we just have to set bgColor and fgColor here
    		for(int yy = 0; yy < cy; yy++) {
    		    if(y + yy < clipY) {
    		    	continue;
    		    }
    		    if(y + yy >= yMax) {
    		    	break;
    		    }
    		    index1 = yy * cx;
    		    index2 = (y + yy) * w + x;
    		    for(int xx = 0; xx < cx; xx++) {
    		    	if(x + xx < clipX) {
    		    	    index1++;
    		    	    index2++;
    		    	    continue;
    		    	}
    		    	if(x + xx >= xMax) {
    		    	    break;
    		    	}
    		    	imgMem[index2++] = (data[index1++] == 0) ? bgColor : fgColor;
    		    }
    		}
	    	break;
	    default:
	    	return;
	}
    }
    
    // draw text
    public void drawText(int font, int flags, int mixMode, int x, int y,
	     int clipX, int clipY, int clipCx, int clipCy,
	     int boxX, int boxY, int boxCx, int boxCy, int bgColor,
	     int fgColor, byte[] text, int length) {
    	
    	fgColor = translateColor(fgColor);
    	bgColor = translateColor(bgColor);

    	int imgMem[] = sImgMem;
    	int w = mysize.width;
    	
    	int index;
    	
    	//debug("RDP RDPRenderer drawText: font = " + font + ", mixMode = " + mixMode + ", x = " + x + ", y = " + y + ", clipX = " + clipX + ", clipY = " + clipY + ", clipCx = " + clipCx + ", clipCy = " + clipCy);
    	RDPFontGlyph glyph;
    	int i, j, xyOffset;
    	byte text2[];
    	
    	if(clipSet) {
    	    int x0 = Math.max(clipX, clipLeft);
    	    int y0 = Math.max(clipY, clipTop);
    	    int x1U = clipX + clipCx;
    	    int x1C = clipLeft + clipWidth;
    	    int y1U = clipY + clipCy;
    	    int y1C = clipTop + clipHeight;
    	    int x1 = Math.min(Math.min(x1U, x1C), w);
    	    int y1 = Math.min(Math.min(y1U, y1C), mysize.height);
    	    clipX = x0;
    	    clipY = y0;
    	    clipCx = x1 - x0;
    	    clipCy = y1 - y0;
    	}
    	
    	// correct some clipping regions
    	if(clipSet) {
    	    int boxRight = boxX + boxCx;
    	    if(boxRight > clipLeft + clipWidth) {
    	    	boxRight = clipLeft + clipWidth;
    	    }
    	    int boxBottom = boxY + boxCy;
    	    if(boxBottom > clipTop + clipHeight) {
    	    	boxBottom = clipTop + clipHeight;
    	    }
    	    if(boxX < clipLeft) {
    	    	boxX = clipLeft;
    	    }
    	    if(boxY < clipTop) {
    	    	boxY = clipTop;
    	    }
    	    boxCx = boxRight - boxX;
    	    boxCy = boxBottom - boxY;
    	}
    	
    	// fill the background
    	if(boxCx > 1) {
    	    for(i = 0; i < boxCy; i++) {
    	    	index = (i + boxY) * w + boxX;
    	    	for(j = 0; j < boxCx; j++) {
    	    	    imgMem[index++] = bgColor;
    	    	}
    	    }
    	}
    	else if(mixMode == RDPConstants.mixOpaque) {
    	    for(i = 0; i < clipCy; i++) {
    	    	index = (i + boxY) * w + boxX;
    	    	for(j = 0; j < clipCx; j++) {
    	    	    imgMem[index++] = bgColor;
    	    	}
    	    }
    	}
    	
    	// Paint text, character by character
    	for(i = 0; i < length; ) {
    	    int c = text[i] & 0xff;
    	    switch(c) {
    	    	case 0xff:
    	    	    if(i + 2 < length) {
    	    	    	int cacheIndex = text[i + 1] & 0xff;
    	    	    	RDPDataBlob blob = new RDPDataBlob((int)(text[i + 2] & 0xff), text);
    	    	    	//debug("RDP RDPRenderer: Putting text into Cache, index = " + cacheIndex + ", size = " + blob.getSize());
    	    	    	cache.putText(cacheIndex, blob);
    	    	    }
    	    	    else {
    	    	    	System.out.println("This shouldn't be happening");
    	    	    	return;
    	    	    }
    	    	    //this will move pointer from start to first character after FF command
    	    	    length -= i + 3;
    	    	    text2 = new byte[length];
    	    	    System.arraycopy(text, i + 3, text2, 0, length);
    	    	    text = text2;
    	    	    i = 0;
    	    	    break;
    	    	case 0xfe:
    	    	    int entryIndex = text[i + 1] & 0xff;
    	    	    RDPDataBlob entry = cache.getText(entryIndex);
    	    	    if(entry != null) {
    	    	    	//debug("RDP RDPRenderer: Using Text from Cache, index = " + entryIndex + ", size = " + entry.getSize());
    	    	    	byte data[] = entry.getData();
    	    	    	if(data[1] == 0 && ((flags & RDPConstants.text2ImplicitX) == 0)) {
    	    	    	    if ((flags & RDPConstants.text2Vertical) != 0) {
    	    	    	    	y += (text[i + 2] & 0xff);
    	    	    	    }
    	    	    	    else {
    	    	    	    	x += (text[i + 2] & 0xff);
    	    	    	    }
    	    	    	}
    	    	    	for(j = 0; j < entry.getSize(); j++) {
    	    	    	    // DO_GLYPH(data, j);
    	    	    	    index = data[j] & 0xff;
    	    	    	    glyph = cache.getFont(font, index);
    	    	    	    if((flags & RDPConstants.text2ImplicitX) == 0) {
    	    	   		xyOffset = data[++j];
    	    	    		if((xyOffset & 0x80) != 0) {
    	    	    	    	    if((flags & RDPConstants.text2Vertical) != 0) {
    	    	    	    		y += (data[j + 1] & 0xff) | ((data[j + 2] & 0xff) << 8);
    	    	    	    	    }
    	    	    	    	    else {
    	    	    	    		x += (data[j + 1] & 0xff) | ((data[j + 2] & 0xff) << 8);
    	    	    	    	    }
    	    	    	    	    j += 2;
    	    	    		}
    	    	    		else {
    	    	    	    	    if((flags & RDPConstants.text2Vertical) != 0) {
    	    	    	    		y += xyOffset;
    	    	    	    	    }
    	    	    	    	    else {
    	    	    	    		x += xyOffset;
    	    	    	    	    }
    	    	    		}
    	    	    	    }
    	    	    	    if(glyph != null) {
    	    	    	    	//glyph.showGlyph();
    	    	    		drawGlyph(mixMode, x, y, clipX, clipY, clipCx, clipCy, glyph, bgColor, fgColor);
    	    	    		if((flags & RDPConstants.text2ImplicitX) != 0) {
    	    	    	    	    x += glyph.getWidth();
    	    	    		}
    	    	    	    }
    	    	    	    else {
    	    	    	    	//debug("RDP RDPRenderer: unable to get Font Glyph: font = " + font + ", index = " + index);
    	    	    	    }
    	    	    	}
    	    	    }
    	    	    else {
    	    	    	debug("RDP RDPRenderer: Error reading Text from Cache");
    	    	    }
    	    	    if(i + 2 < length) {
    	    	    	i += 3;
    	    	    }
    	    	    else {
    	    	    	i += 2;
    	    	    }
    	    	    // this will move pointer from start to first character after FE command
    	    	    length -= i;
    	    	    text2 = new byte[length];
    	    	    System.arraycopy(text, i, text2, 0, length);
    	    	    text = text2;
    	    	    i = 0;
    	    	    break;
    	    	default:
    	    	    // DO_GLYPH(text, i);
    	    	    glyph = cache.getFont(font, text[i] & 0xff);
    	    	    if((flags & RDPConstants.text2ImplicitX) == 0) {
    	    	    	xyOffset = text[++i];
    	    	    	if((xyOffset & 0x80) != 0) {
    	    	    	    if((flags & RDPConstants.text2Vertical) != 0) {
    	    	    	    	y += (text[i + 1] & 0xff) | ((text[i + 2] & 0xff) << 8);
    	    	    	    }
    	    	    	    else {
    	    	    	    	x += (text[i + 1] & 0xff) | ((text[i + 2] & 0xff) << 8);
    	    	    	    }
    	    	    	    i += 2;
    	    	    	}
    	    	    	else {
    	    	    	    if((flags & RDPConstants.text2Vertical) != 0) {
    	    	    	    	y += xyOffset;
    	    	    	    }
    	    	    	    else {
    	    	    	    	x += xyOffset;
    	    	    	    }
    	    	    	}
    	    	    }
    	    	    if(glyph != null) {
    	    	    	drawGlyph(mixMode, x, y, clipX, clipY, clipCx, clipCy, glyph, bgColor, fgColor);
    	    	    	if((flags & RDPConstants.text2ImplicitX) != 0) {
    	    	    	    x += glyph.getWidth();
    	    	    	}
    	    	    }
    	    	    else {
    	    	    	debug("RDP RDPRenderer: unable to get Font Glyph");
    	    	    }
    	    	    i++;
    	    }
	}
	
	if(boxCx > 1) {
	    int x1 = Math.min(boxX, clipX);
	    int y1 = Math.min(boxY, clipY);
	    int x2 = Math.max(clipX + clipCx, boxX + boxCx);
	    int y2 = Math.max(clipY + clipCy, boxY + boxCy);
	    updateImage(x1, y1, x2 - x1 + 1, y2 - y1 + 1, true);
	}
	else {
	    updateImage(clipX, clipY, clipCx, clipCy, true);
	}
    }
    
    // process a screen blit (e.g. moving a window)
    public void screenBlt(int opcode, int x, int y, int cx, int cy, int srcX, int srcY) {
    	int imgMem[] = sImgMem;
    	int w = mysize.width;
    	
    	int dx = x - srcX;
    	int dy = y - srcY;
    	
    	// first read the bitmap data
    	// do we have to do it in two steps because of overlapping?
    	// only if dy == 0 (because of overlapping while arraycopy)
    	// or opcode != copy blit (because a blitting operation is necessary)
    	// or clipping is set (because we also had to care about the clipping
    	// here which makes it slower again)
    	if(clipSet || opcode != RDPConstants.rdpBlitCopy || dy == 0) {
    	    int window[] = getBuffer(cx * cy);
    	    for(int i = 0; i < cy; i++) {
    	    	System.arraycopy(imgMem, (srcY + i) * w + srcX, window, i * cx, cx);
    	    }
    	    // and blit it into the memory image
    	    blit(window, opcode, x, y, cx, cy, 0, 0, cx, cy, cx, 0);
    	}
    	else if(dy < 0) {
    	    // we move the picture up
    	    for(int i = 0; i < cy; i++) {
    	    	System.arraycopy(imgMem, (srcY + i) * w + srcX, imgMem, (y + i) * w + x, cx);
    	    }
    	}
    	else {
    	    // move the picture down
    	    for(int i = cy - 1; i >= 0; i--) {
    	    	System.arraycopy(imgMem, (srcY + i) * w + srcX, imgMem, (y + i) * w + x, cx);
    	    }
    	}
    	
    	// we use a faster drawing method here again
    	// but only if we use copy blit and no clipping is set
    	// because the clippings effect also the source rectangle
    	if(clipSet || opcode != RDPConstants.rdpBlitCopy/* || true /*FIXME*/) {
    	    updateImage(x, y, cx, cy, true);
    	}
    	else {
    	    gvimg.copyArea(srcX, srcY, cx, cy, dx, dy);
    	    updateImage(x, y, cx, cy, false);
    	}
    }
    
    // process a mem blit (e.g. painting bitmaps from the cache)
    public void memBlt(int opcode, int x, int y, int cx, int cy, RDPBitmap bitmap, int srcX, int srcY) {    	
    	
    	int srcCx = bitmap.getWidth();
    	int srcCy = bitmap.getHeight();

    	int data[] = bitmap.getPixels();
    	
    	// blit it into the memory image
    	blit(data, opcode, x, y, cx, cy, srcX, srcY, srcCx, srcCy, srcCx, 0);

    	if(opcode == RDPConstants.rdpBlitCopy) {
    	    // we use the stored image here
    	    // this is faster when an image is drawn more than once (which happens quite often)
    	    // because it has to be rendered only once, the next time we can rely on
    	    // prerendered data here which is stored internally in the image returned
    	    // by bitmap.getImage()
    	    int dx = Math.min(cx, srcCx);
    	    int dy = Math.min(cy, srcCy);
    	    
    	    gvimg.drawImage(bitmap.getImage(), x, y, x + dx, y + dy, srcX, srcY, srcX + dx, srcY + dy, null);
    	    
    	    updateImage(x, y, cx, cy, false);
    	}
    	else {    	        	
    	    updateImage(x, y, cx, cy, true);
    	}
    }
    
    // process a destination blit
    public void destBlt(int opcode, int x, int y, int cx, int cy) {
    	// we only support a few opcodes here because we don't have a source
    	switch(opcode) {
    	    case RDPConstants.rdpBlitClear:
    	    case RDPConstants.rdpBlitInvert:
    	    case RDPConstants.rdpBlitSet:
    	    	blit(null, opcode, x, y, cx, cy, 0, 0, 0, 0, 0, 0);
    	    	updateImage(x, y, cx, cy, true);
    	}
    }
    
    private byte hatchPatterns[][] = {
	{ (byte)0x00, (byte)0x00, (byte)0x00, (byte)0xff, (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00 }, /* 0 - bsHorizontal */
	{ (byte)0x08, (byte)0x08, (byte)0x08, (byte)0x08, (byte)0x08, (byte)0x08, (byte)0x08, (byte)0x08 }, /* 1 - bsVertical */
	{ (byte)0x80, (byte)0x40, (byte)0x20, (byte)0x10, (byte)0x08, (byte)0x04, (byte)0x02, (byte)0x01 }, /* 2 - bsFDiagonal */
	{ (byte)0x01, (byte)0x02, (byte)0x04, (byte)0x08, (byte)0x10, (byte)0x20, (byte)0x40, (byte)0x80 }, /* 3 - bsBDiagonal */
	{ (byte)0x08, (byte)0x08, (byte)0x08, (byte)0xff, (byte)0x08, (byte)0x08, (byte)0x08, (byte)0x08 }, /* 4 - bsCross */
	{ (byte)0x81, (byte)0x42, (byte)0x24, (byte)0x18, (byte)0x18, (byte)0x24, (byte)0x42, (byte)0x81 }  /* 5 - bsDiagCross */
    };

    // process a pattern blit
    public void patBlt(int opcode, int x, int y, int cx, int cy, RDPBrush brush, int bgColor, int fgColor) {
    	fgColor = translateColor(fgColor);
    	bgColor = translateColor(bgColor);
    	
    	int buf[];
    	byte fontMap[];
    	int i, j;
    	RDPFontGlyph glyph;
    	
    	switch(brush.getStyle()) {
    	    case RDPConstants.brushStyleSolid:
    	    	buf = getBuffer(cx * cy);
    	    	for(i = 0; i < cy; i++) {
    	    	    for(j = 0; j < cx; j++) {
    	    	    	buf[i * cx + j] = fgColor;
    	    	    }
    	    	}
    	    	blit(buf, opcode, x, y, cx, cy, 0, 0, cx, cy, cx, 0);
    	    	break;
    	    case RDPConstants.brushStyleHatch:
    	    	buf = getBuffer(8 * 8);
    	    	int index = brush.getPattern()[0];
    	    	if(index > hatchPatterns.length) {
    	    	    return;
    	    	}
    	    	glyph = new RDPFontGlyph(0, 0, 8, 8, hatchPatterns[index]);
    	    	fontMap = glyph.getPixMap();
    	    	for(i = 0; i < 8; i++) {
    	    	    for(j = 0; j < 8; j++) {
    	    	    	// we have to reverse fg and bg color here!
    	    	    	buf[i * 8 + j] = fontMap[i * 8 + j] != 0 ? bgColor : fgColor; 
    	    	    }
    	    	}
    	    	for(i = 0; i < cy; i += 8) {
    	    	    for(j = 0; j < cx; j += 8) {
    	    	    	blit(buf, opcode, x + j, y + i, Math.max(8, cx - j), Math.max(8, cy - i), 0, 0, 8, 8, 8, 0);
    	    	    }
    	    	}
    	    	break;
    	    case RDPConstants.brushStylePattern:
    	    	buf = getBuffer(8 * 8);
    	    	byte iPattern[] = new byte[8];
    	    	for(i = 0; i < 8; i++) {
    	    	    iPattern[7 - i] = brush.getPattern()[i];
    	    	}
    	    	glyph = new RDPFontGlyph(0, 0, 8, 8, iPattern);
    	    	fontMap = glyph.getPixMap();
    	    	for(i = 0; i < 8; i++) {
    	    	    for(j = 0; j < 8; j++) {
    	    	    	// we have to reverse fg and bg color here!
    	    	    	buf[i * 8 + j] = fontMap[i * 8 + j] != 0 ? bgColor : fgColor; 
    	    	    }
    	    	}
    	    	for(i = 0; i < cy; i += 8) {
    	    	    for(j = 0; j < cx; j += 8) {
    	    	    	blit(buf, opcode, x + j, y + i, Math.min(8, cx - j), Math.min(8, cy - i), 0, 0, 8, 8, 8, 0);
    	    	    }
    	    	}
    	    	break;
    	    default:
    	    	debug("RDP RDPRenderer patBlt: unknown brush style: " + brush.getStyle());
    	}
    	
    	updateImage(x, y, cx, cy, true);
    }
    
    // process a triBlt order
    public void triBlt(int opcode, int x, int y, int cx, int cy, RDPBitmap bitmap, int srcX, int srcY, RDPBrush brush, int bgColor, int fgColor) {
    	// This is potentially difficult to do in general. Until someone
    	// comes up with a more efficient way of doing it I am using cases.
    	
    	switch(opcode) {
    	    case 0x69: 	// PDSxxn
    	    	memBlt(RDPConstants.rdpBlitXor, x, y, cx, cy, bitmap, srcX, srcY);
    	    	patBlt(RDPConstants.rdpBlitEquiv, x, y, cx, cy, brush, bgColor, fgColor);
    	    	break;
    	    case 0xb8:	// PSDPxax
    	    	patBlt(RDPConstants.rdpBlitXor, x, y, cx, cy, brush, bgColor, fgColor);
    	    	memBlt(RDPConstants.rdpBlitAnd, x, y, cx, cy, bitmap, srcX, srcY);
    	    	patBlt(RDPConstants.rdpBlitXor, x, y, cx, cy, brush, bgColor, fgColor);
    	    	break;
    	    case 0xc0:	// PSa
    	    	memBlt(RDPConstants.rdpBlitCopy, x, y, cx, cy, bitmap, srcX, srcY);
    	    	patBlt(RDPConstants.rdpBlitAnd, x, y, cx, cy, brush, bgColor, fgColor);
    	    	break;
    	    default:
    	    	debug("RDP RDRRendereBase triBlt: unknown opcode received: 0x" + Integer.toHexString(opcode & 0xff));
    	    	memBlt(RDPConstants.rdpBlitCopy, x, y, cx, cy, bitmap, srcX, srcY);
    	}
    }
    
    // save the desktop to cache
    public void desktopSave(int offset, int x, int y, int cx, int cy) {
    	cache.putDesktop(sImgMem, mysize.width, offset, x, y, cx, cy);
    }
    
    // restore desktop from cache
    public void desktopRestore(int offset, int x, int y, int cx, int cy) {
    	cache.getDesktop(sImgMem, mysize.width, offset, x, y, cx, cy);
    	updateImage(x, y, cx, cy, true);
    }
    
    // set a cursor
    public void setCursor(RDPCursor cursor) {
    	Cursor c = cursor.getCursor();
    	if(c != null) {
    	    comp.setCursor(c);
    	}
    }
        
    
    // clean up
    public void dispose() {
	resetClip();
	blit(null, RDPConstants.rdpBlitClear, 0, 0, mysize.width, mysize.height, 0, 0, 0, 0, 0, 0);
	updateImage(0, 0, mysize.width, mysize.height, true);
	//sImg.flush();
	sImgMem = null;
	vimg.flush();
	gvimg.dispose();
    }
    
}
