package nn.pp.rcrdp;

import java.io.*;
import java.net.*;
import nn.pp.rc.*;

/**
 * class TCPLayer
 *
 * the TCP-Layer of the MS RDP protocol
 * this class encapsulates the low level send/receive/connect/disconnect
 * functions
 */
 
public class TCPLayer implements MonitoringStream {
    private boolean debug = false;
    
    private Socket rdpsock = null;
    
    private DataInputStream in = null;
    private DataOutputStream out = null;
    
    private int countIn = 0;
    private int countOut = 0;
    private int inObserver = 0;
    
    private Object writeMtx = new Object();
    
    private Object readCountMtx = new Object();
    private Object writeCountMtx = new Object();
    
    private RDPStream sIn;
    private RDPStream sOut;
    
    private void debug(String s) {
    	if(debug) System.out.println(s);
    }
    
    // constructor
    public TCPLayer() {
    	sIn = new RDPStream(1000);
    	sOut = new RDPStream(1000);
    }
    
    // get and clear read bytes
    public MonitorData getAndClearIn() {
    	MonitorData ret;
    	synchronized(readCountMtx) {
    	    ret = new MonitorData(countIn);
    	    countIn = 0;
    	}
    	return ret;
    }
    
    // get and clear read bytes
    public int getAndClearInObserver() {
    	int ret;
    	synchronized(readCountMtx) {
    	    ret = inObserver;
    	    inObserver = 0;
    	}
    	return ret;
    }
    
    // get and clear sent bytes
    public int getAndClearOut() {
    	int ret;
    	synchronized(writeCountMtx) {
    	    ret = countOut;
    	    countOut = 0;
    	}
    	return ret;
    }
    
    
    // Initialise TCP transport data packet
    public RDPStream init(int maxlen) {
    	RDPStream s = sOut;
    	
    	if(maxlen > s.getSize()) {
    	    s.reallocateNoCopy(maxlen);
    	}
    	s.clear(maxlen);
    	
    	s.setPosition(0);
    	s.markEnd(maxlen);
    	
    	return s;
    }
    
    public RDPStream initNew(int maxlen) {
    	RDPStream s = new RDPStream(maxlen);
    	return s;
    }
    
    // Establish a connection on the TCP layer
    public void connect(String server, int port) throws IOException, SocketException {
    	rdpsock = new Socket(server, port);
    	rdpsock.setTcpNoDelay(true);
	in = new DataInputStream(new BufferedInputStream(rdpsock.getInputStream()));
	out = new DataOutputStream(rdpsock.getOutputStream());
    }
    
    // Disconnect on the TCP layer
    public void disconnect() {
    	try {
	    in.close();
	    out.close();
	    rdpsock.close();
	}
	catch (IOException e) {
	    // nothing
	}
	finally {
	    in = null;
	    out = null;
	    rdpsock = null;
	}
    }
    
    // Send TCP transport data packet
    public void send(RDPStream s) throws IOException {
    	int len = s.getEnd();
    	if(len > 0) {
    	    synchronized(writeMtx) {
    	    	out.write(s.getPacket(), 0, len);
    	    	out.flush();
    	    }
    	    synchronized(writeCountMtx) {
    	    	countOut += len;
    	    }
	}
	else if(len == 0) {
	    return;
	}
	else {
	    throw new IOException(T._("TCP send: end not marked"));
	}
    }
    
    // Receive a message on the TCP layer
    public RDPStream receive(int len) throws IOException {
    	RDPStream s = sIn;
    	
    	if(len > s.getSize()) {
    	    s.reallocate(len);
    	}
    	
    	if(len > 0) {
    	    s.setPosition(0);
    	    in.readFully(s.getPacket(), 0, len);
    	    synchronized(readCountMtx) {
    	    	countIn += len;
    	    	inObserver = 1;
    	    }
    	    s.markEnd(len);
    	}
    	else {
    	    throw new IOException(T._("TCP receive: len <= 0"));
    	}

    	return s;
    }
    
}
