#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <sys/poll.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <getopt.h>
#include <asm/io.h>
#include <pp/i2c.h>
#ifdef PRODUCT_PDU
# include <pp/bmc/bmc.h>
#endif

static const char *usage_msg =
"\n"
"   Options:\n"
"     -d <device>                ..  the i2c device file, default is /dev/i2c-0\n"
"     -o <saddr>                 ..  set own I2C slave address\n"
"     -i <saddr>                 ..  I2C slave address\n"
#ifdef PRODUCT_PDU
"     -u <file>                  ..  Update CY8 I2C slave device for PDU with binary file\n"
#endif
"     -a <ioaddr>                ..  read IO address \n"
"     -a <ioaddr> -v <value>     ..  write value to IO address \n"
"     -x <ioaddr> -v <value>     ..  write value to IO address (no read back)\n"
"     -I                         ..  select integer address input (default is hex)\n"
"     -s                         ..  silent\n"
"     -b                         ..  burst mode\n"
"     -m                         ..  smbus read\n"
"     -r                         ..  read plain without addressing\n"
"     -c <num>                   ..  read plain num (max. 10) bytes\n"
"     -w -v <value>              ..  write plain without addressing\n"
"     -z <num>                   ..  write num (max. 10) bytes (1,2,3...) plain without smbus addressing\n"
"     -p <b0> <b1> <b2> .. <bn>  ..  send plain data packet\n"
"     -f                         ..  flush the i2c slave buffer\n"
"     -l <start>:<end> (0 - all) ..  lookup the i2c bus for devices\n"
"     -R <retries>               ..  number of retries for lookup\n"
"\n";

#define DEFAULT_FILENAME	"/dev/i2c-0"

void print_version(void);

int do_read_i2c = 0;
int do_write_i2c = 0;
int do_read_plain = 0;
int do_write_plain = 0;
int silent_write = 0;
int tft_con = 0;
int adc_con = 0;
#ifdef PRODUCT_PDU
int do_cy8_i2c_update = 0;
#endif
int do_init_ragnar = 0;
int do_integer_scan = 0;
int do_info_print = 0;
int do_write_video = 0;
int do_smbusread = 0;
int do_set_slave = 0;
int do_write_block = 0;
int do_flush = 0;
int do_lookup = 0;
int do_send_packet = 0;
int do_read_block = 0;

int	i2c_addr = 0;
int	reg_addr = 0;
int 	reg_val  = 0;
int	lookup_retries = 1;
int     i2c_burstlen = 1;
int	i2c_slave_addr = 0;
u_int	write_num = 0;
u_int	read_num = 0;
u_int	lookup_first = 0, lookup_last = 0;
unsigned char write_buf[256];


void print_version(void){

    printf("%s: %s (c) 2001-2005 PEPPERCON AG info@peppercon.com\n",__FILE__,__DATE__);
	
}

int
main(int argc, char ** argv)
{
    int fd;
#ifdef PRODUCT_PDU
    FILE *fp;
#endif /*PRODUCT PDU*/
    int errflag = 0;
    int c;
    u_int i;
    char *endptr;
    char* filename = NULL;
    
#ifdef PRODUCT_PDU
    char* ufilename = NULL;
    while ((c = getopt(argc, argv, "fz:u:x:o:a:v:i:sI?b:mrwl:d:pc:R:"))
	   != -1)
#else
    while ((c = getopt(argc, argv, "fz:x:o:a:v:i:sI?b:mrwl:d:pc:R:"))
	   != -1)
#endif /*PRODUCT PDU*/
	switch (c) {
	  case 'd':
	      filename = strdup(optarg);
	      break;
	  case 'o':
	      do_set_slave = 1;
	      i2c_slave_addr = strtol(optarg, &endptr, 16);
	  case 'a': 	do_read_i2c++;
	      // fall through
	  case 'x':
	      if (do_integer_scan)
		  reg_addr = strtol(optarg, &endptr, 10);
	      else
		  reg_addr = strtol(optarg, &endptr, 16);
#if 0
	      if ((optarg[0] == '-') || !((optarg != '\0') && (*endptr == '\0')) || errno ||
			(optarg == NULL) || (*optarg == '\0')) {
		  printf("reg_addr: %x\n",reg_addr);
		  errflag++;
	      }
#endif
	      break;		
	  case 'v':	do_write_i2c++; 
	      reg_val = strtoul(optarg, &endptr, 16);
	      if (	(optarg[0] == '-') || !((optarg != '\0') && (*endptr == '\0')) || errno ||
			(optarg == NULL) || (*optarg == '\0')) {
		  printf("reg_val: %x\n",reg_val);
		  errflag++;
	      }
	      break;
	  case 's': 	silent_write++;
	      break;
	  case 'i': 	i2c_addr = strtol(optarg, &endptr, 16);
	      break;
#ifdef PRODUCT_PDU
	  case 'u': 	do_cy8_i2c_update = 1; ufilename = strdup(optarg);
	      break;
#endif /*PRODUCT PDU*/
	  case 'I': 	do_integer_scan=1;
	      break;
	  case 'b': 	i2c_burstlen = strtol(optarg, &endptr, 16);
	      break;
	  case 'm': 	do_smbusread=1;
	      break;
	  case 'r':	do_read_plain = 1;
	      break;
	  case 'c':	do_read_block = 1; read_num = strtoul(optarg, &endptr, 10);
	      break;
	  case 'w':	do_write_plain = 1;
	      break;   
	  case 'z':	do_write_block = 1; write_num = strtoul(optarg, &endptr, 10);
	      break;
          case 'p':
              do_send_packet = 1;
              write_num = argc - optind;
              if (write_num > sizeof(write_buf)) write_num = sizeof(write_buf);
              for (i = 0; i < write_num; i++) {
                  write_buf[i] = strtol(argv[i + optind], &endptr, 16);
              }
              break;
	  case 'f':	do_flush = 1;
	      break;
	  case 'l':
	      do_lookup = 1;
	      if (!strcmp(optarg, "0")) {
	          lookup_first = 0x08;
	          lookup_last = 0x77;
	      } else {
	          int ret = sscanf(optarg, "%02x:%02x",
	          		   &lookup_first, &lookup_last);
	          if (ret != 2) {
	              errflag++;
	          }
	          if (lookup_first > lookup_last) {
	              int tmp = lookup_first;
	              lookup_first = lookup_last;
	              lookup_last = tmp;
	          }
	          if (lookup_first < 0x08) lookup_first = 0x08;
	          if (lookup_last > 0x77) lookup_last = 0x77;		  
	      }
	      break;
	  case 'R':
	      lookup_retries = strtol(optarg, &endptr, 10);
	      break;
	  case '?':
	      errflag++;
	}
	
    if (errflag) {
	printf(usage_msg);
	return 3;
    }
 	
    /* Open the I2C interface */
    if (!filename) filename = strdup(DEFAULT_FILENAME);
    if ((fd = open(filename,O_RDWR)) < 0) {
	/* ERROR HANDLING; you can check errno to see what went wrong */
	fprintf(stderr, "Error Opening I2C device %i.\n",errno);
	exit(1);
    }
    
    /* lookup the bus for devices */  
    if (do_lookup) {
    	int r = 0;
    	int found = 0;
    	
    	printf("Scanning the i2c bus from 0x%02x to 0x%02x (Retries: %d) ...\n",
	       lookup_first, lookup_last, lookup_retries);

	while(found == 0 && lookup_retries-- > 0) {
	    for (i = lookup_first; i <= lookup_last; i++) {
		if (ioctl(fd,I2C_SLAVE_FORCE,i) < 0) {
		    /* ERROR HANDLING; you can check errno to see what went wrong */
		    fprintf(stderr, "IOCTL I2C device %i (0x%02x).\n", errno, i);
		    continue;
		}
		
		if (!i2c_smbus_write_quick(fd, 0)) {
		    printf("slave addr %02x responding\n", i);
		    found++;
		    continue;
		}
		// more heuristic search, just do a dummy read to
		// put devices which hang because of the quick write back to life
		i2c_smbus_read_byte_data(fd, 0);
		if ( i2c_smbus_read_byte_data(fd, 0) != -1) {
		    printf("slave addr %02x responding (read reg 0 returned something)\n", i);
		    found++;
		}
	    }
    	
	    if (found) {
		printf("Found %d device%s.\n", found, found == 1 ? "" : "s");
	    } else {
		printf("No devices found.\n");
		r = 1;
	    }
	}
    	
    	close(fd);
    	return r;
    }
    
    /* flush slave buffer */
    
    if (do_flush) {
    	if (ioctl(fd, I2C_SLAVE_FLUSH_BUFFER, 0) < 0) {
    	    fprintf(stderr, "Error flushing i2c buffer: %i.\n", errno);
    	}
    	else {
    	    printf("i2c buffer flushed.\n");
    	}
    	close(fd);
    	return 0;
    }
    
    /* set I2C slave address */
    
    if (do_set_slave) {
    	if (ioctl(fd, I2C_SLAVE_SET_ADDRESS, i2c_slave_addr) < 0) {
    	    fprintf(stderr, "Error setting I2C slave address: %i.\n", errno);
    	}
    	else {
    	    printf("I2C slave address set to 0x%02x.\n", i2c_slave_addr);
    	}
    	close(fd);
    	return 0;
    }

    /* Read or write registers with addressing */

    if (i2c_addr) {
	
	if (ioctl(fd, I2C_SLAVE_FORCE, i2c_addr) < 0) {
	    /* ERROR HANDLING; you can check errno to see what went wrong */
	    fprintf(stderr, "IOCTL I2C device %i.\n",errno);
	    exit(1);
	}
	if (silent_write == 0) printf("I2C Addr: 0x%02x   \n",i2c_addr);

	if (do_read_i2c && !do_write_i2c) {
	    printf("Reg 0x%02x: 0x%02x\n",reg_addr, i2c_smbus_read_byte_data(fd,reg_addr));
	    close(fd);
	    return 0;
	}
		
	if (do_read_i2c && do_write_i2c) {
	    i2c_smbus_write_byte_data(fd,reg_addr,reg_val);
	    if (silent_write == 0) 
		printf("Reg 0x%02x: 0x%02x\n",reg_addr, i2c_smbus_read_byte_data(fd,reg_addr));
	    close(fd);
	    return 0;
	}

	if (!do_read_i2c && do_write_i2c && !do_write_plain) {
	    i2c_smbus_write_byte_data(fd,reg_addr,reg_val);
	    if (silent_write == 0) 
		printf("Reg 0x%02x: 0x%02x\n",reg_addr, reg_val);
	    close(fd);
	    return 0;
	}

	if (do_read_plain) {
	    printf("Read 0x%02x\n", i2c_smbus_read_byte(fd));
	    close(fd);
	    return 0;
	}
	
	if (do_read_block) {
	    char data[10];
	    u_int count = (read_num > sizeof(data) ? sizeof(data) : read_num);
  	
	    read(fd, data, count);
	    printf("Read:");
	    for (i = 0; i < count; i++) {
	    	printf(" 0x%02x", data[i]);
	    }
	    printf("\n");
	    close(fd);
	    return 0;
	}

	if (do_write_plain) {
	    i2c_smbus_write_byte(fd, reg_val);
	    close(fd);
	    return 0;
	}
	
	if (do_write_block) {
	    char data[] = { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 };
	    write(fd, data, (write_num > sizeof(data) ? sizeof(data) : write_num));
	    close(fd);
	    return 0;
	}

        if (do_send_packet) {
            printf("Write %d bytes: [%02x]", write_num, i2c_addr << 1);
            for (i = 0; i < write_num; i++) printf(" %02x", write_buf[i]);
            printf("\n");
            fflush(stdout);
            sleep(1);
            i = write(fd, write_buf, write_num);
            if (i != write_num) fprintf(stderr, "Error %d.\n", errno);
	    close(fd);
	    return 0;
        }
#ifdef PRODUCT_PDU
        if (do_cy8_i2c_update) {
            unsigned char data[78],status;
            int k,ch,err = 0,len = 0,times = 0;
            int ca=0,cb=0;		

	    /*Set register 0x09 to 1; It means that switch to firmware-upload*/
    	    unsigned char   set2upld[2]={0x09,0x01};

   	    if ((fp = fopen(ufilename,"rb")) == NULL){
         	fprintf(stderr, "Error Opening update binary file %i.\n",errno);
           	exit(1);
            }
	    while ((ch = fgetc(fp)) != EOF)
		len++;

	    fseek(fp,0,SEEK_SET);
	    times = ((len - 20) / 78) + 2;    // i2c transmit times of cy8.bin
 	    write(fd,set2upld,2);
 	    usleep(1000000);

	    for (k = 0; k < times; k++){
		if (k == 0 || k == (times - 1)){
              	    fread(data,sizeof(unsigned char),10,fp);
		    do{		
			err = 0;
		    	if (write(fd,data,10) != 10) err++;
		        usleep(50000);
			if (0 > read(fd,&status,1)) {
                    	    fprintf(stderr, "i2c read from %#x failed", i2c_addr);
                    	    err++;
			}
                  	if(status != 0x20) err++;
                    }while(err != 0);
		    ca++;
		}
		else{
		     //if(status == 0x20){
              	         fread(data,sizeof(unsigned char),78,fp);
		 	 do{		
                         	err = 0;
			    	if (write(fd,data,78) != 78) err++;
                     		usleep(50000);
                    		if (0 > read(fd,&status,1)) {
     			   	    fprintf(stderr, "i2c read from %#x failed", i2c_addr);
	   		    	    err++;
			        }
			 	if(status != 0x20) err++;
                     	 }while(err != 0);
			cb++;
                    // }
                }

	    }
	    printf("Done for cy8 at %#x\n", i2c_addr);
            printf("\n");
	    fclose(fp);	
            close(fd);
            return 0;
        }
#endif/*PRODUCT PDU*/

    }

#if 0
    if (i2c_addr){
	
	if (ioctl(fd,I2C_SLAVE_FORCE,i2c_addr) < 0) {
	    /* ERROR HANDLING; you can check errno to see what went wrong */
	    fprintf(stderr, "IOCTL I2C device %i.\n",errno);
	    exit(1);
	}
	      	
	if (do_smbusread == 1){
	    printf("Reg 0x%02x: 0x%02x\n",reg_addr, i2c_smbus_read_byte_data(fd,reg_addr));
	    if (do_write_i2c == 0) return 0;

	}
	else{
	      	
	    buf[0] = reg_addr;
		      	
	    if (silent_write == 0) printf("I2C Addr: 0x%02x\n",i2c_addr);
	
	    if (do_read_i2c && !do_write_i2c) {
	
		if (write(fd,buf,2) != 2) {
		    printf("Error during plain I2C write\n");
		} 	
		else {
		}
	
		if (read(fd,buf,2) != 2) {
		    printf("Error during plain I2C read\n");
		} else {
		    printf("Reg 0x%02x: 0x%02x\n",reg_addr, buf[0]);
		}
				
		if (do_write_i2c == 0) return 0;
	    }
	}		
	if (do_read_i2c && do_write_i2c) {
			
	    buf[1] = (unsigned char) reg_val;
	    buf[2] = (unsigned char) (reg_val>>8);
			
	    if ( (retval = write(fd,buf,(i2c_burstlen+1))) != (i2c_burstlen + 1)) {
		printf("Error during plain I2C write %i\n", retval);
	    }
	    else{
		if (silent_write == 0) 
		    printf("Reg 0x%02x: 0x%02x\n",reg_addr, i2c_smbus_read_byte_data(fd,reg_addr));
		return 0;
	    }
			if (i2c_burstlen > 1){

				for (i=0;i<10;i++){
					if ( (retval = write(fd,buf,(i2c_burstlen+1))) != (i2c_burstlen + 1)) {
		       				printf("Error during plain I2C write %i\n", retval);
		    			}
		    		}
			}
		}
	}
#endif
	if(fd) close(fd);
	free(filename);
	return 0;
}


